# encoding: utf-8

import logging
import os
import shutil
import tempfile

from sandbox import sdk2

from sandbox.projects.common import binary_task
from sandbox.projects.rasp.resource_types import MessagePackResource
from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params
from sandbox.projects.rasp.utils.solomon import SolomonProgressMixin
from sandbox.projects.rasp.utils.rasp_scripts import RaspScriptsTask, use_rasp_scripts_params

log = logging.getLogger(__name__)
log.setLevel(logging.INFO)


class RaspSuggestsGenerateSearchStat(
    binary_task.LastBinaryTaskRelease,
    sdk2.Task,
    EmailNotificationMixin,
    JugglerNotificationMixin,
    SolomonProgressMixin,
    RaspScriptsTask
):
    class Requirements(sdk2.Task.Requirements):
        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

        cores = 8  # cores number is required to enable multislot
        ram = 32 * 1024

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600 * 6

        ext_params = binary_task.binary_release_parameters(stable=True)  # нужно для работы LastBinaryTaskRelease

        with sdk2.parameters.Group('General') as mds_block:
            yt_prefix = sdk2.parameters.String('YT prefix', required=True, default='testing')
            days = sdk2.parameters.Integer('Days', required=True, default=90)

        _rasp_scripts_params = use_rasp_scripts_params()
        _email_notification_params = use_email_notification_params()
        _juggler_notification_params = use_juggler_notification_params()

    def set_task_environ(self):
        os.environ['DJANGO_SETTINGS_MODULE'] = 'travel.rasp.suggests_tasks.bin.sandbox.local_settings'

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)  # нужно для работы LastBinaryTaskRelease
        self.add_email_notifications()
        self.add_juggler_notifications(environment=self.Parameters.env_type)

    def on_execute(self):
        self.set_common_environ()

        from travel.rasp.suggests_tasks.suggests.generate.ytwork.search_stat import run
        from travel.rasp.suggests_tasks.suggests.storage import Storage

        storage = Storage(tempfile.gettempdir())

        with self.report_progress(source='generate_search_stat', environment=self.Parameters.env_type):
            run(storage.data_path, ytprefix=self.Parameters.yt_prefix, days=self.Parameters.days)

        stats_data = sdk2.ResourceData(
            MessagePackResource(self, 'Search stat', storage.stat_file,
                                env_type=self.Parameters.env_type,
                                suggests_file=storage.stat_file)
        )

        stats_path = os.path.join(storage.data_path, storage.stat_file)
        shutil.copy(stats_path, str(stats_data.path))
        stats_data.ready()
