# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

from sandbox import sdk2
from sandbox import common
from sandbox.common.types import task as ctt

from sandbox.projects.rasp.utils.email_notifications import EmailNotificationMixin, use_email_notification_params
from sandbox.projects.rasp.smoke.RunSmokeTests import RaspRunSmokeTests


logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)


class RaspRunAllSmokeTests(sdk2.Task, EmailNotificationMixin):
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1 * 1024

        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 1800

        with sdk2.parameters.Group('General') as general:
            env_name = sdk2.parameters.String('Environment for run smoke tests', required=True, default='testing')
            with sdk2.parameters.RadioGroup('Stableness') as stableness:
                stableness.values['stable'] = stableness.Value('Stable', default=True)
                stableness.values['unstable'] = stableness.Value('Unstable')
                stableness.values['all'] = stableness.Value('All')

        _email_notification_params = use_email_notification_params()

    CONFIG_MODULES = [
        'morda_backend',
        'export',
        'morda_front',
        'suburban_selling',
        'api_public',
        'pathfinder_core',
        'blablacar',
        'infocenter',
        'suburban_widget',
    ]

    def on_execute(self):
        with self.memoize_stage.run_tasks:
            tasks = []
            for module in self.CONFIG_MODULES:
                task = RaspRunSmokeTests(
                    self,
                    owner=self.Parameters.owner,
                    priority=self.Parameters.priority,
                    description='{} smoke tests for {}'.format(module, self.Parameters.env_name),
                    tags=self.Parameters.tags,
                    config_module=module,
                    env_name=self.Parameters.env_name,
                    stableness=self.Parameters.stableness
                )
                task.enqueue()
                tasks.append(task)

            raise sdk2.WaitTask(tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

        with self.memoize_stage.check_tasks:
            failed_tasks = [task for task in self.find() if task.status not in {
                ctt.Status.DRAFT,
                ctt.Status.ENQUEUED,
                ctt.Status.ASSIGNED,
                ctt.Status.PREPARING,
                ctt.Status.EXECUTING,
                ctt.Status.FINISHING,
                ctt.Status.SUCCESS,
            }]
            if failed_tasks:
                raise common.errors.TaskFailure(
                    'Smoke tests failed for tasks: {}'.format(
                        ', '.join([
                            'https://sandbox.yandex-team.ru/task/{}/view ({} {} {})'.format(
                                task.id, task.Parameters.config_module,
                                task.Parameters.env_name, task.Parameters.stableness
                            )
                            for task in failed_tasks
                        ])
                    )
                )

    def on_save(self):
        super(RaspRunAllSmokeTests, self).on_save()
        self.add_email_notifications()
