# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.rasp.utils.solomon import SolomonProgressMixin
from sandbox.projects.rasp.utils.juggler import JugglerNotificationMixin, use_juggler_notification_params

log = logging.getLogger(__name__)
log.setLevel(logging.INFO)


class RaspRunSmokeTests(binary_task.LastBinaryTaskRelease, SolomonProgressMixin, JugglerNotificationMixin, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        cores = 10
        ram = 2 * 1024

        class Caches(sdk2.Requirements.Caches):  # enable multislot https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        kill_timeout = 3300
        vault_oauth = sdk2.parameters.String('Vault oauth name', required=True, default='rasp-robot-vault-oauth')
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('General') as general:
            config_module = sdk2.parameters.String('Module config name', required=True, default='morda_backend')
            env_name = sdk2.parameters.String('Environment for run smoke', required=True, default='testing')
            envparams = sdk2.parameters.String('Additional environment params', required=False, default=None)
            with sdk2.parameters.RadioGroup('Stableness') as stableness:
                stableness.values['stable'] = stableness.Value('Stable', default=True)
                stableness.values['unstable'] = stableness.Value('Unstable')
                stableness.values['all'] = stableness.Value('All')

        _juggler_notification_params = use_juggler_notification_params()

    def on_save(self):
        binary_task.LastBinaryTaskRelease.on_save(self)

        self.add_juggler_notifications(
            service='{}.{}.{}'.format(self.type, self.Parameters.config_module, self.Parameters.stableness),
            environment=self.Parameters.env_name
        )

    def on_execute(self):
        vault_oauth = sdk2.Vault.data(self.Parameters.vault_oauth)
        os.environ['RASP_VAULT_OAUTH_TOKEN'] = vault_oauth

        log.info('Run {} smoke tests, {}, {}'.format(
            self.Parameters.stableness,
            self.Parameters.config_module,
            self.Parameters.env_name,
        ))

        with self.report_progress(
            '{}_{}'.format(self.Parameters.config_module, self.Parameters.stableness),
            self.Parameters.env_name
        ):
            from travel.rasp.smoke_tests.smoke_tests.run_smoke_tests import run
            run(
                config_module=self.Parameters.config_module,
                env_name=self.Parameters.env_name,
                stableness=self.Parameters.stableness,
                envparams=self.Parameters.envparams
            )

        log.info('DONE')
