# encoding: utf-8

import logging

import msgpack_utils

from sandbox import sandboxsdk
from sandbox import sdk2
from sandbox.projects.rasp.qloud.UpdateResources import RaspQloudUpdateResources
from sandbox.projects.rasp.utils.email_notifications import (
    EmailNotificationMixin,
    use_email_notification_params,
    BUS_GROUP,
)
from sandbox.projects.rasp.resource_types import MessagePackResource


class RaspSuggestsDistributeData(sdk2.Task, EmailNotificationMixin):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            sandboxsdk.environments.PipEnvironment('u-msgpack-python'),
        )

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        with sdk2.parameters.Group('Qloud parameters') as qloud_block:
            qloud_environments = sdk2.parameters.List(
                'Qloud environment names',
                required=True,
            )
            qloud_token_owner = sdk2.parameters.String('Qloud token owner name', required=True)
            qloud_token_vault = sdk2.parameters.String('Qloud token vault', required=True)

        with sdk2.parameters.Group('MDS parameters') as mds_block:
            mds_bucket = sdk2.parameters.String(
                'Bucket name', required=True, default='{mds_common_bucket}'
            )
            mds_url = sdk2.parameters.String('URL', required=True, default='{mds_url}')
            vault_owner = sdk2.parameters.String('Vault owner name', required=True)
            mds_access_key_id_vault = sdk2.parameters.String('Vault name: access_key_id', required=True,
                                                             default='{mds_access_key_id}')
            mds_access_secret_key_vault = sdk2.parameters.String('Vault name: access_secret_key', required=True,
                                                                 default='{mds_access_secret_key}')
            idconverter_path = sdk2.parameters.String(
                'idconverter.msgpack path',
                default='suggests/idconverter.msgpack',
                required=True,
            )
            objs_data_path = sdk2.parameters.String(
                'objs_data.msgpack path',
                default='suggests/objs_data.msgpack',
                required=True,
            )

        _email_notification_params = use_email_notification_params()

    def _get_mds_client(self, mds_url, access_key_id, access_secret_key):
        import boto3
        session = boto3.session.Session(
            aws_access_key_id=access_key_id,
            aws_secret_access_key=access_secret_key,
        )

        return session.client(
            service_name='s3',
            endpoint_url=mds_url,
        )

    def _get_file_mds(self, mds_client, file_path):
        return mds_client.get_object(
            Bucket=self.Parameters.mds_bucket,
            Key=file_path,
        )['Body'].read()

    def _reformat_msgpack(self, raw_data, reformat_func):
        import umsgpack
        data = umsgpack.loads(raw_data)
        return umsgpack.dumps(reformat_func(data))

    def on_execute(self):
        idconverter = sdk2.ResourceData(
            MessagePackResource(self, 'ID Converter', 'idconverter.msgpack')
        )
        objs_data = sdk2.ResourceData(
            MessagePackResource(self, 'Objects Data', 'objs_data.msgpack')
        )
        mds_client = self._get_mds_client(
            self.Parameters.mds_url,
            sdk2.Vault.data(self.Parameters.vault_owner, self.Parameters.mds_access_key_id_vault),
            sdk2.Vault.data(self.Parameters.vault_owner, self.Parameters.mds_access_secret_key_vault),
        )
        idconverter.path.write_bytes(
            self._reformat_msgpack(
                self._get_file_mds(mds_client, self.Parameters.idconverter_path),
                msgpack_utils.reformat_idconverter,
            )
        )
        objs_data.path.write_bytes(
            self._reformat_msgpack(
                self._get_file_mds(mds_client, self.Parameters.objs_data_path),
                msgpack_utils.reformat_objs_data,
            )
        )

        for env in self.Parameters.qloud_environments:
            logging.debug('Updating {} environment.'.format(env))
            qloud_updater = RaspQloudUpdateResources(
                self,
                description='Update resources for {} environment'.format(env),
                qloud_environment=env,
                resources=(
                    objs_data._ResourceData__resource,
                    idconverter._ResourceData__resource,
                ),
                token_owner=self.Parameters.qloud_token_owner,
                token_name=self.Parameters.qloud_token_vault,

                enable_email_notifications=False
            )
            qloud_updater.enqueue()

    def on_save(self):
        super(RaspSuggestsDistributeData, self).on_save()
        self.add_email_notifications(notifications_group=BUS_GROUP)
