# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from typing import AnyStr, Union

from sandbox import sdk2
from sandbox.common.types import notification as ctn
from sandbox.common.types.task import Status


def use_email_notification_params():
    class _EmailNotificationParams(sdk2.Parameters):
        with sdk2.parameters.Group('Email notifications parameters') as email_notifications_params:
            enable_email_notifications = sdk2.parameters.Bool(
                'Enable email notifications?', required=True, default=True
            )

    return _EmailNotificationParams()


# see more about statuses here https://docs.yandex-team.ru/sandbox/tasks#status
AVIA_GROUP = 'avia-alerts'
BUS_GROUP = 'yatravel-bus-announces'
RASP_GROUP = 'rasp-sandbox'
TRAIN_GROUP = 'train-sandbox-alerts'

SWITCHABLE_NOTIFICATIONS = {
    RASP_GROUP: [
        sdk2.Notification(
            [
                Status.NOT_RELEASED,
                Status.FAILURE,
                Status.EXCEPTION,
                Status.NO_RES,
                Status.TIMEOUT,
                Status.EXPIRED,
            ],
            ['rasp-sandbox-fail'],
            ctn.Transport.EMAIL
        ),
        sdk2.Notification(
            [
                Status.TEMPORARY,
                Status.STOPPED,
            ],
            ['rasp-sandbox-workflow'],
            ctn.Transport.EMAIL
        )
    ],
    TRAIN_GROUP: [
        sdk2.Notification(
            [
                Status.NOT_RELEASED,
                Status.FAILURE,
                Status.EXCEPTION,
                Status.NO_RES,
                Status.TIMEOUT,
                Status.EXPIRED,
                Status.TEMPORARY,
                Status.STOPPED
            ],
            ['train-sandbox-alerts'],
            ctn.Transport.EMAIL
        )
    ],
    BUS_GROUP: [
        sdk2.Notification(
            [
                Status.EXCEPTION,
                Status.NOT_RELEASED,
                Status.SUCCESS,
                Status.NO_RES,
                Status.FAILURE,
                Status.TIMEOUT,
                Status.EXPIRED
            ],
            ['yatravel-bus-announces'],
            ctn.Transport.EMAIL
        )
    ],
    AVIA_GROUP: [
        sdk2.Notification(
            [
                Status.NOT_RELEASED,
                Status.FAILURE,
                Status.EXCEPTION,
                Status.NO_RES,
                Status.TIMEOUT,
                Status.EXPIRED,
            ],
            ['avia-alerts'],
            ctn.Transport.EMAIL
        )
    ]
}


class EmailNotificationMixin(object):
    """
    Для использования:
    1) унаследовать таск от EmailNotificationMixin
    2) в Parameters добавить строку:
    _email_notification_params = use_email_notification_params()
    3) в методе on_save вызвать self.add_email_notifications()

    Тогда в UI Sandbox появится чекбокс enable_email_notifications.
    - С включенным параметром автоматически подтянутся уведомления из SWITCHABLE_NOTIFICATIONS[notifications_group]
    - С выключенным параметром уведомления не будут добавлены
    """

    def add_email_notifications(self, notifications_group=RASP_GROUP):
        # type: (Union[sdk2.Task, EmailNotificationMixin], AnyStr) -> None

        if not self.Parameters.enable_email_notifications:
            return

        switchable_notifications = SWITCHABLE_NOTIFICATIONS.get(notifications_group, [])
        switchable_notifications_recipients = set(sum((n.recipients for n in switchable_notifications), []))

        old_notifications = self.Parameters.notifications or []
        notifications = [
            n for n in old_notifications
            if (
                n.transport != ctn.Transport.EMAIL or
                all(recipient not in switchable_notifications_recipients for recipient in n.recipients)
            )
        ]
        notifications += switchable_notifications

        self.Parameters.notifications = notifications
