# coding: utf-8

import tarfile
import os
import logging
import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt

from sandbox.projects import resource_types
from sandbox.sdk2.helpers import subprocess as subprocess
from sandbox import sdk2
from sandbox.projects.common.nanny import nanny


class RazladkiSuggestSerializedResource(sdk2.Resource):
    """ Razladki Suggest Projects Resourse """
    releasable = True


class RazladkiSuggestBinaryResource(sdk2.Resource):
    """ Razladki Suggest Binary """


class RazladkiDbSharedResource(sdk2.Resource):
    """ db_shared.json file resource """


class RazladkiFileWithProjectNamesResource(sdk2.Resource):
    """ file with project names """
    releasable = True


class MakeRazladkiSuggestData(nanny.ReleaseToNannyTask2, sdk2.Task):
    """ Task download projects from db """

    PATH_PACKET = 'extract_dir'
    PATH_RESULT_DIR = 'projects'
    PATH_TGZ = 'razladki.tar.gz'
    SERIALIZED_OUT_FILE = 'data.bin'
    SUGGEST_MODE = 'serialize'
    SUGGEST_CONFIG = 'config'
    SUGGEST_FILE_WITH_PROJECT_NAMES = 'project_names.txt'

    class Parameters(sdk2.Parameters):

        razladki_shard_resourse = sdk2.parameters.LastReleasedResource(
            'razladki shard resource',
            state=(ctr.State.READY, ctr.State.NOT_READY),
            resource_type=resource_types.RAZLADKI_SHARD,
            required=True
        )

        db_shared_json = sdk2.parameters.LastReleasedResource(
            'db shared file resource',
            state=(ctr.State.READY, ctr.State.NOT_READY),
            resource_type=RazladkiDbSharedResource,
            required=True
        )

        razladki_suggest_binary = sdk2.parameters.LastReleasedResource(
            'razladki suggest binary',
            state=(ctr.State.READY, ctr.State.NOT_READY),
            resource_type=RazladkiSuggestBinaryResource,
            required=True
        )

        release_on_success = sdk2.parameters.Bool('Release if the task complete success', default=False, required=True)
        with release_on_success.value[True]:
            release_status = sdk2.parameters.Bool('Release STABLE if true else TESTING', default=False, required=True)

    def on_execute(self):
        db_shared_path = sdk2.ResourceData(self.Parameters.db_shared_json).path
        razladki_shard_path = sdk2.ResourceData(self.Parameters.razladki_shard_resourse).path
        razladki_suggest_binary_path = sdk2.ResourceData(self.Parameters.razladki_suggest_binary).path.absolute()
        path_packet = self.path(self.PATH_PACKET)
        projects_dir = self.path(self.PATH_RESULT_DIR)
        suggest_config_path = self.path(self.SUGGEST_CONFIG)
        suggest_file_with_project_names_path = self.path(self.SUGGEST_FILE_WITH_PROJECT_NAMES)
        suggest_serialized_out_file_path = self.path(self.SERIALIZED_OUT_FILE)

        os.mkdir(path_packet.as_posix())
        os.mkdir(projects_dir.as_posix())

        tarfile.open(razladki_shard_path.joinpath(self.PATH_TGZ).as_posix()).extractall(path_packet.as_posix())
        python_path = self.path(self.PATH_PACKET).joinpath('env', 'bin', 'python').as_posix()

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('python')) as process_log:
            program = self.path().joinpath(self.PATH_PACKET, 'suggest', 'scripts', 'download_params.py').as_posix()
            args = '--path_to_save={} --db_shared_file={}'.format(projects_dir.as_posix(), db_shared_path.as_posix())
            cmd = '{} {} {}'.format(python_path, program, args)
            subprocess.check_call(cmd, shell=True, stdout=process_log.stdout, stderr=subprocess.STDOUT)

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('python')) as process_log:
            program = self.path(self.PATH_PACKET).joinpath('suggest', 'scripts', 'generate_config.py').as_posix()
            args = '--suggest_config={} --projects_dir={} --file_with_project_names={}'.format(
                suggest_config_path.as_posix(), projects_dir.as_posix(), suggest_file_with_project_names_path.as_posix()
            )
            cmd = '{} {} {}'.format(python_path, program, args)
            subprocess.check_call(cmd, shell=True, stdout=process_log.stdout, stderr=subprocess.STDOUT)

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('suggest')) as process_log:
            program = razladki_suggest_binary_path.as_posix()
            args = '--program-mode={} --bin-path={} --config={}'.format(self.SUGGEST_MODE, suggest_serialized_out_file_path.as_posix(), suggest_config_path.as_posix())
            cmd = '{} {}'.format(program, args)
            subprocess.check_call(cmd, shell=True, stdout=process_log.stdout, stderr=subprocess.STDOUT)

        sdk2.ResourceData(RazladkiSuggestSerializedResource(self, 'razladki suggest serialized data', suggest_serialized_out_file_path.as_posix()))
        sdk2.ResourceData(RazladkiFileWithProjectNamesResource(self, 'razladki suggest file with project names', suggest_file_with_project_names_path.as_posix()))

    def on_success(self, prev_status):
        sdk2.Task.on_success(self, prev_status)
        if self.Parameters.release_on_success is True:
            status = ctt.ReleaseStatus.STABLE if self.Parameters.release_status is True else ctt.ReleaseStatus.TESTING
            self.on_release(dict(
                releaser=self.author,
                release_status=status,
                release_subject='Razladki suggest data',
                email_notifications=dict(to=['atayan1', 'valgushev'], cc=[]),
                release_comments='Razladki suggest data',
            ))

    def on_release(self, params):
        super(MakeRazladkiSuggestData, self).on_release(params)
        self.mark_released_resources(params["release_status"], ttl=3)
