# coding=utf-8

from __future__ import unicode_literals

import logging
import sandbox.projects.resource_types as rt
from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common import dolbilka2
import resource_types as lrt

logger = logging.getLogger(__name__)


class ShootRecommenderManager(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Requirements):
        pass

    class Caches(sdk2.Requirements.Caches):
        pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        dolbilka_param = dolbilka2.DolbilkaExecutor2.Parameters
        requests = sdk2.parameters.Resource(
            'File with requests',
            resource_type=rt.PLAIN_TEXT_QUERIES,
            default=2687332304
        )
        host = sdk2.parameters.String(
            "Host we are shooting at",
            default="itditp-base-test.hamster.yandex.ru"
        )
        recommender_manager_host = sdk2.parameters.String(
            "Host with recommender manager binary running",
            default="sas1-9493.search.yandex.net:16735"
        )
        timeout = sdk2.parameters.Integer(
            "Timeout for http requests",
            default=1000000
        )

    def _get_resource_data_path(self, resource):
        return str(sdk2.ResourceData(resource).path)

    def _prepare_bullets(self):
        bullets_resource = lrt.BULLETS_TO_SHOOT_RECOMMENDER_MANAGER(self, 'Requests we use in shooting', 'bullets.txt')
        requests_path = self._get_resource_data_path(self.Parameters.requests)

        bullets = []
        extra_cgi_list = [
            "waitall=da",
            "timeout={}".format(self.Parameters.timeout),
            "reqinfo=dolbilo",
            "srcrwr=RECOMMENDER_MANAGER:{}:{}".format(self.Parameters.recommender_manager_host, self.Parameters.timeout),
            "srcrwr=RECOMMENDER_SETUP:{}:{}".format(self.Parameters.recommender_manager_host, self.Parameters.timeout),
            "hr=da",
            "recommender_debug_info=true",
        ]
        extra_cgi = '&'.join(extra_cgi_list)
        with open(requests_path, 'r') as file:
            requests_list = file.read().split('\n')
            for req in requests_list:
                bullets.append("{}&{}".format(req, extra_cgi))

        bullets_path = str(bullets_resource.path)
        with open(bullets_path, 'w') as bullets_file:
            bullets_file.write('\n'.join(bullets))

        return bullets_path

    def _create_dolbilo_plan(self, bullets_path):
        d_planner = dolbilka2.DolbilkaPlanner2()
        plan = lrt.DOLBILKA_PLAN_FOR_RECOMMENDER_MANAGER(self, 'Dolbilka plan', 'd-plan')
        plan_path = d_planner.create_plan(bullets_path, host=self.Parameters.host, result_path=self._get_resource_data_path(plan))
        return plan_path

    def _execute_dolbilo(self, plan):
        logger.info("Creating DolbilkaExecutor2...")
        d_executor = dolbilka2.DolbilkaExecutor2(self.Parameters.dolbilka_param)
        logger.info("DolbilkaExecutor2 created!")
        dump_resource = lrt.DOLBILKA_DUMP_FOR_RECOMMENDER_MANAGER(self, 'Dolbilka dump', 'd-dump')
        dump_path = d_executor.run_session(plan, host=self.Parameters.host, dump=self._get_resource_data_path(dump_resource), save_answers=True)
        return dump_path

    def _get_results(self, dumper, dump):
        results = dumper.get_results(dump)
        results_resource = lrt.RESULTS_FOR_RECOMMENDER_MANAGER(self, 'Results after dolbilka session', 'results.txt')
        results_path = str(results_resource.path)
        logger.info("Writing results...")
        with open(results_path, "w") as file:
            file.write(results)

    def _get_responses(self, dumper, dump):
        responses_resource = lrt.RESPONSES_FOR_RECOMMENDER_MANAGER(self, 'Responses after dolbilka session', 'responses.txt')
        responses_path = str(responses_resource.path)
        logger.info("Writing responses...")
        with open(responses_path, "w") as file:
            dumper.get_responses(dump, file, parsed_http_body=True)

    def _get_used_requests(self, dumper, dump):
        requests_resource = lrt.REQUESTS_USED_IN_SHOOTING_RECOMMENDER_MANAGER(self, 'Requests used in dolbilka session', 'requests.txt')
        requests_path = str(requests_resource.path)
        logger.info("Writing used requests...")
        with open(requests_path, "w") as file:
            dumper.get_requests(dump, file)

    def on_execute(self):
        bullets_path = self._prepare_bullets()
        d_plan = self._create_dolbilo_plan(bullets_path)
        d_dump = self._execute_dolbilo(d_plan)
        d_dumper = dolbilka2.DolbilkaDumper2()
        self._get_results(d_dumper, d_dump)
        self._get_used_requests(d_dumper, d_dump)
        self._get_responses(d_dumper, d_dump)
