# coding=utf-8
from __future__ import unicode_literals

import logging
from sandbox.projects.recommender_manager.shoot_recommender_manager import resource_types as shooting_resource_types
from sandbox import sdk2
from sandbox.projects.common import binary_task
import resource_types as lrt

logger = logging.getLogger(__name__)


class SummarizeShootingRecommenderManager(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Requirements):
        pass

    class Caches(sdk2.Requirements.Caches):
        pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        responses = sdk2.parameters.Resource(
            'File with stacked json responses',
            resource_type=shooting_resource_types.RESPONSES_FOR_RECOMMENDER_MANAGER,
            default=3000598917
        )

    def _get_resource_data_path(self, resource):
        return str(sdk2.ResourceData(resource).path)

    def _decode_stacked_json_from_file(self, data, pos=0):
        logger.info("Decoding stacked jsons from file...")
        import re
        from json import JSONDecoder

        decoder = JSONDecoder()
        not_whitespace = re.compile(r'[^\s]')
        while True:
            match = not_whitespace.search(data, pos)
            if not match:
                return
            pos = match.start()
            try:
                json_object, pos = decoder.raw_decode(data, pos)
            except ValueError:
                raise
            yield json_object

    def _is_response_error(self, json_response):
        return "error" in json_response

    def _response_has_recommender_debug_info(self, json_response):
        return "data" in json_response and len(json_response) > 0 and "recommender_debug_info" in json_response["data"][0]

    def _get_not_respond_percentage(self, json_response):
        debug_info = json_response["data"][0]["recommender_debug_info"]
        base_search_not_respond = float(debug_info["BaseSearchNotRespondCount"])
        base_search_count = float(debug_info["BaseSearchCount"])
        return round(base_search_not_respond / base_search_count, 2)

    def _calculate_not_respond_ratio_quantiles(self, summary):
        import numpy as np
        values = np.array(summary["base_search_not_respond_ratios"])
        for quantile in [0.50, 0.90, 0.95, 0.99]:
            summary["base_search_not_respond_ratio_q{}".format(int(quantile * 100))] = round(np.quantile(values, quantile), 2)

    def _save_summary(self, summary):
        import json
        summary_resource = lrt.SUMMARY_AFTER_SHOOTING_RECOMMENDER_MANAGER(self, 'Summary after shooting recommender manager', 'summary.txt')
        summary_path = str(summary_resource.path)
        logger.info("Writing summary...")
        with open(summary_path, "w") as file:
            json.dump(summary, file)

    def on_execute(self):
        summary = {
            "error_resonses": 0,
            "error_messages": [],
            "responses_with_debug_info": 0,
            "base_search_not_respond_ratios": [],
            "base_search_not_respond_ratio_q50": 1.0,
            "base_search_not_respond_ratio_q90": 1.0,
            "base_search_not_respond_ratio_q95": 1.0,
            "base_search_not_respond_ratio_q99": 1.0,
            "unknown_responses": 0
        }
        responses_path = self._get_resource_data_path(self.Parameters.responses)
        with open(responses_path, "r") as f:
            for json_response in self._decode_stacked_json_from_file(f.read()):
                if self._is_response_error(json_response):
                    summary["error_resonses"] += 1
                    summary["error_messages"].append(json_response["error"])
                elif self._response_has_recommender_debug_info(json_response):
                    summary["responses_with_debug_info"] += 1
                    summary["base_search_not_respond_ratios"].append(
                        self._get_not_respond_percentage(json_response)
                    )
                else:
                    summary["unknown_responses"] += 1

        logger.info("Decoded stacked jsons from file OK")
        self._calculate_not_respond_ratio_quantiles(summary)
        self._save_summary(summary)
