# coding=utf-8

from __future__ import unicode_literals

import json
import logging
from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.common.errors import TaskError
from sandbox.projects.common.betas.yappy_api import YappyApi

logger = logging.getLogger(__name__)


class UpdateResourcesForBetaComponent(binary_task.LastBinaryTaskRelease, sdk2.Task):

    class Requirements(sdk2.Requirements):
        pass

    class Caches(sdk2.Requirements.Caches):
        pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        token_owner = sdk2.parameters.String(
            "Token owner",
            default="robot-itditp"
        )
        nanny_token_name = sdk2.parameters.String(
            "Nanny oauth Token",
            default="nanny_oauth_token"
        )
        yappy_token_name = sdk2.parameters.String(
            "Yappy auth token",
            default="yappy_auth_token"
        )
        beta_name = sdk2.parameters.String(
            "Beta name",
            default="itditp-base-test"
        )
        beta_component_type = sdk2.parameters.String(
            "Beta component",
            default="web-mmeta-test"
        )
        nanny_service = sdk2.parameters.String(
            "Nanny service (presumably active) we take resources from",
            default="sas_itditp_mmeta_yp"
        )
        nanny_resource_to_component_local_path = sdk2.parameters.Dict(
            "Dict: {nanny resource} to {local_path_in_beta_component_type}",
            default={
                "httpsearch.for.releases": "mmeta.executable",
                "models.archive": "mmeta.models"
            }
        )
        nanny_url = sdk2.parameters.String(
            "Nanny URL",
            default="http://nanny.yandex-team.ru"
        )
        yappy_url = sdk2.parameters.String(
            "Yappy URL",
            default="https://yappy.z.yandex-team.ru"
        )

    def _get_resources_info(self, nanny_service, nanny_resource_to_component_local_path):
        import infra.nanny.nanny_services_rest.nanny_services_rest.client as nanny
        nanny_token = sdk2.Vault.data(self.Parameters.token_owner, self.Parameters.nanny_token_name)
        nanny_client = nanny.ServiceRepoClient(
            url=self.Parameters.nanny_url,
            token=nanny_token,
            attempts=5,
            timeout=1,
            delay=1,
        )
        attrs = nanny_client.get_active_runtime_attrs(nanny_service)
        sandbox_resources = attrs['content']['resources']['sandbox_files']
        resources_info = {}
        for sb_file_info in sandbox_resources:
            nanny_path = sb_file_info["local_path"]
            if nanny_path in nanny_resource_to_component_local_path:
                info = {
                    "manageType": "SANDBOX_RESOURCE",
                    "localPath": nanny_resource_to_component_local_path[nanny_path],
                    "sandboxResourceId": sb_file_info["resource_id"]
                }
                logger.info("Info got for resource {}: {}".format(nanny_path, json.dumps(info)))
                yappy_path = nanny_resource_to_component_local_path[nanny_path]
                resources_info[yappy_path] = info
            else:
                logger.info("Sandbox resource {} is not in nanny_resource_to_component_local_path!".format(nanny_path))

        logger.info("_get_resources_info result. Resources info: {}".format(json.dumps(resources_info)))
        return resources_info

    def _create_update_patches(self, api_beta):
        patches = api_beta["components"]

        resources = patches[0]["patch"]["resources"]
        for i in range(len(resources)):
            yappy_path = resources[i]["localPath"]
            if yappy_path in self.resources_info:
                logging.info("Changing resource info!")
                resources[i] = self.resources_info[yappy_path]
        return patches

    def _create_patches_and_update(self, beta_name):
        yappy_token = sdk2.Vault.data(self.Parameters.token_owner, self.Parameters.nanny_token_name)
        yappy = YappyApi(self.Parameters.yappy_url, token=yappy_token)
        api_beta = yappy.retrieve_api_beta(self.Parameters.beta_name)
        logging.info("Got api_beta: {}".format(api_beta))

        update_patches = self._create_update_patches(api_beta)
        yappy.update_patches(beta_name, update_patches)
        if not yappy.sleep_for_status(beta_name, "CONSISTENT"):  # wait until beta status is consistent after pushing updates
            raise TaskError('Beta is not consistent for too long...')

    def on_execute(self):
        self.resources_info = self._get_resources_info(
            self.Parameters.nanny_service,
            self.Parameters.nanny_resource_to_component_local_path
        )
        self._create_patches_and_update(self.Parameters.beta_name)
