import logging
import os
import re

from sandbox import common, sdk2
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.common.constants import constants as sdk_constants


STARTREK_API_URL = "https://st-api.yandex-team.ru/v2"
STARTREK_WWW_URL = 'https://st.yandex-team.ru'
STARTREK_TICKET_PATTERN = re.compile(r"([A-Z]+-\d+)")


class AbstractReconfTask(sdk2.Task):
    """
    Base class for ReConf tasks

    """
    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Startrek') as st_params:
            startrek_oauth_vault_owner = sdk2.parameters.String(
                'startrek oauth token vault owner',
            )
            startrek_oauth_vault_name = sdk2.parameters.String(
                'startrek oauth token vault name',
            )

            startrek_auto_ticket = sdk2.parameters.Bool(
                'automatic tickets (update opened/create new)',
                default=False,
                do_not_copy=True,
            )
            with startrek_auto_ticket.value[True]:
                startrek_auto_ticket_queue = sdk2.parameters.String(
                    'startrek queue',
                    do_not_copy=True,
                    required=True,
                )
                startrek_new_ticket_assignee = sdk2.parameters.Staff(
                    'new ticket assignee',
                )
            with startrek_auto_ticket.value[False]:
                startrek_tickets = sdk2.parameters.String(
                    'existing startrek tickets to notify',
                )

    def get_last_released_resource(self, task_input_params, resource_type):
        last_released_task = sdk2.Task.find(
            self.__class__,
            hidden=True,  # testenv produce hidden tasks
            input_parameters=task_input_params,
            status=(common.types.task.Status.RELEASED,)
        ).order(-sdk2.Task.id).first()

        if last_released_task is None:
            raise common.errors.TaskFailure(
                'No released tasks with such parameters found')

        logging.debug('Last released task is ' + str(last_released_task.id))

        resource = sdk2.Resource.find(
            resource_type,
            task=last_released_task,
        ).first()

        logging.debug('Last released resource is ' + str(resource.id))

        return resource

    def get_yamake_program_name(self, yamake_path):
        program_name = None

        with open(yamake_path) as f:
            for line in f:
                match = re.search('(PY.*)PROGRAM\((.*)\)', line)
                if match:
                    program_name = match.group(2)
                    break

        if program_name is None:
            raise RuntimeError(
                'Unable to get program name out of' + yamake_path)

        return program_name

    def make_binaries(self, targets, revision, build_dir='build_dir'):
        """
        Build inplace to save task run overhead in case of YA_MAKE task
        https://wiki.yandex-team.ru/sandbox/cookbook/#using-aapi-fuse

        """
        arcadia_url = sdk2.svn.Arcadia.ARCADIA_TRUNK_URL
        if revision:
            arcadia_url += '@' + str(revision)

        with arcadia_sdk.mount_arc_path(arcadia_url) as arcadia_root:
            self.Context.svn_revision = arcadia_sdk.mounted_path_svnversion(
                arcadia_root)['revision']

            arcadia_sdk.do_build(
                build_system=sdk_constants.YMAKE_BUILD_SYSTEM,
                source_root=arcadia_root,
                targets=targets.keys(),
                results_dir=build_dir,
                clear_build=False,
            )

            for tgt, dst in targets.items():
                yamake_path = os.path.join(arcadia_root, tgt, 'ya.make')
                binary_name = self.get_yamake_program_name(yamake_path)
                os.link(os.path.join(build_dir, tgt, binary_name), dst)

    def cmd(self, cmd, stdin=None, stdout=None, stderr=None, env=None,
            ok_codes=(0,)):
        """
        Run external command

        """
        logger = os.path.basename(cmd[0])
        with sdk2.helpers.ProcessLog(self, logger=logger) as pl:
            rcode = sdk2.helpers.subprocess.Popen(
                cmd,
                stdin=stdin,
                stdout=pl.stdout if stdout is None else stdout,
                stderr=pl.stderr if stderr is None else stderr,
                env=os.environ if env is None else env,
            ).wait()

            if rcode in ok_codes:
                return rcode

            raise common.errors.TaskError(
                ' '.join(cmd) + ' exited with ' + str(rcode))

    def collect_startrek_tickets(self, make_links=False):
        self.st_tickets = []

        if self.Context.linked_startrek_tickets:
            self.st_tickets = self.Context.linked_startrek_tickets

        elif self.Parameters.startrek_tickets:
            self.st_tickets = set(STARTREK_TICKET_PATTERN.findall(
                self.Parameters.startrek_tickets))

        elif self.Parameters.arc_rev is not None:
            # don't collect from commit message if st OAuth creds not provided
            if self.st_api is None:
                return

            commit_info = sdk2.svn.Arcadia.log(
                sdk2.svn.Arcadia.ARCADIA_TRUNK_URL,
                revision_from=self.Parameters.arc_rev,
                limit=1,
            )[0]

            self.st_tickets = set(STARTREK_TICKET_PATTERN.findall(
                commit_info['msg']))

        self.st_tickets = sorted(self.st_tickets)

        if make_links:
            for ticket in self.st_tickets:
                self.make_startrek_ticket_link(ticket)

    def create_startrek_ticket(self, queue, summary, **kwargs):
        reply = self.st_api.issues(queue=queue, summary=summary, **kwargs)
        self.st_tickets.append(reply['key'])
        self.make_startrek_ticket_link(reply['key'])

    def format_startrek_url(self, url, label=None):
        return '((' + url + ('' if label is None else ' ' + str(label)) + '))'

    def format_startrek_resource_url(self, url, label=None, text_mode=False):
        return self.format_startrek_url(
            url + ('?force_text_mode=1' if text_mode else ''),
            label=label,
        )

    def init_startrek_api(self):
        if self.Parameters.startrek_oauth_vault_owner:
            self.st_api = common.rest.Client(
                STARTREK_API_URL,
                sdk2.Vault.data(
                    self.Parameters.startrek_oauth_vault_owner,
                    self.Parameters.startrek_oauth_vault_name,
                )
            )
        else:
            self.st_api = None

    def make_startrek_comment(self, text):
        # borrowed from projects/sandbox/build_sandbox/__init__.py
        for ticket in self.st_tickets:
            try:
                self.st_api.issues[ticket].comments(text=text)
            except common.rest.Client.HTTPError:
                logging.exception("Failed to make comment to %s", ticket)

    def make_startrek_ticket_link(self, ticket_id):
        # borrowed from projects/sandbox/build_sandbox/__init__.py
        api_cmd = "remotelinks?notifyAuthor=false&backlink=false"
        try:
            self.st_api.issues[ticket_id][api_cmd](
                relationship="relates",
                key=self.id,
                origin="ru.yandex.sandbox",
            )
        except common.rest.Client.HTTPError:
            logging.exception("Failed to make link to %s", ticket_id)

    def on_finish(self, prev_status, status):
        self.Context.linked_startrek_tickets = self.st_tickets

    def on_execute(self):
        self.init_startrek_api()
        self.collect_startrek_tickets(make_links=True)

        if self.st_tickets:
            tmpl = '<a href="' + STARTREK_WWW_URL + '/{ticket}">{ticket}</a>'
            links = []
            for ticket in self.st_tickets:
                links.append(tmpl.format(ticket=ticket))

            self.set_info('Related tickets: ' + ', '.join(links),
                          do_escape=False)

    def on_release(self, params):
        super(AbstractReconfTask, self).on_release(params)

        self.init_startrek_api()
        self.collect_startrek_tickets()
