import json
import logging
import os
import shutil

from sandbox import common, sdk2
from sandbox.common.types.task import Semaphores
from sandbox.projects.common.juggler import jclient
from sandbox.projects.reconf.common import AbstractReconfTask


RES_DIR = 'res'
AGGRS_JSDK_NEW = os.path.join(RES_DIR, 'aggregates.jsdk.json')
AGGRS_JSDK_OLD = os.path.join(RES_DIR, 'aggregates.jsdk.json.BAK')
AGGRS_NESTED_DIFF = os.path.join(RES_DIR, 'aggregates.nested-diff')
AGGRS_UNIFIED_DIFF = os.path.join(RES_DIR, 'aggregates.unified-diff')


class ReconfJugglerAggregates(sdk2.Resource):
    """ReConf-Juggler aggregates bundle"""
    any_arch = False
    auto_backup = True
    releasable = True


class ReconfJuggler(AbstractReconfTask):
    """
    ReConf-Juggler aggregates builder/deployer.
    See https://a.yandex-team.ru/arc/trunk/arcadia/infra/reconf_juggler

    """
    class Requirements(sdk2.Requirements):
        client_tags = common.types.client.Tag.Group.LINUX
        ram = 16 * 1024  # Mb

    class Parameters(AbstractReconfTask.Parameters):
        with sdk2.parameters.Group('Aggregates') as aggrs_params:
            build_new_aggregates = sdk2.parameters.Bool(
                'build new aggregates',
                default=True,
            )

            with build_new_aggregates.value[True]:
                arc_uri = sdk2.parameters.String(
                    'aggregates builder path',
                    default='infra/reconf_juggler/examples/simple/bin',
                    required=True,
                )

                with sdk2.parameters.String('tools', multiline=True) as tools:
                    tools.values.FROM_SOURCES = tools.Value(default=True)
                    tools.values.LAST_RELEASED = None

                with tools.value['FROM_SOURCES']:
                    arc_rev = sdk2.parameters.Integer('revision', default=None)

            with build_new_aggregates.value[False]:
                resource = sdk2.parameters.Resource(
                    "existing resource",
                    resource_type=ReconfJugglerAggregates,
                    required=True,
                )

            with tools.value['LAST_RELEASED']:
                autodeploy = sdk2.parameters.Bool(
                    'autodeploy',
                    default=False,
                )

        with sdk2.parameters.Group('Deploy') as deploy_params:
            jsdk_mark = sdk2.parameters.String(
                'juggler-sdk mark',
            )
            juggler_oauth_vault_owner = sdk2.parameters.String(
                'juggler oauth token vault owner',
            )
            juggler_oauth_vault_name = sdk2.parameters.String(
                'juggler oauth token vault name',
            )

    def deploy_aggregates(self, filename):
        """
        Deploy aggregates to Juggler

        """
        env = os.environ.copy()
        env['JUGGLER_OAUTH_TOKEN'] = sdk2.Vault.data(
            self.Parameters.juggler_oauth_vault_owner,
            self.Parameters.juggler_oauth_vault_name,
        )

        logging.info('Deploy aggregates to Juggler from ' + filename)
        with open(filename) as aggregates_file:
            self.cmd(
                [self.jsdk_bin, 'load', '--mark', self.Parameters.jsdk_mark],
                env=env,
                stdin=aggregates_file,
            )

    def get_aggregates_from_juggler(self, mark, filename):
        """
        Fetch current aggregates from juggler

        """
        tag = 'a_mark_' + self.Parameters.jsdk_mark
        self.cmd([
            self.jsdk_bin, 'dump', '--tags', tag,
            '--dump-path', filename
        ])

    def build(self):
        self.make_startrek_comment(
            'Aggregates build started in task ' +
            self.format_startrek_url(
                common.utils.get_task_link(self.id),
                label=self.id,
            ) +
            '. Results will be reported here when ready',
        )

        os.mkdir(RES_DIR)
        self.jsdk_bin = os.path.join(RES_DIR, 'juggler-sdk')

        if self.Parameters.tools == 'FROM_SOURCES':
            logging.info('Building binaries')

            self.builder_bin = os.path.join(RES_DIR, 'builder')
            self.jdiff_bin = os.path.join(RES_DIR, 'jdiff')

            self.make_binaries(
                {
                    self.Parameters.arc_uri: self.builder_bin,
                    "infra/reconf_juggler/tools/jdiff/bin": self.jdiff_bin,
                    "contrib/python/juggler_sdk/cli": self.jsdk_bin,
                },
                self.Parameters.arc_rev,
            )
        elif self.Parameters.tools == 'LAST_RELEASED':
            logging.info('Fetching last released binaries')

            resource = self.get_last_released_resource(
                {'arc_uri': self.Parameters.arc_uri},
                ReconfJugglerAggregates,
            )
            self.builder_resource_id = resource.id
            path = str(sdk2.ResourceData(resource).path.absolute())

            self.builder_bin = os.path.join(path, 'builder')
            self.jdiff_bin = os.path.join(path, 'jdiff')
            shutil.copy(os.path.join(path, 'juggler-sdk'), self.jsdk_bin)
        else:
            raise NotImplementedError

        logging.info('Building aggregates')
        with open(AGGRS_JSDK_NEW, 'w') as out_file:
            self.cmd(
                [
                    self.builder_bin,
                    '--dump-resolved', os.path.join(RES_DIR, 'resolved.json'),
                    '--log', 'DEBUG',
                    '--target', 'jsdk_dump',
                ],
                stdout=out_file,
            )

        self.has_diff = False
        if self.Parameters.jsdk_mark:
            # build diffs before resource (should contain diffs)
            self.has_diff = self.diff() != 0

        if self.Parameters.tools == 'LAST_RELEASED' and self.Parameters.autodeploy:
            resource = self.get_last_released_resource(
                {'arc_uri': self.Parameters.arc_uri},
                ReconfJugglerAggregates,
            )
            if self.builder_resource_id != resource.id:
                raise common.errors.TaskFailure(
                    'New aggregates builder released since build started.')

        resource = ReconfJugglerAggregates(
            self,
            description='Juggler aggregates and related stuff',
            path=RES_DIR,
        )
        sdk2.ResourceData(resource).ready()

        comment = ['Aggregates build finished: ' + self.format_startrek_url(
            common.utils.get_resource_link(resource.id), label=resource.id)]

        if self.has_diff:
            base_url = resource.http_proxy
            diff_url = base_url + '/' + os.path.basename(AGGRS_NESTED_DIFF)

            self.set_info(
                'Difference in aggregates <a href="' + diff_url + '">found</a>',
                do_escape=False,
            )

            if self.Parameters.startrek_auto_ticket:
                self.create_startrek_ticket(
                    queue=self.Parameters.startrek_auto_ticket_queue,
                    summary='Build aggregates by ' + self.Parameters.arc_uri,
                    description='Subj.',
                    assignee=self.Parameters.startrek_new_ticket_assignee,
                )

            comment.append('')
            comment.append('Please verify changes using {} or {} diff.'.format(
                self.format_startrek_resource_url(
                    diff_url,
                    label='nested',
                    text_mode=True,
                ),
                self.format_startrek_resource_url(
                    base_url + '/' + os.path.basename(AGGRS_UNIFIED_DIFF),
                    label='unified',
                    text_mode=True,
                )
            ))
            comment.append(
                'If all looks reasonable, just release (status does not '
                'matter) {} to deploy aggregates to Juggler'.format(
                    self.format_startrek_url(
                        common.utils.get_task_link(self.id),
                        label='task',
                    ),
                ),
            )
        else:
            self.set_info('No difference in aggregates')
            comment.append('No difference in aggregates')

        self.make_startrek_comment('\n'.join(comment))

        return resource

    def deploy(self, resource):
        self.make_startrek_comment(
            'Aggregates deploy started in task ' +
            self.format_startrek_url(
                common.utils.get_task_link(self.id),
                label=self.id,
            ),
        )

        res_data = sdk2.ResourceData(resource)
        res_root = os.path.dirname(str(res_data.path.absolute()))
        self.jsdk_bin = os.path.join(res_root, RES_DIR, 'juggler-sdk')

        logging.info('Dumping current aggregates from Juggler')
        self.get_aggregates_from_juggler(
            self.Parameters.jsdk_mark,
            'jsdk_current'
        )

        with open('jsdk_current') as f:
            current_aggregates = json.load(f)

        with open(os.path.join(res_root, AGGRS_JSDK_OLD)) as f:
            previous_aggregates = json.load(f)

        if current_aggregates != previous_aggregates:
            raise common.errors.TaskFailure(
                'Aggregates in Juggler had changed since diff build.'
            )

        self.deploy_aggregates(os.path.join(res_root, AGGRS_JSDK_NEW))

        jsdk_backup = os.path.basename(AGGRS_JSDK_OLD)
        self.make_startrek_comment(
            'Aggregates deploy finished\n\nBackup for previous aggregates ' +
            'available in ((' + resource.http_proxy + '/' + jsdk_backup +
            ' ' + jsdk_backup + '))'
        )

    def diff(self):
        logging.info('Dumping current aggregates from Juggler')
        self.get_aggregates_from_juggler(
            self.Parameters.jsdk_mark,
            AGGRS_JSDK_OLD,
        )

        logging.info('Calculating diffs for aggregates')
        exit_code = self.cmd(
            [
                self.jdiff_bin,
                '--ifmt', 'jsdk',
                '--ofmt', 'nested',
                '--ignore-default-opts',
                '--ignore-mark-tags',
                '--ofile', AGGRS_NESTED_DIFF,
                AGGRS_JSDK_OLD, AGGRS_JSDK_NEW,
            ],
            ok_codes={0, 8}
        )
        exit_code = self.cmd(
            [
                self.jdiff_bin,
                '--ifmt', 'jsdk',
                '--ofmt', 'unified',
                '--ignore-default-opts',
                '--ignore-mark-tags',
                '--ofile', AGGRS_UNIFIED_DIFF,
                AGGRS_JSDK_OLD, AGGRS_JSDK_NEW,
            ],
            ok_codes={0, 8}
        ) or exit_code

        return exit_code

    def on_execute(self):
        super(ReconfJuggler, self).on_execute()

        if self.Parameters.build_new_aggregates:
            self.build()
            if self.Parameters.autodeploy:
                if self.has_diff:
                    # emit warns to be able to track deploys using charts
                    self.send_juggler_event(status='WARN', desc='Diff present')
                    self.run_deploy_task()
                else:
                    self.send_juggler_event(desc='No diff')

        else:
            self.deploy(self.Parameters.resource)

    def on_release(self, params):
        super(ReconfJuggler, self).on_release(params)
        self.mark_released_resources(params['release_status'], ttl=180)
        self.run_deploy_task()

    def run_deploy_task(self):
        # separated task required to be able to use semaphores
        deployer = ReconfJuggler(
            self,
            description='Deploy aggregates for task <a href={}>{}</a>'.format(
                common.utils.get_task_link(self.id),
                self.id,
            ),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
        )

        deployer.Parameters.build_new_aggregates = False
        deployer.Parameters.resource = \
            ReconfJugglerAggregates.find(task=self).first()

        # ... tried to copy whole parameters but with no luck =(
        deployer.Parameters.jsdk_mark = str(self.Parameters.jsdk_mark)
        deployer.Parameters.juggler_oauth_vault_owner = str(
            self.Parameters.juggler_oauth_vault_owner)
        deployer.Parameters.juggler_oauth_vault_name = str(
            self.Parameters.juggler_oauth_vault_name)

        deployer.Parameters.startrek_oauth_vault_owner = str(
            self.Parameters.startrek_oauth_vault_owner)
        deployer.Parameters.startrek_oauth_vault_name = str(
            self.Parameters.startrek_oauth_vault_name)
        deployer.Parameters.startrek_tickets = ', '.join(self.st_tickets)

        # use semaphore to avoid concurrent deploy
        deployer.Requirements.semaphores = Semaphores(
            acquires=[Semaphores.Acquire(
                name='juggler-sdk-mark-{}'.format(self.Parameters.jsdk_mark),
                capacity=1,
            )]
        )

        deployer.save()
        deployer.enqueue()

    def send_juggler_event(self, status='OK', desc='Ok'):
        jclient.send_events_to_juggler(
            'reconf_juggler_sync',
            jclient.to_juggler_service_name(self.Parameters.jsdk_mark),
            status,
            desc,
        )
