import os
import time
import logging
import requests
from sandbox import sandboxsdk
import sandbox.common.types.resource as ctr
from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common.search.components import FusionSearch
from sandbox.projects.common.nanny.nanny import NannyClient
from sandbox.projects.saas.common.resources import SaasRtyserverConfigsBundle


class RefreshNaming():
    def __init__(self, configs_suffix):
        suffix = "-{0}".format(configs_suffix) if configs_suffix else ""
        self.env_configs_suffix = configs_suffix
        self.basesearch_conf = "basesearch-refresh" + suffix
        self.oxygen_conf = "OxygenOptions.cfg" + suffix
        self.oxygen_med_conf = "OxygenOptionsMed.cfg" + suffix
        self.oxygen_rt_conf = "OxygenOptionsRt.cfg" + suffix
        self.environment_conf = "environment" + suffix
        self.rtyserver_conf = "rtyserver.conf-{0}".format(configs_suffix if configs_suffix else "common")

        self.filenames = {
            self.environment_conf : None,
            self.rtyserver_conf : resource_types.FUSION_SEARCH_CONFIG,
            self.basesearch_conf : resource_types.SEARCH_CONFIG,
            self.oxygen_conf: resource_types.FUSION_OXYGEN_CONFIG,
            "distributors.conf": None,
            "models.archive": resource_types.DYNAMIC_MODELS_ARCHIVE,
            "models": resource_types.RTYSERVER_MODELS,
            "canon_configs": SaasRtyserverConfigsBundle,
            "rtyserver": resource_types.RTYSERVER_EXECUTABLE,
        }


class RunFusionProductionEnv(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        disk_space = 143360
        ram = 81920

    class Parameters(sdk2.Task.Parameters):
        nanny_service_name = sdk2.parameters.String(
            'Nanny service name',
            required=True,
            default_value="saas_yp_samohod",
        )

        configs_suffix = sdk2.parameters.String(
            'Configs suffix (for canon_configs)',
            required=False,
            default_value="",
        )

        docs_to_index = sdk2.parameters.Integer(
            'Document to index from distributors',
            required=True,
            default_value=1000,
        )

        rtyserver_binary = sdk2.parameters.Resource(
            'RTYServer executable',
            resource_type=[resource_types.RTYSERVER_EXECUTABLE, resource_types.RTYSERVER],
            state=(ctr.State.READY, ),
            required=False,
        )

        fusion_search_config = sdk2.parameters.Resource(
            'rtyserver.conf-common',
            resource_type=resource_types.FUSION_SEARCH_CONFIG,
            state=(ctr.State.READY, ),
            required=False,
        )

        base_search_config = sdk2.parameters.Resource(
            'basesearch-refresh',
            resource_type=resource_types.SEARCH_CONFIG,
            state=(ctr.State.READY, ),
            required=False,
        )

        oxygen_config = sdk2.parameters.Resource(
            'OxygenOptions.cfg',
            resource_type=resource_types.FUSION_OXYGEN_CONFIG,
            state=(ctr.State.READY, ),
            required=False,
        )

        canon_configs = sdk2.parameters.Resource(
            'Configs bundle',
            resource_type=SaasRtyserverConfigsBundle,
            state=(ctr.State.READY, ),
            required=False,
        )

        models_archive = sdk2.parameters.Resource(
            'models.archive',
            resource_type=resource_types.DYNAMIC_MODELS_ARCHIVE,
            state=(ctr.State.READY, ),
            required=False,
        )

        models = sdk2.parameters.Resource(
            'models',
            resource_type=resource_types.RTYSERVER_MODELS,
            state=(ctr.State.READY, ),
            required=False,
        )

    class Context(sdk2.Task.Context):
        files = {}
        resources = {}

    def get_static_files(self, static_section, filenames_to_get, params_files_names):
        res = {}
        for item in static_section:
            local_path = item["local_path"]
            if local_path in filenames_to_get:
                if local_path in params_files_names:
                    logging.info('Nanny static resource "{}" is ignored, because of an override in task params'.format(local_path))
                    continue
                res[local_path] = item["content"]
        return res

    def get_url_files(self, url_section, filenames_to_get, params_files_names):
        res = {}
        for item in url_section:
            local_path = item["local_path"]
            if local_path in filenames_to_get:
                if local_path in params_files_names:
                    logging.info('Nanny URL resource "{}" is ignored, because of an override in task params'.format(local_path))
                    continue
                reply = requests.get(item["url"], timeout=10)
                reply.raise_for_status()
                res[local_path] = reply.text
        return res

    def lookup_resource(self, nanny_resource_desc):
        if "resource_id" in nanny_resource_desc:
            logging.info("Fetching resource by id: %s", str(nanny_resource_desc))
            return sdk2.Resource[nanny_resource_desc["resource_type"]].find(
                state=ctr.State.READY,
                id=nanny_resource_desc["resource_id"],
            ).first()
        # Lookup by task id
        logging.info("Fetching resource by task: %s", str(nanny_resource_desc))
        task_id = nanny_resource_desc["task_id"]
        # # For debugging in a local sandbox, remap task_ids manually:
        # dbg_local_remap = { u"655057160" : u"8" }
        # task_id = dbg_local_remap.get(task_id, task_id)
        task = sdk2.Task[task_id]
        return sdk2.Resource[nanny_resource_desc["resource_type"]].find(
            task=task).limit(1).first()

    def get_resources(self, sanbox_files_section, filenames_to_get, params_files_names):
        res = {}
        for item in sanbox_files_section:
            local_path = item["local_path"]
            if local_path in filenames_to_get:
                if local_path in params_files_names:
                    logging.info('Nanny sandbox resource "{}" is ignored, because of an override in task params'.format(local_path))
                    continue
                res_obj = self.lookup_resource(item)
                if res_obj is not None:
                    res[local_path] = res_obj
                else:
                    logging.error('A sandbox resource is missing! local_path: {}'.format(local_path))
        return res

    def on_execute(self):
        time_zero = time.time()
        time_labels = []
        params_files = dict()

        if self.Parameters.configs_suffix:
            suffix = self.Parameters.configs_suffix
        else:
            suffix = ""

        naming = RefreshNaming(suffix)

        if self.Parameters.rtyserver_binary:
            rtyserver_binary = str(sdk2.ResourceData(self.Parameters.rtyserver_binary).path)
            params_files["rtyserver"] = rtyserver_binary
        else:
            rtyserver_binary = "./rtyserver"

        if self.Parameters.fusion_search_config:
            fusion_search_config = str(sdk2.ResourceData(self.Parameters.fusion_search_config).path)
            params_files[naming.rtyserver_conf] = fusion_search_config
        else:
            fusion_search_config = naming.rtyserver_conf

        if self.Parameters.base_search_config:
            base_search_config = str(sdk2.ResourceData(self.Parameters.base_search_config).path)
            params_files[naming.basesearch_conf] = base_search_config
        else:
            base_search_config = naming.basesearch_conf

        if self.Parameters.oxygen_config:
            params_files[naming.oxygen_conf] = str(sdk2.ResourceData(self.Parameters.oxygen_config))

        if self.Parameters.models_archive:
            models_archive = str(sdk2.ResourceData(self.Parameters.models_archive).path.parent)
            params_files["models.archive"] = models_archive
        else:
            models_archive = '.'

        if self.Parameters.models:
            models = str(sdk2.ResourceData(self.Parameters.models).path)
            params_files["models"] = models
        else:
            models = "./models"

        if self.Parameters.canon_configs:
            canon_dir = str(sdk2.ResourceData(self.Parameters.canon_configs).path)
            params_files["canon_configs"] = canon_dir
        else:
            canon_dir = "./canon_configs"

        token = sdk2.Vault.data("RTYSERVER-ROBOT", "nanny_token")
        service = self.Parameters.nanny_service_name
        nanny_url = "https://nanny.yandex-team.ru/"
        client = NannyClient(nanny_url, token)
        res = client.get_service_runtime_attrs(service)
        resources_descr = res["content"]["resources"]

        local_files = self.get_static_files(resources_descr.get("static_files", {}), naming.filenames, params_files)
        local_files.update(self.get_url_files(resources_descr.get("url_files", {}), naming.filenames, params_files))
        resources = self.get_resources(resources_descr.get("sandbox_files", {}), naming.filenames, params_files)

        for name, resource in resources.iteritems():
            self.Context.resources[name] = resource.id

        store_local_files(local_files)
        store_nanny_resources(resources)
        store_param_files(params_files)
        apply_canon_configs(naming, canon_dir)
        make_local_fixes(naming)

        workdir_path = sandboxsdk.paths.make_folder("workdir")
        detach_path = sandboxsdk.paths.make_folder("detach")
        state_path = sandboxsdk.paths.make_folder("state")
        rtindex_path = sandboxsdk.paths.make_folder("shm")
        medindex_path = sandboxsdk.paths.make_folder("medwebcache")

        # create empty file, production file generated by jinja
        templates_path = sandboxsdk.paths.make_folder("templates")
        open(templates_path + "/replicas.conf", "w").close()
        open("replicas.conf", "w").close()

        fusion_search = FusionSearch(workdir_path, None,
                                     fusion_config_path=fusion_search_config,
                                     config_file=base_search_config,
                                     environment_file_path=naming.environment_conf,
                                     binary=rtyserver_binary,
                                     static_data_path=models,
                                     extra_params={"CONFIG_PATH": ".",
                                                   "MODELS_PATH": models_archive,
                                                   "DETACH_DIRECTORY": detach_path,
                                                   "STATE_ROOT": state_path,
                                                   "RTINDEX_DIRECTORY": rtindex_path,
                                                   "MEDINDEX_DIRECTORY": medindex_path,
                                                   "BSCONFIG_INAME": "refresh",
                                                   "CPU_LIMIT": "8",
                                                   "PQ_CONSUMER": "saas-refresh-quick-tester",
                                                   "CONFIGS_SUFFIX": naming.env_configs_suffix
                                                   },
                                     patch_params={"Server.ModulesConfig.DOCFETCHER.SearchOpenDocAgeSec": "3000000"},
                                     from_sdk2=True
                                     )
        logging.info("Fusion config: %s", fusion_search.fusion_config_path)
        # logging.info("Fusion cmd: %s", fusion_search.run_cmd_patcher(None))
        time_labels.append(str.format("Starting rtyserver: {0}", time.time() - time_zero))
        fusion_search.start()
        fusion_search.wait(timeout=3600)  # wait for DB backup to be fetched
        time_labels.append(str.format("Fetched backup: {0}", time.time() - time_zero))

        # move docfetcher position
        set_df_attempts = 3
        while set_df_attempts > 0:
            result = fusion_search.run_fusion_command("set_docfetcher_timestamp&age=1800&increment_only=1", 300)
            if result.get('result', False):
                break
            time.sleep(10)
            set_df_attempts = set_df_attempts - 1
        if set_df_attempts > 0:
            time_labels.append(str.format("Set docfetcher timestamp: {0}", time.time() - time_zero))
        else:
            time_labels.append(str.format("Set docfetcher timestamp failed, time: {0}", time.time() - time_zero))

        # waiting for |docs_to_index| documents to be fetched by the indexer
        while True:
            result = fusion_search.get_info_server()['result']
            docs_in_disk = int(result['docs_in_disk_indexers'])
            final_indices_count = count_final_indices(result)
            logging.info("Fusion has: %u docs in disk index and %u final indices", docs_in_disk, final_indices_count)
            if docs_in_disk > self.Parameters.docs_to_index:
                break
            time.sleep(30)
        time_labels.append(str.format("Fetched {0} documents: {1}", self.Parameters.docs_to_index, time.time() - time_zero))

        fusion_search.run_fusion_command("pause_docfetcher")
        fusion_search.run_fusion_command("disable_indexing")

        time.sleep(30)  # 10s may be not enough to empty indexing queues
        fusion_search.run_fusion_command("wait_empty_indexing_queues")
        fusion_search.run_fusion_command("reopen_indexes")
        time.sleep(30)  # 10s may be not enough to close indexers
        fusion_search.run_fusion_command("create_merger_tasks")

        while True:
            result = fusion_search.get_info_server()['result']
            docs_in_disk = int(result['docs_in_disk_indexers'])
            final_indices_count = count_final_indices(result)
            logging.info("Fusion merging has: %u docs in disk index and %u final indices", docs_in_disk, final_indices_count)
            if docs_in_disk == 0 and final_indices_count == 1:
                break
            time.sleep(30)
        time_labels.append(str.format("Merge finished: {0}", time.time() - time_zero))
        fusion_search.kill_softly()
        time_labels.append(str.format("rtyserver shutdown: {0}", time.time() - time_zero))
        self.Context.report = str.format("<pre>{0}</pre>", "\n".join(time_labels))
        self.Context.save()

    @sdk2.footer()
    def footer(self):
        return self.Context.report or "No report"


def count_final_indices(info_server_result):
    indices = info_server_result['indexes']
    return sum((index['type'] == 'FINAL' or index['type'] == 'PREPARED') and index['realm'] == 'Persistent' for _, index in indices.iteritems())


def store_local_files(local_files):
    for fname, fcontent in local_files.iteritems():
        fu.write_file(fname, fcontent)


def store_nanny_resources(nanny_resources):
    for fname, res in nanny_resources.iteritems():
        resource_data = sdk2.ResourceData(res)  # synchronizing resource data on disk
        resource_path = resource_data.path
        if not os.path.exists(fname):
            os.symlink(str(resource_path), fname)
            logging.info('ln -s {} {}'.format(str(resource_path), fname))


def store_param_files(params_files):
    for fname, path in params_files.iteritems():
        if not os.path.exists(fname):
            os.symlink(str(path), fname)
            logging.info('ln -s {} {}'.format(str(path), fname))


def apply_canon_configs(refresh_naming, canon_dir):
    if not os.path.exists(canon_dir):
        return

    for config_file, _ in refresh_naming.filenames.iteritems():
        canon_config_file = os.path.join(canon_dir, config_file)
        if not os.path.exists(canon_config_file):
            continue
        if not os.path.exists(config_file):
            os.symlink(canon_config_file, config_file)
            logging.info('Will use canon file {}'.format(config_file))
        else:
            logging.info('Will ignore canon file {}'.format(config_file))


def make_local_fixes(refresh_naming):
    if not os.path.exists(refresh_naming.oxygen_med_conf):
        os.symlink(refresh_naming.oxygen_conf, refresh_naming.oxygen_med_conf)
    if not os.path.exists(refresh_naming.oxygen_rt_conf):
        os.symlink(refresh_naming.oxygen_conf, refresh_naming.oxygen_rt_conf)
