import os
import re
import time
import logging
from collections import deque

from sandbox import sdk2
from sandbox.sdk2.vcs import svn
from sandbox.projects.common import string
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common.build import resource_types
from sandbox.projects.release_machine import core as rm_core
from sandbox.projects.release_machine import rm_utils
from sandbox.projects.release_machine.helpers import commit
from sandbox.projects.release_machine.helpers import svn_helper as rm_svn
from sandbox.projects.release_machine.helpers import merge_helper as rm_merge
import sandbox.common.types.task as ctt
import sandbox.projects.common.tasklet.executor as tasklet_executor
import sandbox.projects.common.vcs.arc as arc_api
import sandbox.projects.release_machine.core.const as rm_const


NEW_ARC_TAG_CONTEXT__ARC_HASH = "arc_hash"
NEW_ARC_TAG_CONTEXT__BRANCH = "branch"
NEW_ARC_TAG_CONTEXT__TAG = "tag"


def get_arc_hash_for_revision(arc_client, rev):
    """
    Return arc hash for given svn revision.

    :param arc_client: instance for working with Arcadia repository through Arc Vcs.
    :param rev: svn revision from trunk
    :return: arc hash or None in case of exception
    """
    logging.info("Try to get arc hash for revision %s", rev)
    try:
        with arc_client.init_bare() as arcadia_src_dir:
            arc_hash = arc_client.svn_rev_to_arc_hash(arcadia_src_dir, str(rev))
        return arc_hash
    except Exception as exc:
        logging.exception("Got exception %s", exc)
        return None


def get_arc_hash_for_revision_in_branch(arc_client, rev, c_info, branch_num):
    """
    Return arc hash for given svn revision in branch.

    :param arc_client: instance for working with Arcadia repository through Arc Vcs.
    :param rev: svn revision from trunk
    :param c_info: Component info class instance
    :param branch_num: branch number
    :return: arc hash or None in case of exception
    """
    full_branch_path = c_info.full_branch_path(branch_num)
    logging.info("Try to get arc hash for revision %s in branch %s", rev, full_branch_path)
    svn_branch_log = svn.Arcadia.log(full_branch_path, revision_from="r0", revision_to="HEAD", stop_on_copy=True)
    logging.debug("Got svn branch log %s", svn_branch_log)
    tag_num = None
    for index, rev_log in enumerate(svn_branch_log):
        if rev_log["revision"] == rev:
            tag_num = index
            logging.debug("Got tag_number %s for revision %s", tag_num, rev)
            break
    if tag_num is None:
        logging.warning("Could not get tag_num for revision %s", rev)
        return None
    try:
        arc_full_branch_path = c_info.svn_cfg__arc_branch_path(branch_num)
        arc_branch_log = fetch_arc_branch_log(arc_client, arc_full_branch_path)
        arc_hash = arc_branch_log[tag_num]["commit"]
        logging.info("Got arc_hash %s for revision %s", arc_hash, rev)
        return arc_hash
    except Exception as exc:
        eh.log_exception("Got exception %s", exc)
        return None


def fetch_arc_branch_log(arc_client, arc_full_branch_path):
    """
    Fetch branch log.

    :param arc_client: instance for working with Arcadia repository through Arc Vcs.
    :param arc_full_branch_path: path to branch
    :return: list with dicts for each revision in branch, started from merge_base revision
    """
    with arc_client.mount_path(None, None, fetch_all=False, extra_params=["--vfs-version", "2"]) as arcadia_src_dir:
        arc_client.fetch(arcadia_src_dir, branch=arc_full_branch_path)
        arc_client.checkout(arcadia_src_dir, branch=arc_full_branch_path)

        return get_arc_branch_log(arc_client, arcadia_src_dir, arc_full_branch_path)


def get_arc_branch_log(arc_client, arcadia_src_dir, arc_full_branch_path):
    """
    Get filtered branch log from arc branch.

    Get all commits to arc branch, filter them by __ignored__ marker in commit message and return.
    :param arc_client: instance for working with Arcadia repository through Arc Vcs.
    :param arcadia_src_dir: mounted arcadia source dir
    :param arc_full_branch_path: path to branch
    :return:
    """
    branch_first_commit = string.all_to_unicode(arc_client.get_merge_base(arcadia_src_dir, branch=arc_full_branch_path))
    logging.debug("Got first arc_hash %s for branch %s", branch_first_commit, arc_full_branch_path)
    branch_log = arc_client.log(
        arcadia_src_dir,
        start_commit="{}^".format(branch_first_commit),
        end_commit="HEAD",
        as_dict=True,
    )[::-1]
    logging.debug("Got full branch log for arc branch %s: %s", arc_full_branch_path, branch_log)
    return filter_arc_ignored_commits(branch_log)


def get_all_branches(arc_client):
    """
    Return all branches in arc.

    :param arc_client: instance for working with Arcadia repository through Arc Vcs.
    :return: set with branch names
    """
    with arc_client.mount_path(None, None, fetch_all=False, extra_params=["--vfs-version", "2"]) as arcadia_src_dir:
        arc_client.fetch(arcadia_src_dir, branch="releases/")
        branch_response = arc_client.branch(arcadia_src_dir, all=True, as_dict=True)
        branches = {
            string.left_strip(branch["name"], "arcadia/")
            for branch in branch_response if branch["name"].startswith("arcadia/releases")
        }
    logging.debug("Got all branches: %s", branches)
    return branches


def get_synced_arc_commit(svn_rev_log):
    """
    Search for synced arc commit in svn log message.

    :param svn_rev_log: svn log for revision
    :return: arc hash if found else None
    """
    synced_arc_commit = rm_const.SYNCED_ARC_COMMIT.search(svn_rev_log["msg"])
    if synced_arc_commit:
        synced_arc_commit = synced_arc_commit.group(1)
        logging.debug("Got synced_arc_commit %s for svn_rev %s", synced_arc_commit, svn_rev_log["revision"])
        return synced_arc_commit
    logging.debug("Could not get synced arc_commit for svn_rev %s", svn_rev_log["revision"])
    return None


def _is_synced_with_arc_manual_commit(svn_rev_log_msg):
    """
    Search for synced arc manual commit in svn log message.

    :param svn_rev_log_msg: svn log message for revision
    :return: True if svn revision is synced with arc manual commit
    """
    synced_arc_manual_commit = rm_const.SYNCED_ARC_MANUAL_COMMIT_MSG.search(svn_rev_log_msg)
    if synced_arc_manual_commit:
        logging.debug("Svn revision %s is synced with arc manual commit")
        return True
    return False


def filter_ignored_revisions(svn_branch_log):
    """
    Skip svn revisions with flag `__ignored__` in commit message

    :param svn_branch_log: list with BranchCommit instances for svn branch
    :return: list with BranchCommit instances
    """
    revisions_to_skip = set()
    for rev_log in svn_branch_log[::-1]:
        if "__ignored__" in rev_log["msg"]:
            revisions_to_skip.add(rev_log["revision"])
            revisions_to_skip.update(rm_merge.get_ignored_revisions_from_commit_message(rev_log["msg"]))
    logging.debug("Got revisions to ignore: %s", revisions_to_skip)
    filtered_svn_branch_log = [rev_log for rev_log in svn_branch_log if rev_log["revision"] not in revisions_to_skip]
    logging.debug("Got filtered_svn_branch_log: %s", filtered_svn_branch_log)
    return filtered_svn_branch_log


def filter_arc_ignored_commits(arc_branch_log):
    """
    Skip arc commits with flag `__ignored__` in commit message

    :param arc_branch_log: list with BranchCommit instances for arc branch
    :return: list with BranchCommit instances
    """
    commits_to_skip = set()

    for commit_log in arc_branch_log[::-1]:

        # RMDEV-3300
        if commit_log["commit"] == "a895d1cf1afac6671cb4715138d0c44455f23e30":
            logging.info("Commit %s ignored: RMDEV-3300", commit_log["commit"])
            commits_to_skip.add(commit_log["commit"])
            continue

        if "__ignored__" in commit_log["message"]:
            logging.debug("Got ignored marker in %s", commit_log["commit"])
            commits_to_skip.add(commit_log["commit"])
            commits_to_skip.update(rm_merge.get_arc_ignored_revisions_from_commit_message(commit_log["message"]))

    logging.debug("Got commits to ignore: %s", commits_to_skip)

    filtered_arc_branch_log = [
        commit_log for commit_log in arc_branch_log if commit_log["commit"] not in commits_to_skip
    ]

    logging.debug("Got filtered arc branch log: %s", filtered_arc_branch_log)

    return filtered_arc_branch_log


def _get_merges_to_svn_branch(c_info, branch_num):
    """
    Parse svn branch log and return prepared BranchCommit instances.

    :param c_info: Component info class instance
    :param branch_num: branch number
    :return: list with BranchCommit instances
    """
    svn_full_branch_path = c_info.full_branch_path(branch_num)
    svn_branch_log = svn.Arcadia.log(
        os.path.join(svn_full_branch_path),
        revision_from="r0",
        revision_to="HEAD",
        stop_on_copy=True,
    )
    filtered_svn_branch_log = filter_ignored_revisions(svn_branch_log)
    logging.debug("Got filtered_svn_branch_log for branch %s: %s", svn_full_branch_path, filtered_svn_branch_log)
    svn_initial_revs = []
    for rev_log in filtered_svn_branch_log:
        merged_revs = rm_merge.get_merges_from_commit_message(rev_log["msg"])
        synced_arc_commit = get_synced_arc_commit(rev_log)
        if merged_revs:
            svn_initial_revs.append(
                rm_const.BranchCommit(
                    action_type=rm_const.ActionType.MERGE,
                    current_hash=synced_arc_commit,
                    current_rev=rev_log["revision"],
                    svn_revs=[merged_revs],
                )
            )
            continue
        rollbacked_revs = rm_merge.get_rollbacks_from_commit_message(rev_log["msg"])
        if rollbacked_revs:
            svn_initial_revs.append(
                rm_const.BranchCommit(
                    action_type=rm_const.ActionType.ROLLBACK,
                    current_hash=synced_arc_commit,
                    current_rev=rev_log["revision"],
                    svn_revs=[rollbacked_revs],
                )
            )
            continue
        is_manual_commit = _is_synced_with_arc_manual_commit(rev_log["msg"])
        if is_manual_commit:
            svn_initial_revs.append(
                rm_const.BranchCommit(
                    action_type=rm_const.ActionType.MANUAL_COMMIT,
                    current_hash=synced_arc_commit,
                    current_rev=rev_log["revision"],
                    svn_revs=[],
                )
            )
        elif rm_svn.SvnHelper.is_branching(rev_log):
            svn_initial_revs.append(
                rm_const.BranchCommit(
                    action_type=rm_const.ActionType.MANUAL_COMMIT,
                    current_rev=rev_log["revision"],
                    svn_revs=[],
                )
            )
    logging.debug("Svn initial revs: %s", svn_initial_revs)
    return svn_initial_revs


def _get_initial_arc_hash(arc_commit_log):
    """
    Prepare BranchCommit instance for arc commit.

    :param arc_commit_log: arc commit log
    :return: BranchCommit instance
    """
    commit_msg = arc_commit_log["message"]
    commit_hash = arc_commit_log["commit"]

    for parse_function, action_type in [
        (rm_merge.get_arc_cherry_pick_from_commit_message, rm_const.ActionType.MERGE),
        (rm_merge.get_arc_revert_from_commit_message, rm_const.ActionType.ROLLBACK),
        (rm_merge.get_arc_merges_from_commit_message, rm_const.ActionType.MERGE),
        (rm_merge.get_arc_rollbacks_from_commit_message, rm_const.ActionType.ROLLBACK),
    ]:
        initial_hashes = parse_function(commit_msg)
        if initial_hashes:
            logging.debug("Got inititial hash(es) for commit %s: %s", commit_hash, initial_hashes)
            return rm_const.BranchCommit(
                action_type=action_type,
                current_hash=commit_hash,
                arc_hashes=initial_hashes,
            )
    logging.debug("Could not get initial hash for commit %s, commit_msg: %s", commit_hash, commit_msg)
    return rm_const.BranchCommit(action_type=rm_const.ActionType.MANUAL_COMMIT, current_hash=commit_hash)


def check_commit_changeset_is_empty(arc_client, arc_commit_hash):
    """
    Check whether commit is empty

    :param arc_client: instance for working with Arcadia repository through Arc Vcs.
    :param arc_commit_hash: arc commit hash
    :return: True if arc commit is empty else False
    """
    with arc_client.init_bare() as arcadia_src_dir:
        arc_commit_info = arc_client.show(
            arcadia_src_dir,
            commit=arc_commit_hash,
            name_status=True,
            as_dict=True,
        )
        logging.debug("Got info for commit %s: %s", arc_commit_hash, arc_commit_info)
        if not arc_commit_info[0]["names"]:
            logging.debug("Got empty commit %s", arc_commit_hash)
            return True
        return False


def _get_merges_to_arc_branches(arc_client, c_info, branch_num):
    """
    Parse arc branch log and return prepared BranchCommit instances.

    :param arc_client: instance for working with Arcadia repository through Arc Vcs.
    :param c_info: Component info class instance
    :param branch_num: branch number
    :return: list with BranchCommit instances
    """
    arc_full_branch_path = c_info.svn_cfg__arc_branch_path(branch_num)
    branch_log = fetch_arc_branch_log(arc_client, arc_full_branch_path)
    initial_hashes = [
        rm_const.BranchCommit(
            action_type=rm_const.ActionType.MANUAL_COMMIT,
            current_hash=branch_log[0]["commit"],
            arc_hashes=[],
        )
    ]  # Empty list for branching commit
    clear_branch_log = []
    for commit_log in branch_log[1:]:   # Skip branching commit
        if not check_commit_changeset_is_empty(arc_client, commit_log["commit"]):
            clear_branch_log.append(commit_log)
    logging.debug("Clear merged revs: %s", clear_branch_log)
    initial_hashes.extend([_get_initial_arc_hash(commit_log) for commit_log in clear_branch_log])
    logging.debug("Arc merged revs: %s", initial_hashes)
    return initial_hashes


def _get_svn_revision_for_arc_commit_in_trunk(arc_client, arc_commit_hash):
    """
    Search svn revision equal to arc commit.

    :param arc_client: instance for working with Arcadia repository through Arc Vcs.
    :param arc_commit_hash: arc commit hash
    :return: svn revision if found equal revision else None
    """
    with arc_client.init_bare() as arcadia_src_dir:
        initial_commit_log = arc_client.log(
            arcadia_src_dir,
            start_commit=arc_commit_hash,
            max_count=1,
            as_dict=True,
        )
        if not initial_commit_log:
            logging.debug("Could not get log for %s", arc_commit_hash)
            return None
        logging.debug("Arc log %s", initial_commit_log)
        svn_rev = initial_commit_log[0]["revision"]
        logging.debug("Got svn revision %s for arc hash %s", svn_rev, arc_commit_hash)
        return svn_rev


def convert_arc_hashes_to_svn_revs(arc_client, arc_hashes, c_info, branch_number):
    """
    Search svn revisions equal to arc commits.

    If c_info is None, we should search for commits in trunk, else in c_info branch with `branch_number`.

    :param arc_client: instance for working with Arcadia repository through Arc Vcs.
    :param arc_hashes: list of arc commit hashes
    :param c_info: Component info class instance
    :param branch_number: branch number
    :return: list with svn revisions
    """
    result_revs = []
    if branch_number is None:
        for arc_hash in arc_hashes:
            result_revs.append(_get_svn_revision_for_arc_commit_in_trunk(arc_client, arc_commit_hash=arc_hash))
    else:
        branch_log = _get_merges_to_svn_branch(c_info, branch_number)
        for branch_commit in branch_log:
            if branch_commit.current_hash in arc_hashes:
                result_revs.append(branch_commit.current_rev)
    logging.debug("Got svn revisions %s for arc hashes %s", ", ".join(map(str, result_revs)), ", ".join(arc_hashes))
    return result_revs


def extract_revisions(revs_string, arc_client, reverse=False, c_info=None, branch_number=None):
    """
    Split revisions string into tokens, convert into svn revisions (if needed) and sort them.

    Strings examples are "100, r200, 300  400"
    or "605babfb59230d9e251145ab790ad6800aff9517, a205c0b86e62d346798285a8546c0a7be33136ce".

    :param arc_client: instance for working with Arcadia repository through Arc Vcs.
    :param revs_string: input revision descriptor string
    :param reverse: sort in reverse order (e.g. for rollback)
    :param c_info: Component info class instance
    :param branch_number: branch number
    :return: splitted revisions list
    """
    check = rm_merge.split_and_check_revs_correctness(revs_string)
    if check.ok:
        revs = check.result
        # Check that we have arc hashes in check.result
        if len(revs[0]) == rm_const.ARC_HASH_LENGTH:
            logging.debug("We should cast arc hashes %s to svn revisions", ", ".join(revs))
            revs = convert_arc_hashes_to_svn_revs(arc_client, revs, c_info, branch_number)
        return rm_core.Ok(sorted(map(int, revs), reverse=reverse))
    else:
        return check


def _update_svn_revisions_for_arc_commit(arc_client, arc_commit, svn_revs):
    """
    Update arc_commit.svn_revs list with revisions equal to arc_commit.arc_hashes.

    :param arc_client: instance for working with Arcadia repository through Arc Vcs.
    :param arc_commit: BranchCommit instance
    :param svn_revs: list with BranchCommits instances
    :return:
    """
    arc_hash_to_svn_rev = {r.current_hash: r.current_rev for r in svn_revs if r.current_hash}
    if arc_commit.arc_hashes is None:
        arc_commit.arc_hashes = []
    if arc_commit.action_type == rm_const.ActionType.MANUAL_COMMIT:
        arc_commit.arc_hashes = [arc_commit.current_hash]
    elif arc_commit.action_type == rm_const.ActionType.ROLLBACK:
        logging.debug("Got hashes to rollback %s", arc_commit.arc_hashes)
        svn_revs_to_rollback = []
        for arc_hash_to_rollback in arc_commit.arc_hashes:
            if arc_hash_to_rollback in arc_hash_to_svn_rev:
                svn_revs_to_rollback.append(arc_hash_to_svn_rev[arc_hash_to_rollback])
            else:
                svn_rev = _get_svn_revision_for_arc_commit_in_trunk(arc_client, arc_hash_to_rollback)
                if not svn_rev:
                    break
                svn_revs_to_rollback.append(svn_rev)
        arc_commit.svn_revs = svn_revs_to_rollback
    elif arc_commit.action_type == rm_const.ActionType.MERGE:
        svn_revs_to_merge = []
        for commit_hash in arc_commit.arc_hashes:
            svn_rev = _get_svn_revision_for_arc_commit_in_trunk(arc_client, commit_hash)
            if not svn_rev:
                break
            svn_revs_to_merge.append(svn_rev)
        arc_commit.svn_revs = svn_revs_to_merge


def prepare_arc_hashes(arc_client, revs, merge_path, c_info):
    """
    Find arc hashes corresponding to svn revisions

    :param arc_client: instance for working with Arcadia repository through Arc Vcs.
    :param revs: list with svn revisions in right order, i.e. for rollback in reverse order
    :param merge_path: MergePath entity
    :param c_info: Component info class instance
    :return: list with arc_hashes corresponding to revs
    """
    if isinstance(merge_path, rm_svn.RollbackReleaseMachinePath):
        branch_num = merge_path.branch_num
        svn_merged_revs = _get_merges_to_svn_branch(c_info, branch_num)
        svn_rev_to_arc_hash = {rev.current_rev: rev.current_hash for rev in svn_merged_revs}
        arc_hashes = []
        for rev in revs:
            if rev in svn_rev_to_arc_hash:
                logging.debug("Got arc hash %s for revision %s", svn_rev_to_arc_hash[rev], rev)
                arc_hashes.append(svn_rev_to_arc_hash[rev])
            else:
                logging.debug("Probably revision %s is in trunk", rev)
                trunk_hash = get_arc_hash_for_revision(arc_client, abs(rev))
                trunk_rev = _get_svn_revision_for_arc_commit_in_trunk(arc_client, trunk_hash)
                if trunk_rev == rev:
                    logging.debug("Got trunk arc hash %s", trunk_hash)
                    arc_hashes.append(trunk_hash)
                else:
                    logging.debug("Strange revision: %s, arc_hash %s", rev, trunk_hash)
        if len(arc_hashes) != len(revs):
            logging.debug("We got different revs and arc hashes quantity, %s and %s", len(arc_hashes), len(revs))
        return arc_hashes
    else:
        return [get_arc_hash_for_revision(arc_client, abs(rev)) for rev in revs]


def check_branch_existence(arc_client, arc_branch_path):
    """
    Check arc branch existence.

    :param arc_client: instance for working with Arcadia repository through Arc Vcs.
    :param arc_branch_path: path to branch
    :return: whether branch exists or not
    """
    with arc_client.init_bare() as arcadia_src_dir:
        return arc_client.fetch(
            arcadia_src_dir,
            branch=arc_branch_path,
        )


def commit_in_arc(
    task,
    arc_client,
    revs,
    descr,
    commit_info,
    merge_path,
    do_commit=True,
    c_info=None,
    commit_user="",
):
    """
    Perform commit in arc.

    :param task: Task which performs action
    :param arc_client: instance for working with Arcadia repository through Arc Vcs.
    :param revs: List of revisions to merge
    :param str descr: parameter from task
    :param commit_info: Instance of svn_helper.CommitInfo
    :param merge_path: MergePath entity
    :param boolean do_commit: True if need commit else False
    :param c_info: Component info class instance
    :param commit_user: User name, to be commit author
    :return: CommitResult object
    """
    diff_resolver_msg = commit.format_diff_resolver_msg(commit_info)

    if merge_path.action == rm_const.RollbackMode.TRUNK_MODE:
        logging.debug("We do not need to rollback commit in arc from trunk because it will sync with SVN")
        return rm_const.CommitResult(rm_const.CommitStatus.changeset_is_empty, None)

    arc_full_branch_path = c_info.svn_cfg__arc_branch_path(merge_path.branch_num)
    if not check_branch_existence(arc_client=arc_client, arc_branch_path=arc_full_branch_path):
        logging.debug("We don't have branch %s, nothing was committed, so commit result is empty", arc_full_branch_path)
        return rm_const.CommitResult(rm_const.CommitStatus.failed, None)
    with arc_client.mount_path(None, None, fetch_all=False, extra_params=["--vfs-version", "2"]) as arcadia_src_dir:
        arc_client.fetch(arcadia_src_dir, branch=arc_full_branch_path)
        arc_client.checkout(arcadia_src_dir, branch=arc_full_branch_path, track=True)
        arc_commit_hashes = prepare_arc_hashes(arc_client, revs, merge_path, c_info)
        for arc_hash in arc_commit_hashes:
            try:
                merge_path.arc_merge_func(arc_client=arc_client, arcadia_src_dir=arcadia_src_dir, arc_hash=arc_hash)
            except Exception as exc_cherry_pick:
                conflicts = process_cherry_pick_exception(exc_cherry_pick, task)
                paths_with_conflicts = {}
                for conflict_path, blob_id in conflicts.items():
                    paths_with_conflicts[conflict_path] = arc_client.show(arcadia_src_dir, blob_id)
                save_files_with_conflicts_from_arc(task, paths_with_conflicts)
                return rm_const.CommitResult(rm_const.CommitStatus.failed, None)
        arc_client.reset(arcadia_src_dir, mode=arc_api.ResetMode.SOFT, branch="HEAD~{}".format(len(arc_commit_hashes)))
        whole_commit_message = ""
        if do_commit:
            whole_commit_message = commit.create_commit_message(
                task=task,
                merge_path=merge_path,
                revs=arc_commit_hashes,
                diff_resolver_msg=diff_resolver_msg,
                descr=descr,
                commit_user=commit_user,
                need_check=False,
            )
            try:
                arc_client.commit(arcadia_src_dir, message=whole_commit_message)
            except Exception as exc_commit:
                return process_commit_exception(exc_commit, task, arc_full_branch_path)
            try:
                arc_client.push(arcadia_src_dir, upstream=arc_full_branch_path)
            except Exception as exc_push:
                eh.log_exception(
                    "Got exception while push: %s\nMaybe there was concurrent merge, try to pull changes", exc_push
                )
                arc_client.pull(arcadia_src_dir, rebase=True)
                arc_client.push(arcadia_src_dir)

        head_commit = arc_client.log(
            arcadia_src_dir,
            path=arc_full_branch_path,
            start_commit="HEAD",
            max_count=1,
            as_dict=True
        )[0]
        logging.debug("Got HEAD commit msg:\n %s", head_commit["message"])
        return prepare_commit_results(task, do_commit, whole_commit_message, head_commit)


def save_files_with_conflicts_from_arc(task, paths_with_conflicts):
    """
    Write diff from files with conflicts to resource
    :param task: MergeToStable or RollbackCommit
    :param paths_with_conflicts: dict with paths and full conflicts
    """
    if not task.Parameters.merge_conflicts.path.exists():
        task.Parameters.merge_conflicts.path.mkdir()
    for path, conflict in paths_with_conflicts.items():
        conflict_file_path = task.Parameters.merge_conflicts.path / path.replace('/', '__')
        fu.write_file(conflict_file_path, conflict)
    logging.debug("All conflict files saved")


def process_cherry_pick_exception(exc_cherry_pick, task):
    logging.debug("Got exception %s", exc_cherry_pick)
    conflicts_msg = "ERROR: there are some conflicts:\n"
    if conflicts_msg in str(exc_cherry_pick):
        conflict_part = str(exc_cherry_pick).split(conflicts_msg)[1]
        conflict_lines = conflict_part.split("\n")
        conflicts = {}
        for conflict_line in conflict_lines:
            if "wasn't performed." in conflict_line:
                break
            conflict = conflict_line.strip().split("  ")
            conflicts[conflict[1]] = conflict[2]
        task.set_info("Got conflicts: {}".format(conflicts))
        return conflicts
    else:
        raise exc_cherry_pick


def prepare_commit_results(task, do_commit, whole_commit_message, head_commit):
    if not do_commit:
        task.set_info("Dry run mode, do nothing...")
        return rm_const.CommitResult(rm_const.CommitStatus.success, None)
    if head_commit["message"].split("\n")[:2] == whole_commit_message.split("\n")[:2]:
        task.set_info("Committed hash {}".format(head_commit["commit"]))
        return rm_const.CommitResult(rm_const.CommitStatus.success, head_commit["commit"])
    else:
        return rm_const.CommitResult(rm_const.CommitStatus.failed, None)


def process_commit_exception(exc_commit, task, arc_full_branch_path):
    """
    Check whether commit result is empty changeset.

    :param exc_commit: exception caught while commit
    :param task: Task which performs action
    :param arc_full_branch_path: path to branch
    :return: CommitResult object with changeset_is_empty CommitStatus
    """
    logging.debug("Got exception while commit", exc_commit)
    empty_changeset_msg = "OUTPUT: On branch {}\nnothing to commit, working directory clean".format(
        arc_full_branch_path
    )
    if empty_changeset_msg in str(exc_commit):
        task.set_info("Nothing was merged (changeset is empty)")
        return rm_const.CommitResult(rm_const.CommitStatus.changeset_is_empty, None)
    else:
        raise exc_commit


def rollback_bad_svn_revisions(task, revs, c_info, branch_num):
    """
    Rollback revs from svn branch because they don't have corresponding arc commits.
    :param task: Task which performs action
    :param revs: List of revisions to rollback
    :param c_info: Component info class instance
    :param branch_num: branch number
    """
    task.set_info("Rollback revisions {revs} from branch {branch}".format(
        revs=", ".join(map(str, revs)),
        branch=c_info.full_branch_path(branch_num),
    ))
    merge_type = rm_svn.RollbackReleaseMachinePath(branch_num, c_info, getattr(task.ramdrive, "path", None))
    commit_info = rm_svn.SvnHelper.extract_commit_info(revs, merge_type.full)
    common_path = rm_svn.get_common_path(revs, merge_type.full)

    task_description = "Rollback bad commits\n__ignored__: {revs}".format(
        revs=", ".join("r{}".format(rev) for rev in revs)
    )
    commit_result = commit.perform_action(
        task,
        common_path=common_path,
        revs=revs,
        descr=task_description,
        commit_info=commit_info,
        merge_path=merge_type,
        do_commit=True,
        ignore_ancestry=getattr(task.Parameters, "force_merge", False),
        checkout_rev=None,
        c_info=c_info,
        commit_user=task.author,
    )
    if commit_result.status == rm_const.CommitStatus.failed:
        eh.check_failed("Could not rollback bad revisions, failing")


def _process_broken_svn_branch(task, arc_client, c_info, branch_num, revisions_to_rollback):
    """
    Rollback bad svn revisions from branch, return commit to sync.

    :param task: Task which performs action
    :param arc_client: instance for working with Arcadia repository through Arc Vcs.
    :param c_info: Component info class instance
    :param branch_num: branch number
    :param revisions_to_rollback: list with revisions from branch to rollback
    :return: BranchCommit instance to sync and list with svn BranchCommit instances
    """
    rollback_bad_svn_revisions(task, revisions_to_rollback, c_info, branch_num)
    svn_merged_revs = _get_merges_to_svn_branch(c_info, branch_num)
    return _get_commit_to_sync(task, arc_client, c_info, branch_num, svn_merged_revs)


def get_broken_svn_commits(task, svn_merged_revs, arc_merged_hashes):
    """
    Check whether any commits in svn branch which don't have prototype in arc branch.

    :param task: Task which performs action
    :param svn_merged_revs: list with BranchCommit instances for svn branch
    :param arc_merged_hashes: list with BranchCommit instances for arc branch
    :return: list with svn revisions to rollback if there are incompatible commits in arc and svn, [] else
    """
    logging.debug("Try to find broken svn commits")
    for current_index, svn_rev in enumerate(svn_merged_revs[1:], 1):
        if current_index >= len(arc_merged_hashes):
            logging.debug("All arc merged commits have images in svn branch")
            return []
        arc_commit = arc_merged_hashes[current_index]

        # In case c_info.svn_cfg__use_zipatch_only == True for any arc_commit.current_hash
        # svn_rev.action_type will be rm_const.ActionType.MANUAL_COMMIT.
        # For other components svn_rev.action_type and arc_commit.action_type should be equal.
        if (
            svn_rev.action_type != rm_const.ActionType.MANUAL_COMMIT
            and svn_rev.action_type != arc_commit.action_type
        ) or svn_rev.current_hash != arc_commit.current_hash:
            logging.debug(
                "SVN rev: %s, %s\nArc commit: %s, %s",
                svn_rev.action_type,
                svn_rev.current_hash,
                arc_commit.action_type,
                arc_commit.current_hash,
            )
            task.set_info(
                "Got incompatible BranchCommit instances from svn {svn_rev} and arc {arc_commit}".format(
                    svn_rev=svn_rev,
                    arc_commit=arc_commit,
                )
            )
            return [rev.current_rev for rev in svn_merged_revs[:current_index - 1:-1]]
    return []


def _get_commit_to_sync(task, arc_client, c_info, branch_num, svn_merged_revs):
    """
    Compare arc and svn branches in order to find commits to sync.

    :param task: Task which performs action
    :param arc_client: instance for working with Arcadia repository through Arc Vcs.
    :param c_info: Component info class instance
    :param branch_num: branch number
    :param svn_merged_revs: list with BranchCommit instances for svn branch
    :return: BranchCommit instance to sync and list with svn BranchCommit instances
    """
    arc_merged_hashes = _get_merges_to_arc_branches(arc_client, c_info, branch_num)
    logging.debug("svn_merged_revs %s, arc_merged_hashes %s", svn_merged_revs, arc_merged_hashes)

    broken_svn_commits = get_broken_svn_commits(task, svn_merged_revs, arc_merged_hashes)
    if not broken_svn_commits and len(svn_merged_revs) > len(arc_merged_hashes):
        task.set_info(
            "Got more commits in svn branch {len_svn_revs} than in arc branch {len_arc_commits}".format(
                len_svn_revs=len(svn_merged_revs),
                len_arc_commits=len(arc_merged_hashes),
            )
        )
        broken_svn_commits = [rev.current_rev for rev in svn_merged_revs[:len(arc_merged_hashes) - 1:-1]]

    if broken_svn_commits:
        if not task.Parameters.use_sync_conflict_resolver:
            eh.check_failed("Could not sync arc and svn branches")
        task.set_info(
            "Need to rollback svn revisions: {broken_svn_commits}".format(
                broken_svn_commits=", ".join(map(str, broken_svn_commits)),
            )
        )
        return _process_broken_svn_branch(task, arc_client, c_info, branch_num, broken_svn_commits)

    if len(svn_merged_revs) == len(arc_merged_hashes):
        logging.info("All commits are synced")
        return None
    arc_commit_to_sync = arc_merged_hashes[len(svn_merged_revs)]  # First commit in arc branch not equal to svn branch
    logging.debug("Got arc commit %s to sync", arc_commit_to_sync)
    return arc_commit_to_sync


def get_and_merge_zipatch(task, arc_client, c_info, branch_num, commit_to_sync, svn_branch_head):
    branch_path = os.path.join(
        c_info.svn_cfg__branches_folder,
        c_info.svn_cfg__branch_name,
        c_info.svn_cfg__branch_folder_name(branch_num),
        "arcadia",
    )
    patch_path = "{}.zipatch".format(svn_branch_head)
    zipatch_task_resource = sdk2.ResourceData(resource_types.ZIPATCH_OUTPUT(
        task=task,
        description="Zipatch for arc_hash {arc_hash} in branch {branch}".format(
            arc_hash=commit_to_sync.current_hash,
            branch=c_info.full_branch_path(branch_num),
        ),
        path=patch_path,
    ))
    with arc_client.mount_path(None, None, fetch_all=False, extra_params=["--vfs-version", "2"]) as arcadia_src_dir:
        arc_client.fetch(
            arcadia_src_dir,
            branch=c_info.svn_cfg__arc_branch_path(branch_num),
        )
        arc_client.create_zipatch(
            arcadia_src_dir,
            arc_hash=commit_to_sync.current_hash,
            output_path=str(zipatch_task_resource.path),
            svn_prefix=branch_path,
            no_copy=True,
            svn_revision=svn_branch_head,
            with_revprop=["arcanum:review-skip=yes"],
        )
        arc_commit_log = arc_client.log(
            arcadia_src_dir,
            start_commit=commit_to_sync.current_hash,
            max_count=1,
            as_dict=True,
        )
    zipatch_task_resource.ready()
    ya_commit_task = tasklet_executor.TaskletExecutor(
        None,
        priority=ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.HIGH)
    )
    zipatch_resource = sdk2.Resource.find(resource_types.ZIPATCH_OUTPUT, task_id=task.id).limit(1).first()
    logging.debug("Task Resources: %s", zipatch_resource)
    ya_commit_task.Parameters.resource_type = "SANDBOX_TASKS_BINARY"
    ya_commit_task.Parameters.resource_attrs = {"task_type": "TASKLET_YA_COMMIT_TASK", "release": "stable"}
    ya_commit_task.Parameters.owner = task.owner
    ya_commit_task.Parameters.yav_token_vault = "SEARCH-RELEASERS:yav_oauth_token"
    ya_commit_task.Parameters.ssh_user = rm_const.ROBOT_RELEASER_USER_NAME
    ya_commit_task.Parameters.tasklet_name = "YaCommitTask"
    # YaCommitTask interprets `{` and `}` as string formatting symbols, we should escape them
    # See https://st.yandex-team.ru/DEVTOOLSSUPPORT-9005#60bf9544305e1a7632f66e3b for details
    arc_commit_log[0]["message"] = arc_commit_log[0]["message"].replace("{", "{{").replace("}", "}}")
    ya_commit_task.Parameters.tasklet_input = {
        "ArcadiaUrl": "{}/arcadia@{}".format(c_info.full_branch_path(branch_num), svn_branch_head),
        "Patch": "zipatch:{}".format(zipatch_resource.http_proxy),
        "SshKey": {
            "uuid": "ver-01e9gnh7fhvtp7ajqs0b36k8w8",
            "key": "ssh_key"
        },
        "SshLogin": rm_const.ROBOT_RELEASER_USER_NAME,
        "Message": "Sync manual commit in arc branch {branch_path}.\n"
                   "{sync_msg}\nOriginal commit message:\n{arc_commit_msg}".format(
            branch_path=c_info.svn_cfg__arc_branch_path(branch_num),
            sync_msg=_get_sync_msg(commit_to_sync),
            arc_commit_msg=re.sub(rm_const.REVIEW_REPLACE_RE, 'REVIEW ID: \\2', arc_commit_log[0]["message"]),
        ),
        "Force": True,
    }
    ya_commit_task.save().enqueue()
    logging.debug("Run YaCommit task %s", ya_commit_task.id)
    start_time = int(time.time())
    while True:
        ya_commit_status = rm_utils.get_task_field(ya_commit_task.id, "status", task.server)
        logging.debug("Got status %s", ya_commit_status)
        if ya_commit_status in ctt.Status.Group.SUCCEED:
            logging.debug("YaCommit task finished")
            return
        if ya_commit_status in ctt.Status.Group.BREAK:
            logging.debug("Got status %s after 5th try, failing", ya_commit_status)
            eh.check_failed("Could not merge manual commit in svn branch")
        if int(time.time()) - start_time > 420:
            logging.debug("Got status %s after 420 seconds, failing", ya_commit_status)
            eh.check_failed("Could not merge manual commit in svn branch")
        time.sleep(20)


def _get_sync_msg(commit_to_sync):
    return "Sync arc commit: {arc_hash}\n".format(arc_hash=commit_to_sync.current_hash)


def sync_branch_from_arc(task, arc_client, c_info, branch_num):
    """
    Sync svn branch with arc branch by merging or rolling back revisions.

    :param task: Task which performs action
    :param arc_client: instance for working with Arcadia repository through Arc Vcs.
    :param c_info: Component info class instance
    :param branch_num: branch number
    :return: rm_const.CommitResult
    """

    if not check_branch_existence(arc_client=arc_client, arc_branch_path=c_info.svn_cfg__arc_branch_path(branch_num)):
        logging.debug("There is no branch %s, exit", c_info.svn_cfg__arc_branch_path(branch_num))
        raise arc_api.ArcCommandFailed("Could not find branch {}".format(c_info.svn_cfg__arc_branch_path(branch_num)))

    svn_merged_revs = _get_merges_to_svn_branch(c_info, branch_num)
    commit_to_sync = _get_commit_to_sync(task, arc_client, c_info, branch_num, svn_merged_revs)

    if commit_to_sync is None:
        task.set_info("[{}, branch {}] All commits were synced. Nothing to sync".format(c_info.name, branch_num))
        return rm_const.CommitResult(rm_const.CommitStatus.changeset_is_empty)

    commits_to_sync = deque([commit_to_sync])

    while len(commits_to_sync):

        logging.info("Commits to sync: %s", commits_to_sync)

        commit_to_sync = commits_to_sync.popleft()

        if not commit_to_sync:
            break

        _update_svn_revisions_for_arc_commit(arc_client, commit_to_sync, svn_merged_revs)

        if c_info.svn_cfg__use_zipatch_only:
            get_and_merge_zipatch(task, arc_client, c_info, branch_num, commit_to_sync, svn_merged_revs[-1].current_rev)
        else:
            if commit_to_sync.action_type == rm_const.ActionType.MERGE:
                merge_type = rm_svn.MergeReleaseMachinePath(branch_num, c_info, getattr(task.ramdrive, "path", None))
                commit_info = rm_svn.SvnHelper.extract_commit_info(commit_to_sync.svn_revs, rm_svn.TRUNK_PATH)
                common_path = rm_svn.get_common_path(
                    commit_to_sync.svn_revs,
                    sdk2.svn.Arcadia.ARCADIA_BASE_URL,
                    merge_from_trunk=True,
                )

            elif commit_to_sync.action_type == rm_const.ActionType.ROLLBACK:
                merge_type = rm_svn.RollbackReleaseMachinePath(branch_num, c_info, getattr(task.ramdrive, "path", None))
                commit_info = rm_svn.SvnHelper.extract_commit_info(commit_to_sync.svn_revs, merge_type.full)
                common_path = rm_svn.get_common_path(commit_to_sync.svn_revs, merge_type.full)
            elif commit_to_sync.action_type == rm_const.ActionType.MANUAL_COMMIT:
                # Use YaCommit
                get_and_merge_zipatch(
                    task,
                    arc_client,
                    c_info,
                    branch_num,
                    commit_to_sync,
                    svn_merged_revs[-1].current_rev,
                )
            else:
                raise TypeError(
                    "Unexpected action type {} ({})".format(
                        commit_to_sync.action_type,
                        type(commit_to_sync.action_type),
                    ),
                )

            if commit_to_sync.action_type != rm_const.ActionType.MANUAL_COMMIT:
                task.set_info(
                    "Got merge_type {merge_type} for revisions {rev_list}".format(
                        merge_type=merge_type,
                        rev_list=commit_to_sync.svn_revs,
                    ),
                )
                task_description = _get_sync_msg(commit_to_sync)
                commit.perform_action(
                    task,
                    common_path=common_path,
                    revs=commit_to_sync.svn_revs,
                    descr=task_description,
                    commit_info=commit_info,
                    merge_path=merge_type,
                    do_commit=True,
                    ignore_ancestry=getattr(task.Parameters, "force_merge", False),
                    checkout_rev=None,
                    c_info=c_info,
                    commit_user=task.author,
                )

        svn_merged_revs = _get_merges_to_svn_branch(c_info, branch_num)
        new_commit_to_sync = _get_commit_to_sync(task, arc_client, c_info, branch_num, svn_merged_revs)

        if new_commit_to_sync is not None and new_commit_to_sync.current_hash == commit_to_sync.current_hash:
            task.set_info(
                "[{component_name}, branch {branch_num}] <strong>FAILURE</strong>: "
                "Got same commit {commit} to sync, probably previous sync failed".format(
                    component_name=c_info.name,
                    branch_num=branch_num,
                    commit=commit_to_sync.current_hash,
                ),
                do_escape=False,
            )
            return rm_const.CommitResult(rm_const.CommitStatus.failed, revision=new_commit_to_sync.current_hash)

        if new_commit_to_sync is not None:
            commits_to_sync.append(new_commit_to_sync)

    return rm_const.CommitResult(rm_const.CommitStatus.success)


def make_new_arc_tag_from_branch(
    arc_client, c_info, branch_num, tag_num,
    check_with_svn_rev=None,
    tags_start_from=1,  # 1 - legacy RM, 0 for compatibility with new CI
    branch_commit_hash=None,
):
    arc_full_branch_path = c_info.svn_cfg__arc_branch_path(branch_num)
    arc_full_tag_path = c_info.arc_full_tag_path(tag_num, branch_num)
    logging.info("Starting to create new arc tag %s from branch %s", arc_full_tag_path, arc_full_branch_path)

    result_context = {
        NEW_ARC_TAG_CONTEXT__ARC_HASH: branch_commit_hash,
        NEW_ARC_TAG_CONTEXT__BRANCH: arc_full_branch_path,
        NEW_ARC_TAG_CONTEXT__TAG: arc_full_tag_path,
    }

    with arc_client.mount_path(None, None, fetch_all=False, extra_params=["--vfs-version", "2"]) as arcadia_src_dir:
        logging.info("Starting to fetch and checkout arc branch %s", arc_full_branch_path)
        arc_client.fetch(arcadia_src_dir, branch=arc_full_branch_path)
        arc_client.checkout(arcadia_src_dir, branch=arc_full_branch_path)

        if not branch_commit_hash:

            logging.info("Trying to guess branch commit hash since it is not provided in method args")

            branch_log = get_arc_branch_log(arc_client, arcadia_src_dir, arc_full_branch_path)
            branch_commit = branch_log[tag_num - tags_start_from]
            branch_commit_hash = branch_commit["commit"]
            result_context[NEW_ARC_TAG_CONTEXT__ARC_HASH] = branch_commit_hash

        else:

            logging.info("Got predefined branch commit hash: %s", branch_commit_hash)

        if check_with_svn_rev is not None and tag_num > tags_start_from:
            full_svn_branch_path = c_info.svn_cfg__branch_path(branch_num)
            branch_path_with_rev = "{}@{}".format(full_svn_branch_path, check_with_svn_rev)
            rev_log = svn.Arcadia.log(branch_path_with_rev, limit=1)[0]
            arc_hash = get_synced_arc_commit(rev_log)
            if arc_hash != branch_commit_hash:
                return rm_core.Error("Arc hash from svn commit msg {} and commit hash from arc {} are not equal".format(
                    arc_hash, branch_commit_hash
                ))

        arc_client.fetch(arcadia_src_dir, branch=arc_full_tag_path)
        tag_info_all = string.all_to_unicode(arc_client.list_tags(mount_point=arcadia_src_dir))
        logging.debug("Got tags by tag path: %s", tag_info_all)
        if arc_full_tag_path in tag_info_all:
            tag_info_specific = string.all_to_unicode(
                arc_client.list_tags(mount_point=arcadia_src_dir, points_at=branch_commit_hash)
            )
            logging.debug("Got tags with specific hash: %s", tag_info_specific)
            if arc_full_tag_path in tag_info_specific:
                return rm_core.Ok(
                    "Arc tag {} already exists at {}".format(arc_full_tag_path, branch_commit_hash),
                    context=result_context,
                )
            else:
                existing_tag_log = arc_client.log(
                    arcadia_src_dir,
                    path=arc_full_tag_path,
                    start_commit="HEAD",
                    max_count=1,
                    as_dict=True
                )[0]
                return rm_core.Error("Arc tag {} already exists. Tag log: {}. Requested hash: {}".format(
                    arc_full_tag_path, existing_tag_log, branch_commit_hash
                ))
        logging.info("Pushing new tag %s to arc with hash %s", arc_full_tag_path, branch_commit_hash)
        arc_client.push(
            mount_point=arcadia_src_dir,
            refspecs=[(branch_commit_hash, arc_full_tag_path)],
        )
        tag_info_specific = arc_client.list_tags(mount_point=arcadia_src_dir, points_at=branch_commit_hash)
        logging.debug("Tag info: %s", tag_info_specific)
        return rm_core.Ok(
            "New arc tag was created: {}. Hash: {}".format(arc_full_tag_path, branch_commit_hash),
            context=result_context,
        )
