import argparse
import os
import re
import subprocess

from collections import namedtuple


os.environ['COLUMNS'] = '180'

RELEASE_TYPE_KEYS = ('stable', 'testing', 'unstable', 'prestable')
ReleaseTypes = namedtuple('ReleaseType', RELEASE_TYPE_KEYS)
RT = ReleaseTypes(stable='stable', testing='testing', unstable='unstable', prestable='prestable')
RT_DEPRECATED = ReleaseTypes(stable='stable', testing='test', unstable='unstable', prestable='prestable')
RELEASE_ATTR = 'released'
RELEASE_ATTR_DEPRECATED = 'release'

ARCADIA_ROOT_ENV_KEY = 'ARCADIA_ROOT'
ARCADIA_ROOT = os.environ.get(ARCADIA_ROOT_ENV_KEY) or '.'
DEFAULT_TASK_ROOT = 'sandbox/projects/release_machine/tasks'
BIN_DIR = 'bin'
YA = 'ya'
TASK_BIN_NAME_REGEXP = re.compile(r'SANDBOX(?:_PY[2]?3)?_TASK\(([^)]*)\)')


def get_ya(arcadia_root):
    return os.path.join(arcadia_root, YA)


def get_task_dir(arcadia_root, task_type):
    task_dir = ''.join(map(lambda s: s.lower().capitalize(), task_type.split('_')))
    print("Task dir detected: {}".format(task_dir))
    return os.path.join(arcadia_root, DEFAULT_TASK_ROOT, task_dir)


def detect_task_bin_dir(task_dir):
    task_bin_path = os.path.join(task_dir, BIN_DIR)
    if not os.path.isdir(task_bin_path):
        task_bin_path = task_dir
    print("Task binary path detected: {}".format(task_bin_path))
    return task_bin_path


def check_task_dir(task_dir, task_bin_path):
    print('Checking {} (task source path)'.format(task_dir))
    if not os.path.isdir(task_dir):
        print('... not a directory')
        exit(1)
    print('Checking {} (task binary path)'.format(task_bin_path))
    if not os.path.exists(os.path.join(task_bin_path, 'ya.make')):
        print('... no ya.make found')
        exit(1)
    print('... OK')


def get_task_bin_name(task_bin_path):
    task_bin_ya_make = os.path.join(task_bin_path, 'ya.make')
    with open(task_bin_ya_make, 'r') as yamake:
        for line in yamake:
            match = TASK_BIN_NAME_REGEXP.search(line)
            if not match:
                continue
            name = match.group(1).strip()
            if not name:
                continue
            return name
    return 'bin'


def _run_command(command, runner=subprocess.check_call):
    print(command)
    result = runner(command, shell=True)
    print('Done!')
    return result


def run_build(arcadia_root, task_bin_path):
    print("Building target {} ...".format(task_bin_path))
    ya = get_ya(arcadia_root)
    # use release build to obtain stripped binaries for binary tasks (and fast Python)
    # see RMDEV-3137 and DEVTOOLSSUPPORT-17050 for reference
    command = "{ya} make --build=release {path}".format(ya=ya, path=task_bin_path)
    _run_command(command)


def run_list_types(task_bin_path, task_bin_name, task_type):
    print("Checking list-types ...")
    output = _run_command(
        '{} content --list-types'.format(os.path.join(task_bin_path, task_bin_name)),
        runner=subprocess.check_output
    )
    print("Checking current task type ({}) in list ...".format(task_type))
    assert task_type in output, "{} not found among task binary task types: {}".format(task_type, output)
    print('Done!')


def run_upload(task_bin_path, task_bin_name, task_type, release_attr, release_type, force=True):
    print('Uploading...')
    command = (
        '{task_bin} upload '
        '--attr task_type="{task_type}" '
        '--attr {release_attr}="{release_type}"'.format(
            task_bin=os.path.join(task_bin_path, task_bin_name),
            task_type=task_type,
            release_attr=release_attr,
            release_type=release_type,
        )
    )

    if force:
        command += ' --force'
    _run_command(command)


def main():
    parser = argparse.ArgumentParser(
        formatter_class=argparse.RawDescriptionHelpFormatter,
        description='\n'.join((
            "Builds and uploads Sandbox binary task.",
            "",
            "By default tasks are searched among {} subdirectories.".format(DEFAULT_TASK_ROOT,),
            "Note that you should either have the {} environment key set or you'll need to use the '-a' option.".format(
                ARCADIA_ROOT_ENV_KEY,
            ),
            "",
            "If a task has a 'bin' subdirectory then it will be used as a build target. "
            "Otherwise the script will use task root path.",
        )),
        epilog='\n'.join((
            "Detected arcadia root: {}".format(ARCADIA_ROOT),
        )),
    )

    parser.add_argument(
        '-a',
        '--arcadia-root',
        help=(
            'Path to Arcadia root. '
            'There is no need to specify it here if you have {} environment variable set correctly.'.format(
                ARCADIA_ROOT_ENV_KEY,
            )
        ),
        default=ARCADIA_ROOT,
    )
    parser.add_argument(
        '-d',
        '--deprecated',
        action='store_true',
        help='Use deprecated version of LastBinaryTaskRelease',
    )
    parser.add_argument(
        '-t',
        '--task-dir',
        type=str,
        required=False,
        help='Path to the task directory. Unless provided the task will be searched for in {}'.format(
            DEFAULT_TASK_ROOT,
        ),
    )
    parser.add_argument('task_type', metavar='TASK_TYPE', type=str, nargs=1, help='Task type (SCREAMING_SNAKE_CASE)')
    parser.add_argument(
        'release_type',
        metavar='RELEASE_TYPE',
        type=str,
        nargs=1,
        choices=RELEASE_TYPE_KEYS,
        default=RT.testing,
        help='Release type: {}'.format(' | '.join(RELEASE_TYPE_KEYS)),
    )

    args = parser.parse_args()

    task_type = args.task_type[0]
    release_type_key = args.release_type[0]
    task_dir = args.task_dir or get_task_dir(args.arcadia_root, task_type)
    task_bin_path = detect_task_bin_dir(task_dir)
    check_task_dir(task_dir, task_bin_path)
    task_bin_name = get_task_bin_name(task_bin_path)

    if args.deprecated:
        release_types = RT_DEPRECATED
        release_attr = RELEASE_ATTR_DEPRECATED
    else:
        release_types = RT
        release_attr = RELEASE_ATTR

    run_build(args.arcadia_root, task_bin_path)
    run_list_types(task_bin_path, task_bin_name, task_type)
    run_upload(task_bin_path, task_bin_name, task_type, release_attr, getattr(release_types, release_type_key))


if __name__ == '__main__':
    try:
        main()
    except AssertionError as ae:
        print("!!!!!")
        print("ERROR: {}".format(ae))
        exit(1)
