# -*- coding: utf-8 -*-
import json
import logging

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.common import requests_wrapper
from sandbox.projects.common import decorators


class RMClient(object):
    DELETE_COMPONENT = "api/release_engine.services.Model/deleteComponent"
    POST_NOTIFICATION_DECLARATIVE_CONFIG = (
        "api/release_engine.services.Notification/postNotificationDeclarativeConfig"
    )
    POST_PROTO_EVENTS = "api/release_engine.services.Event/postProtoEvents"
    POST_PRE_RELEASE = "api/release_engine.services.Testenver/postPreRelease"
    POST_RELEASE = "api/release_engine.services.Testenver/postRelease"
    POST_STATE = "api/release_engine.services.Model/postState"
    GET_COMPONENT = "api/release_engine.services.Model/getComponent"
    GET_COMPONENTS = "api/release_engine.services.Model/getComponents"
    GET_SCOPES = "api/release_engine.services.Model/getScopes"
    GET_EVENTS = "api/release_engine.services.Model/getEvents"
    GET_STATE = "api/release_engine.services.Model/getState"
    GET_OK_TO_RELEASE = "api/release_engine.services.Model/okToRelease"
    ADD_COMPONENT_CHAT = "api/release_engine.services.Model/addComponentChat"
    GET_COMPONENT_CHATS = "api/release_engine.services.Model/getComponentChats"

    def __init__(
        self,
        host=rm_const.Urls.RM_URL,
        specific_logging_level=None
    ):
        requests_wrapper.disable_insecure_warnings()
        self.logger = logging.getLogger(__name__)
        if specific_logging_level:
            self.logger.setLevel(specific_logging_level)
            decorators.logger.setLevel(specific_logging_level)
        self.host = host
        self.status_code = None

    @decorators.retries(max_tries=5, delay=1, default_instead_of_raise=True)
    def _do_post(self, url, data=None, json_data=None, return_json=False, add_headers=None):
        headers = {"Ignore-Defaults": "yes"}
        if add_headers:
            headers.update(add_headers)
        r = requests_wrapper.post(
            url=url,
            data=data,
            json=json_data,
            headers=headers,
        )
        self.logger.debug("Request: %s, Response status is %s", r.url, r.status_code)
        self.logger.debug("Request data: %s", data)
        self.logger.debug("Request json: %s", json.dumps(json_data))
        self.status_code = r.status_code
        if r.status_code >= 400:
            self.logger.error('[post] HTTP error %s, response %s', r.status_code, r.text)
            self.logger.error("Got response headers %s, response body %s", r.headers, r.content)
        if r.status_code in [403, 404, 501]:  # access denied, not found, not implemented
            return None if return_json else r  # no need to retry
        requests_wrapper.check_status_code(r)  # retry other bad codes
        self.logger.debug("ReqID: %s", r.headers.get("x-req-id", "unknown"))
        self.logger.debug("Response is: %s", r.text)
        if return_json:
            return r.json()
        return r

    def delete_component(self, component_name, irreversible=False):
        component = {
            "component_name": component_name,
            "irreversible": irreversible,
        }
        return self._do_post(self.host + self.DELETE_COMPONENT, json_data=component, return_json=True)

    def post_proto_events(self, event_messages):

        from google.protobuf import json_format

        json_data = [
            json_format.MessageToDict(message, including_default_value_fields=True)
            for message in event_messages
        ]

        return self.post_proto_events_as_json(json_data)

    def post_proto_events_as_json(self, event_json_data):
        logging.debug("post_proto_events: Going to send the following data: %s", event_json_data)

        return self._do_post(
            self.host + self.POST_PROTO_EVENTS,
            json_data={"events": event_json_data},
        )

    def add_component_chat(self, component_name, chat_name, chat_id, transport_type):
        return self._do_post(
            url=self.host + self.ADD_COMPONENT_CHAT,
            json_data={
                "component_name": component_name,
                "name": chat_name,
                "chat_id": str(chat_id),
                "transport_type": transport_type,
            },
            return_json=True,
        )

    def get_component_chats(self, component_name):
        return self._do_post(
            self.host + self.GET_COMPONENT_CHATS,
            json_data={
                "component_name": component_name,
            },
            return_json=True,
        )

    def post_pre_release(self, revision, testenv_db, component_name, scope_number, add_headers=None):
        return self._do_post(
            url=self.host + self.POST_PRE_RELEASE,
            json_data={
                "revision": revision,
                "testenv_db": testenv_db,
                "component_name": component_name,
                "scope_number": scope_number,
            },
            return_json=True,
            add_headers=add_headers,
        )

    def post_release(
        self,
        revision, testenv_db, component_name,
        release_type, release_item, release_notes,
        scope_number, tag_number,
        add_headers=None,
    ):
        return self._do_post(
            url=self.host + self.POST_RELEASE,
            json_data={
                "revision": revision,
                "testenv_db": testenv_db,
                "component_name": component_name,
                "release_type": release_type,
                "release_item": release_item,
                "release_notes": release_notes,
                "release_handler": "release_action",
                "scope_number": scope_number,
                "tag_number": tag_number,
            },
            return_json=True,
            add_headers=add_headers,
        )

    def post_notification_declarative_config(self, notifications):

        return self._do_post(
            url=self.host + self.POST_NOTIFICATION_DECLARATIVE_CONFIG,
            json_data={
                "notifications": notifications,
            },
            return_json=True,
        )

    def get_component(self, component_name, return_json=True):
        component = {"component_name": component_name}
        response = self._do_post(self.host + self.GET_COMPONENT, json_data=component, return_json=return_json)
        return response

    def get_components(self):
        return self._do_post(self.host + self.GET_COMPONENTS, return_json=True)

    def get_events(self, component_name, event_types, from_ts="", till_ts="", length=10):
        return self._do_post(
            self.host + self.GET_EVENTS,
            json_data={
                "component_name": component_name,
                "length": length,
                "event_types": event_types,
                "from_ts": from_ts,
                "till_ts": till_ts,
            },
            return_json=True,
        )

    def get_scopes(self, component_name, limit=10, start_scope_number=None, return_json=True):
        json_data = {"component_name": component_name, "limit": limit}
        if start_scope_number:
            json_data["start_scope_number"] = start_scope_number
        return self._do_post(
            self.host + self.GET_SCOPES,
            json_data=json_data,
            return_json=return_json
        )

    def get_state(self, component_name, key="/"):
        response = self._do_post(
            self.host + self.GET_STATE,
            json_data={"component_name": component_name, "key": key},
            return_json=True,
            add_headers={"x-horadric-dump-defaults": "1"}
        )
        if response is None:
            return
        return response.get('entries', [])

    def get_current_version(self, component_name, deploy_level=rm_const.ReleaseStatus.stable, resource_name=None):
        """
        Get current version of component on specified deploy_level.

        :param component_name: Name of RM component
        :param deploy_level: One of ReleaseStatus attributes
        :param resource_name: [optional] Specific resource to get version for
        :return: Info about current production version.
            Empty dict if no version found.
            Ex: {} or {"scope_number": "11", "tag_number": "2", "deploy_time": 1571057633, "build_task_id": "123456"}
        """
        curr_versions = self.get_state(component_name, "/current_version$")
        if curr_versions is None:
            return {}
        versions = {i["key"]: i["value"] for i in curr_versions}
        keys = []
        if resource_name:  # first, try to find version of specific resource
            keys.append("/current_version${}${}".format(resource_name, deploy_level))
        keys.append("/current_version${}".format(deploy_level))
        for k in keys:
            curr_version = versions.get(k)
            if curr_version:
                self.logger.debug("[%s] Got version by key '%s': %s", component_name, k, curr_version)
                return json.loads(curr_version)
            else:
                self.logger.debug("[%s] Not found version by key '%s': %s", component_name, k, curr_version)
        self.logger.warning("[%s] No version found. Probably, component is new", component_name)
        return {}

    def post_state(self, component_name, key, value):
        data = {
            "component_name": component_name,
            "key": key,
            "value": value,
        }
        return self._do_post(self.host + self.POST_STATE, json_data=data, return_json=True)

    def set_branch_scope_description(self, component_name, scope_number, description):
        return self.post_state(component_name, "/br{}/$description".format(scope_number), description)

    def get_branch_scope_description(self, component_name, scope_number):
        description = self.get_state(component_name, "/br{}/$description".format(scope_number))
        if not description:
            return
        return description[0]['value']

    def get_ok_to_release(self, component_name, scope_number, tag_number):
        data = {
            "component_name": component_name,
            "scope_number": scope_number,
            "tag_number": tag_number,
        }
        return self._do_post(
            self.host + self.GET_OK_TO_RELEASE, json_data=data, return_json=True,
            add_headers={"x-horadric-dump-defaults": "1"}
        )
