from __future__ import unicode_literals

import functools
import six
from six.moves import reduce

from abc import ABCMeta
from abc import abstractproperty


class ComponentInfoAttributeError(Exception):
    """Custom exception for c_info inner attribute errors (RMDEV-1196)."""

    pass


def attribute_error_wrapper(fn):

    @functools.wraps(fn)
    def wrapped(*args, **kwargs):
        try:
            return fn(*args, **kwargs)
        except AttributeError as ae:
            raise ComponentInfoAttributeError("ComponentInfoAttributeError: {}".format(ae.message))

    return wrapped


def wrap_cls_attribute_errors(cls):
    for attr_name in cls.__dict__:
        attr = getattr(cls, attr_name)
        if isinstance(attr, property):
            setattr(
                cls,
                attr_name,
                property(
                    fget=attribute_error_wrapper(attr.fget),
                    fset=attr.fset,
                    fdel=attr.fdel,
                    doc=attr.__doc__,
                ),
            )
    return cls


@six.add_metaclass(ABCMeta)
class ComponentInfoCfgBase(object):
    @abstractproperty
    def _cfg_cls(self):
        pass

    def __init__(self):
        super(ComponentInfoCfgBase, self).__init__()
        self._cfg = self._cfg_cls()

    def __getattr__(self, item):
        return reduce(getattr, [self._cfg] + item.split('__'))


class ComponentInfoMeta(ABCMeta):

    def __new__(mcs, name, bases, cls_dict):
        name = str(name)
        cfg_cls = cls_dict['_cfg_cls']
        cls_dict['name'] = property(lambda obj: cfg_cls.name)
        cls_dict['display_name'] = cfg_cls.display_name
        for base in bases:
            cls_dict.update(base.__dict__)
        new_cls = ABCMeta.__new__(mcs, name, (ComponentInfoCfgBase, ) + tuple(bases), cls_dict)
        new_cls.__doc__ = "{name} is a component info class for {component_name}".format(
            name=name,
            component_name=cfg_cls.name,
        )
        return wrap_cls_attribute_errors(new_cls)
