# -*- coding: utf-8 -*-
"""
**Release-machine**.

Here are base classes for components released using release-machine.
Maintainer: mvel@, ilyaturuntaev@
"""
from __future__ import unicode_literals

import datetime
import json
import logging
import os
import re
import six
import typing
from abc import ABCMeta
from abc import abstractmethod
from abc import abstractproperty

import sandbox.common.types.task as ctt
import sandbox.projects.common.sdk_compat.task_helper as rm_th
import sandbox.projects.abc.client as abc_client
import sandbox.projects.release_machine.helpers.deploy as rm_deploy
import sandbox.projects.release_machine.helpers.responsibility_helper as rm_responsibility
import sandbox.projects.release_machine.helpers.staff_helper as rm_staff_helper
import sandbox.projects.release_machine.helpers.svn_helper as rm_svn
import sandbox.projects.release_machine.changelogs as rm_ch
import sandbox.projects.release_machine.security as rm_sec
import sandbox.sdk2 as sdk2
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import resource_selectors
from sandbox.projects.common import time_utils as tu
from sandbox.projects.common.testenv_client import TEClient
from sandbox.projects.release_machine import client
from sandbox.projects.release_machine import core as rm_core
from sandbox.projects.release_machine import rm_utils
from sandbox.projects.release_machine.components.components_info import base as c_info_base
from sandbox.projects.release_machine.components.components_info import mixin as mixin
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.sdk2.vcs import svn


@six.add_metaclass(ABCMeta)
class ComponentInfoGeneral(object):
    """
    Base class for release machine components.

    Here should be methods that are common for ALL types of release components.
    """

    _svn_cfg__main_url = None

    def __call__(self):
        """Just in case if somebody tries to call instance, not instantiate class."""
        return self

    def abc_service_name(self, rm_token=None):
        # type: (typing.Optional[str]) -> typing.Optional[str]
        return rm_responsibility.get_responsible_abc_service_name(self.responsible, rm_token)

    def stable_release_notes(self, **kwargs):
        message = []
        st_issue = kwargs.get("st_issue", None)
        item = kwargs.get("item", None)
        add_info = kwargs.get("add_info", None)

        if st_issue:
            message.append("Priemka: {}".format(lb.st_link(st_issue.key, plain=True)))
        if item:
            message.append("Release item: {}".format(item.name))
        if add_info:
            message.append(add_info)
        responsible_for_release = self.get_responsible_for_release()
        if responsible_for_release:
            message.append("Responsible for release: {}".format(responsible_for_release))

        return "\n".join(message)

    def get_stable_release_label(self, release_item, release_num):
        return "{} ({}): {}".format(self.name, release_item.name, release_num)

    def testing_release_notes(self, item, release_num=None, release_stage=None):
        if release_stage:
            message = ["Send '{}' to '{}'".format(item.name, release_stage)]
        else:
            message = ["Send '{}'".format(item.name)]
        if getattr(self, "changelog_cfg__wiki_page", None):
            message.append(
                "Changelog: {}{}".format(rm_const.Urls.WIKI, self.changelog_major_url(release_num))
            )

        return "\n".join(message)

    def get_last_release(self, stage=None):
        # type: (ctt.ReleaseStatus) -> typing.Generator[rm_core.ReleasedResource]
        stage = stage or self.releases_cfg__default_stage
        for res_info in self.releases_cfg__resources_info:
            resource_id, release_time = resource_selectors.by_last_released_task(
                resource_type=res_info.resource_type,
                attrs=res_info.attributes,
                stage=stage
            )
            if not resource_id:
                logging.debug("[%s] Last released resource for '%s' not found", self.name, res_info)
                continue
            logging.info(
                "[%s] Last released resource for '%s' found: %s",
                self.name, res_info.resource_type, resource_id
            )
            resource = sdk2.Resource[resource_id]
            major_num, minor_num = self.get_release_numbers_from_resource(resource, res_info)

            release_time_ts = tu.to_unixtime(release_time)

            yield rm_core.ReleasedResource(
                id=resource_id,
                build_task_id=resource.task_id,
                major_release=major_num,
                minor_release=minor_num,
                timestamp=release_time_ts,
                component=self.name,
                status=stage,
                owner=resource.owner,
                resource_name=res_info.resource_name,
            )

    def get_release_numbers(self, release_item):
        """
        Get release numbers from all possible ways.

        :return major_num, minor_num
        """
        release_nums = self.get_release_numbers_from_attrs(release_item.resource)
        if not release_nums[0]:
            build_arc_url = rm_utils.get_input_or_ctx_field(release_item.build_task_id, release_item.build_ctx_key)
            release_nums = self.svn_cfg__get_release_numbers(build_arc_url)
        if release_nums is None:
            logging.error("Unable to get release numbers from url: %s. Return None, None", build_arc_url)
            return None, None
        release_nums = tuple(str(i) if i is not None else i for i in release_nums)
        logging.debug("Got release numbers: %s", release_nums)
        return release_nums

    def get_release_numbers_from_resource(self, resource, item_data):
        # type: (sdk2.Resource, typing.Union[SandboxResourceData, ReleasedResourceInfo]) -> typing.Tuple[typing.Optional[str], typing.Optional[str]]
        """
        Get release numbers from resource.

        :return major_num, minor_num
        """
        release_nums = self.get_release_numbers_from_attrs(resource)

        if not release_nums[0]:

            logging.info("Trying to get release numbers from arc url")

            build_arc_url = rm_utils.get_input_or_ctx_field(resource.task_id, item_data.build_ctx_key)
            logging.info("Build arc url %s", build_arc_url)

            release_nums = self.svn_cfg__get_release_numbers(build_arc_url)

            if release_nums is None:
                logging.error("Unable to get release numbers from url: %s. Return None, None", build_arc_url)
                return None, None

        release_nums = tuple(str(i) if i is not None else i for i in release_nums)
        logging.info("Got release numbers: %s", release_nums)

        return release_nums

    @staticmethod
    def get_release_numbers_from_attrs(resource):
        # type: (sdk2.Resource) -> typing.Tuple[typing.Optional[typing.Union[str, int]], typing.Optional[typing.Union[str, int]]]
        return None, None

    def kpi_alert_allowed(self):
        """
        Check if KPI alerts are allowed to send.

        :return True when alerts sending is allowed by component settings
        """
        if not int(self.release_viewer_cfg__kpi_alert):
            # zero `release_viewer_cfg__kpi_alert` value is no alerting at all
            return False

        if self.release_viewer_cfg__kpi_alert_skip_weekends:
            # RMDEV-465
            today = datetime.datetime.today()
            if today.weekday() >= 5:
                # skip Saturday and Sunday
                return False

            if today.month == 1 and today.day <= 7:
                # skip 1-7th of January
                return False

        return True

    @staticmethod
    def get_deploy_services(deploy_info, nanny_client):

        logging.debug("Getting all deploy services for %s", deploy_info)

        all_services = deploy_info.services

        for dashboard in deploy_info.dashboards:
            all_services.extend(nanny_client.get_dashboard_services(dashboard))

        all_services = set(all_services)

        logging.debug("Got the following set of services for the given deploy info: %s", all_services)

        return all_services

    def get_last_deploy(self, token=None, only_level=None):
        """
        Get list of last deployed resources for all deploy watchers.

        :return: List[rm_core.DeployedResource]
        """
        deploy_data = []
        try:
            deploy_watcher = rm_deploy.DEPLOY_WATCHER.get(self.releases_cfg__deploy_system.name)
            logging.info("Deploy watcher found: %s", deploy_watcher)
            if deploy_watcher:
                deploy_data = deploy_watcher(self, token).last_deploy(only_level)
        except Exception as e:
            logging.warning("Unable to get deploy_data\n%s", e, exc_info=True)
        logging.info("Got deploy data: %s", deploy_data)
        return deploy_data

    def resolved_releasable_items_iter(self):
        # type: () -> typing.Generator[ri.ReleasableItem]
        """Iterate over all releasable items, including dynamically created."""
        for item in self.releases_cfg__releasable_items:
            if isinstance(item, ri.ReleasableItem):
                yield item
            elif isinstance(item, ri.DynamicReleasableItem):
                # todo: construct ReleasableItems from DynamicReleasableItems somehow
                continue
            else:
                eh.check_failed("Unknown releasable item type: {}".format(item))

    def get_last_deploy_proto(self, token=None, stage=None):
        # type: (str, typing.Optional[str]) -> typing.Optional[release_and_deploy_pb2.RmComponentVersions]

        deploy_watchers = self._get_deploy_watchers(token, stage)

        if not deploy_watchers:
            logging.info("No deploy watchers found for %s", self.name)
            return

        from release_machine.common_proto import release_and_deploy_pb2

        item_versions = []

        for item in self.resolved_releasable_items_iter():
            logging.info("Processing releasable item %s", item)

            if not item.deploy_infos:
                logging.info("Deploy info not found, continue")
                continue

            deployed_versions = []

            for deploy_info in item.iter_deploy_infos(stage):
                logging.info("Processing deploy info: %s", deploy_info)
                deploy_watcher = deploy_watchers[deploy_info.deploy_system]
                if deploy_watcher:
                    last_deployed_versions = deploy_watcher.last_deploy_proto(item.data, deploy_info)
                    if last_deployed_versions:
                        deployed_versions.extend(last_deployed_versions)
                    else:
                        logging.info("Last deployed version not found, skip")
                else:
                    logging.info("No deploy watcher for '%s' found", deploy_info.deploy_system.name)

            item_versions.append(
                release_and_deploy_pb2.ReleaseItemVersion(
                    release_item_name=item.name,
                    deployed_versions=deployed_versions,
                )
            )

        return release_and_deploy_pb2.RmComponentVersions(release_item_versions=item_versions)

    def _get_deploy_watchers(self, token, stage=None):
        """
        Get all deploy watcher's instances for component's releasable_items.
        Used for caching requests to deploy systems for different releasable items deployed to same service.
        """
        result = {}
        for item in self.resolved_releasable_items_iter():
            for deploy_info in item.iter_deploy_infos(stage):
                if deploy_info.deploy_system not in result:
                    deploy_watcher = rm_deploy.DEPLOY_WATCHER.get(deploy_info.deploy_system.name)
                    result[deploy_info.deploy_system] = deploy_watcher(self, token) if deploy_watcher else None
        return result

    @staticmethod
    def release_diff(prod, candidate):
        """
        Get release diff between prod and candidate.

        :param prod: response of get_current_version handler (dict)
        :param candidate: rm_core.ReleasedResource
        :return: ReleaseDiff
        """
        logging.debug("Release diff between:\n%s\nand\n%s", prod, candidate.to_json())

        prod_major_number = int(prod.get("scope_number", 0))
        candidate_major_number = int(candidate.major_release)

        if candidate_major_number > prod_major_number:
            position = rm_core.ReleaseDiff.Position.new
        elif candidate_major_number < prod_major_number:
            position = rm_core.ReleaseDiff.Position.old
        else:
            position = rm_core.ReleaseDiff.Position.same

        return rm_core.ReleaseDiff(position, staging=candidate.status)

    def check_release_blocked(self, release_stage, major_release_num, minor_release_num):
        """
        Check if release is blocked by something.

        :param release_stage: rm_const.ReleaseStatus
        :param major_release_num: int Major Release Number (Branch Scope)
        :param minor_release_num: int Minor Release Number (Tag index)
        :return: Ok() if release is allowed, Error({"msg": str, "fail_task": bool}) if release is blocked
        """
        return rm_core.Ok()

    def get_release_followers(self, release_stage):
        return []

    def get_release_notes(
        self,
        release_stage,
        release_items,
        release_author,
        major_release_num=0,
        minor_release_num=0,
        custom_release_notes="",
    ):
        # common part
        message = []
        if release_items:
            message.append("Release items:\n{}".format("\n".join("- {}".format(i) for i in release_items)))
            message.append("")
        # custom part
        if custom_release_notes:
            message.append(custom_release_notes)
            message.append("")
        # footer part
        message.append("Release author: {}".format(release_author))
        responsible = self.get_responsible_for_release()
        if responsible:
            message.append("Responsible for release: {}".format(responsible))
        return "\n".join(message)

    def get_release_subject(
        self,
        release_stage,
        major_release_num=0,
        minor_release_num=0,
        add_info="",
    ):
        subj = "'{}' to {}: {}".format(
            self.display_name, release_stage, self.release_id(major_release_num, minor_release_num)
        )
        if add_info:
            subj = "{}. {}".format(subj, add_info)
        return subj

    @decorators.memoize
    def get_responsible_for_release(self, token=None):
        return rm_responsibility.get_responsible_user_login(self.releases_cfg__responsible, token)

    @decorators.memoize
    def get_responsible_for_component(self, token=None):
        """
        Do NOT override this method

        :return: str, component responsible user
        """
        return rm_responsibility.get_responsible_user_login(self.responsible, token)

    def get_responsible_for_wiki_page(self, token=None):
        return rm_responsibility.get_responsible_user_login(self.changelog_cfg__wiki_page_owner, token)

    def cleanup_after_release(self, release_stage, major_release_num, minor_release_num):
        """Do all cleaning work after release/deploy."""
        pass

    def after_release(
        self,
        release_results, release_stage,
        major_release_num, minor_release_num,
        st_helper=None, st_issue=None, task=None,
    ):
        """Do all necessary after release work, not related to cleaning up."""
        self._after_release_st(
            release_results, release_stage,
            major_release_num, minor_release_num,
            st_helper=st_helper, st_issue=st_issue,
            task=task,
        )

    def _after_release_st(
        self,
        release_results, release_stage,
        major_release_num, minor_release_num,
        st_helper=None, st_issue=None,
        task=None,
    ):
        """Do all manipulations with st tickets after release."""
        if not self.notify_cfg__use_startrek:
            return

        ok = all(i.ok for i in release_results)
        release_id = self.release_id(major_release_num, minor_release_num)
        marker = "[after release]"
        task_marker = "[done by task: {}]".format(lb.task_wiki_link(task.id)) if task else ""

        if task:
            release_item_name = "{} {}".format(
                self.name,
                getattr(task.Parameters, "release_item_name", ""),
            ).strip()
        else:
            release_item_name = self.name

        if not st_issue:
            logging.info("No startrek ticket provided, nowhere to post notifications")
        elif not self.notify_cfg__st__notify_on_release_to_release_st_ticket:
            logging.info("Component '%s' does not allow to post release notifications to tickets", self.name)
        elif (
            self.notify_cfg__st__notify_on_release_to_release_st_ticket is not True and
            release_stage not in self.notify_cfg__st__notify_on_release_to_release_st_ticket
        ):
            logging.info(
                "Component '%s' allows notifications about %s stages only. Skip stage '%s'",
                self.name, list(self.notify_cfg__st__notify_on_release_to_release_st_ticket), release_stage,
            )
        else:
            logging.info("Going to post release notifications to tickets")
            msg = "Release '{release_item_name}' {release_id} was sent to '{stage}' at {dt} **[{status}]**".format(
                release_item_name=release_item_name,
                release_id=release_id,
                stage=release_stage,
                dt=tu.date_ymdhm(),
                status=("OK" if ok else "ERROR"),
            )
            st_helper.create_comment(
                st_issue, "\n".join(
                    ["=====+{}".format(msg)] +
                    list(map(str, release_results)) +
                    ["", task_marker]
                ),
                notify=self.notify_cfg__st__notify_on_robot_comments_to_tickets
            )
            st_issue = st_helper.get_ticket_by_key(st_issue.key)
            st_issue.update(description="{}\n====={}".format(st_issue.description, msg))

        if ok and release_stage == rm_const.ReleaseStatus.stable:
            if self.notify_cfg__st__close_prev_tickets_stage == rm_const.PipelineStage.release:
                st_helper.close_prev_tickets(
                    self, major_release_num,
                    "{} Closing previous tickets after release of {} version\n{}".format(
                        marker, release_id, task_marker,
                    )
                )
            if self.notify_cfg__st__close_linked_tickets:
                st_helper.close_linked_tickets(self, major_release_num)
            query = self.notify_cfg__st__on_release_close_tickets_by_query(st_issue.key) if st_issue else None
            if query:
                self.close_tickets_by_query(
                    query, st_helper,
                    "{} Close ticket, '{}' {} has been released to {}\n{}".format(
                        marker, self.name, release_id, release_stage, task_marker
                    )
                )
            st_helper.execute_transition(self, major_release_num, rm_const.Workflow.DEPLOYING, task)


class NoTrunkReleased(ComponentInfoGeneral):
    """
    Components with intermediary stages (branches, tags) before or after release.

    Suppose, that every such component has tags.
    """

    _first_rev = 0
    _last_rev = 0

    # Path to current root dir, for example "arcadia:/arc/branches/middle/stable-1/"
    _svn_current_path = ""

    def __init__(self):
        super(NoTrunkReleased, self).__init__()
        self._first_revs = {}

    def get_active_releases(self, token=None):
        """Try to get last deploy with fallback to last release."""
        stage = self.releases_cfg__default_stage
        resources = self.get_last_deploy(only_level=stage, token=token)
        if not resources:
            logging.warning("Unable to get last deployed resource, try last released")
            resources = list(self.get_last_release(stage))
        logging.debug("Got active releases: %s", resources)
        return resources

    def path_for_released_resource(self, resource):
        raise NotImplementedError

    @property
    def first_rev(self):
        if not self._first_rev:
            if self.last_scope_num <= 1:
                logging.info("Probably, it's the first usage of release machine")
                self._first_rev = int(self.last_rev) - 1
            else:
                logging.info("First revision is not specified, try to detect it")
                try:
                    resource = self.get_active_releases()[0]
                    major_release = resource.major_release
                    release_path = self.full_scope_path(major_release)
                except Exception as e:
                    major_release = self.last_scope_num - 1
                    eh.log_exception(
                        "Unable to get last deployed and last stable released path. "
                        "Falling back to previous major release number: {}".format(major_release), e
                    )
                    # RMDEV-881
                    release_path = self.full_scope_path(major_release)
                    while not Arcadia.check(release_path) and major_release > 0:
                        logging.info(
                            "Skipping #%s since there's no such branch in Arcadia (%s does not exist)",
                            major_release,
                            release_path,
                        )
                        major_release -= 1
                        release_path = self.full_scope_path(major_release)
                    if not major_release:
                        raise Exception("Unable to determine first revision since no previous branches can be found")
                trunk_rev_of_branching = rm_svn.SvnHelper.get_last_trunk_revision_before_copy(
                    release_path,
                    repo_base_url=self.svn_cfg__repo_base_url,
                    trunk_url=self.svn_cfg__trunk_url,
                )
                self._first_rev = trunk_rev_of_branching + 1
            logging.info("First revision detected: %s", self._first_rev)
        return self._first_rev

    @property
    def last_rev(self):
        if not self._last_rev:
            logging.info("Last revision is not specified, try to detect it")
            try:
                last_rev = Arcadia.info(self.last_scope_path)["commit_revision"]
                logging.info("Last revision detected: %s", last_rev)
                self._last_rev = int(last_rev)
            except Exception as exc:
                eh.log_exception("Unable to detect last_rev. Let it be 0", exc)
        return self._last_rev

    def set_first_revs(self, revision=0):
        if revision:
            self._first_revs = {int(revision): self.releases_cfg__resources_info}

    def set_first_rev(self, first_rev=0):
        self._first_rev = int(first_rev)

    def set_last_rev(self, last_rev=0):
        self._last_rev = int(last_rev)

    def get_stable_release_label(self, release_item, release_num):
        release_numbers = self.get_release_numbers(release_item)
        if release_numbers:
            release_numbers = "-".join([i for i in release_numbers if i])

        return "{} ({}): {}".format(self.name, release_item.name, release_numbers or release_num)

    # common methods for any release type:
    @abstractproperty
    def prev_major_release_path(self):
        pass

    @abstractproperty
    def last_scope_path(self):
        pass

    @abstractmethod
    def full_scope_path(self, release_num):
        pass

    @abstractproperty
    def last_scope_num(self):
        pass

    @abstractmethod
    def scope_path_short(self, num):
        pass

    @abstractmethod
    def scope_folder_name(self, num):
        pass

    @abstractmethod
    def last_tag_num(self, *args):
        pass

    def get_tag_info_from_build_task(self, build_task_id, build_ctx_key):  # todo: replace with get_release_numbers

        json_ci_context = rm_utils.get_ctx_field(build_task_id, rm_const.CI_CONTEXT_KEY)

        if json_ci_context:

            version_info = json_ci_context.get("version_info", {})

            logging.info("CI context version_info: %s", version_info)

            branch_number = version_info.get("major", "0") or "0"
            tag_number = version_info.get("minor", "0") or "0"

            return int(branch_number) or None, int(tag_number)

        build_arc_url = rm_utils.get_input_or_ctx_field(build_task_id, build_ctx_key)
        if not build_arc_url:  # RMDEV-503: In some cases, there is no build_arc_url parameter
            return None, None
        tag_patterns = [
            r".*/{}/arcadia".format(self.svn_cfg__tag_folder_pattern),
            r".*/{}".format(self.svn_cfg__tag_folder_pattern),
        ]
        for tag_pattern in tag_patterns:
            found_tag = re.findall(tag_pattern, build_arc_url)
            if found_tag:
                if isinstance(found_tag[0], tuple):
                    return found_tag[0]
                return None, found_tag[0]
        eh.check_failed("Can't get tag from task: {} by build_ctx_key = {}. Tag patterns: {}".format(
            lb.task_link(build_task_id, plain=True), build_ctx_key, tag_patterns
        ))

    def update_config_wiki_pages(self, *args):
        pass

    def cleanup_on_post_deploy(self, major_release, st_helper):
        pass

    def get_release_notes(
        self,
        release_stage,
        release_items,
        release_author,
        major_release_num=0,
        minor_release_num=0,
        st_issue=None,
        custom_release_notes="",
    ):
        # common part
        message = []
        if release_items:
            message.append("Release items:\n{}".format("\n".join("- {}".format(i) for i in release_items)))
            message.append("")
        if major_release_num and getattr(self, "changelog_cfg__wiki_page", None):  # todo: to something with this ugly hack
            url = [
                rm_const.Urls.WIKI.strip("/"),
                self.changelog_cfg__wiki_page.strip("/"),
                self.scope_folder_name(major_release_num).strip("/"),
            ]
            if getattr(self, "svn_cfg__branch_folder_name", None):  # means Branched component
                url.append(self.svn_cfg__tag_folder_name(major_release_num, minor_release_num).strip("/"))
            message.append("Changelog: {}".format("/".join(url)))
            message.append("")
        # stage-specific part
        if release_stage == rm_const.ReleaseStatus.stable:
            if st_issue:
                message.append("Acceptance ticket: {}".format(lb.st_link(st_issue.key, plain=True)))
        message.append("")
        # custom part
        if custom_release_notes:
            message.append(custom_release_notes)
            message.append("")
        # changelog part
        if getattr(self, "changelog_cfg__add_to_release_notes", None):
            changelog_lines = rm_ch.build_changelog_lines(
                self.name,
                major_release_num,
                minor_release_num,
            )
            if changelog_lines:
                message.extend(changelog_lines)
                message.append("")
        # footer part
        message.append("Release author: {}".format(release_author))
        responsible = self.get_responsible_for_release()
        if responsible:
            message.append("Responsible for release: {}".format(responsible))
        return "\n".join(message)


class Tagged(NoTrunkReleased):
    """
    Components without branches phase in release cycle.

    Typical release cycle: tag -> build -> tests -> release
                            |                         ^
                            changelog -> ticket ------^
    """

    @property
    def prev_major_release_path(self):
        try:
            last_released_res = self.get_active_releases()[0]
            tag_n = last_released_res.major_release
            return self.full_tag_path(tag_n)
        except Exception as exc:
            eh.log_exception("Smth goes wrong %s", exc)
            logging.debug("Return prev_tag_path")
            return self.prev_tag_path()

    def path_for_released_resource(self, resource):
        logging.debug("Getting release path for resource: %s", resource)
        tag_num = None
        if resource and resource.major_release:
            if int(resource.major_release) > 100000:
                logging.warning("Major release: %s > 100000. Probably, it is not tag", tag_num)
            else:
                tag_num = int(resource.major_release)
        if tag_num is None:
            tag_num = self.last_tag_num() - 1

        while tag_num > 0:
            release_path = self.full_tag_path(tag_num)
            if Arcadia.check(release_path):
                return release_path
            logging.debug(
                "Skipping #%s since there's no such tag in Arcadia (%s does not exist)",
                tag_num, release_path,
            )
            tag_num -= 1

    @property
    def last_scope_path(self):
        return self.last_tag_path()

    def full_scope_path(self, release_num):
        return self.full_tag_path(release_num)

    @property
    def last_scope_num(self):
        return self.last_tag_num()

    def scope_path_short(self, num):
        return os.path.join(self.svn_cfg__tag_name, self.scope_folder_name(num))

    def scope_folder_name(self, num):
        return self.svn_cfg__tag_folder_name(num)

    __last_tag_num = None

    def prev_tag_path(self):
        return self.full_tag_path(self.last_tag_num() - 1)

    def last_tag_path(self):
        return self.full_tag_path(self.last_tag_num())

    def next_tag_path(self):
        return self.full_tag_path(self.last_tag_num() + 1)

    def tag_path(self, tag_num, tag_dir, branch_num=None):
        return os.path.join(tag_dir, self.svn_cfg__tag_folder_name(tag_num))

    def full_tag_path(self, tag_num, branch_num=None):
        # Branch_num is never used, but needed for the same interface as in Branched components
        return self.tag_path(tag_num, self.svn_cfg__tag_dir)

    def relative_tag_path(self, tag_num, branch_num=None):
        return self.tag_path(
            tag_num,
            os.path.join(self.svn_cfg__REPO_NAME, self.svn_cfg__tag_folder, self.svn_cfg__tag_name),
        )

    def last_tag_num(self):
        if self.__last_tag_num is None:
            logging.debug("Getting last tag number")
            self.__last_tag_num = int(rm_svn.SvnHelper.get_highest_folder(
                self.svn_cfg__tag_dir, self.svn_cfg__tag_folder_pattern, only_num=True)
            )
        return self.__last_tag_num

    @staticmethod
    def get_release_numbers_from_attrs(resource):
        return rm_th.get_resource_attr(resource, "major_release_num"), None


class Branched(NoTrunkReleased):
    """Components with branches phase in release cycle."""

    def new_release_trigger_activated(self, *args, **kwargs):
        """
        Check if new release is triggered.

        :param nanny_token: Nanny token value used to determine current version of component as self.get_last_deploy's parameter
        :param rm_client: sandbox.projects.release_machine.client.RMClient instance
        :rtype bool
        :return: True if Release machine should try to create new release, False otherwise.
        """
        return self.success_deploy_trigger_activated(*args, **kwargs)

    def success_deploy_trigger_activated(
        self,
        how_many_resources_should_be_deployed=all,
        *args, **kwargs
    ):
        """
        Check if deployed component version differs from last scope known to release machine.

        :param how_many_resources_should_be_deployed: any or all.
        :param nanny_token: Nanny token value used to determine current version of component as self.get_last_deploy's parameter
        :param rm_client: sandbox.projects.release_machine.client.RMClient instance
        :rtype bool
        """
        if not self.svn_cfg__allow_autobranches:
            return False
        nanny_token = kwargs.pop("nanny_token", None)
        rm_client = kwargs.pop("rm_client", client.RMClient())

        current_scope_num = int(self.last_scope_num)

        # Detect if last release was deployed to production
        deployed_resources = self.get_last_deploy(nanny_token, only_level=rm_const.ReleaseStatus.stable)
        if not deployed_resources:
            return False
        new_resource_deployed = how_many_resources_should_be_deployed([
            int(deployed_resource.major_release) == current_scope_num
            for deployed_resource in deployed_resources or []
        ])
        logging.debug("New release was%s deployed", "" if new_resource_deployed else " not")
        if not new_resource_deployed:
            return False

        # Check if new release was already created
        pre_release_button_events = rm_client.get_events(self.name, ["PreReleaseButton"], length=1).get("events", [])
        last_pre_release_scope_num = int(json.loads(pre_release_button_events[0]["info"])["scope_number"])
        next_scope_num = current_scope_num + 1
        new_release_created = pre_release_button_events and last_pre_release_scope_num >= next_scope_num
        logging.debug("New release %d was %s created", last_pre_release_scope_num, "already" if new_release_created else "not")

        return not new_release_created

    def stable_release_initialized_for_the_latest_scope(self, *args, **kwargs):
        """
        Check if anyone pushed Release button on the latest branch (any tag of this branch)

        RMDEV-2353
        """

        logging.info("Checking if anyone pushed Release button on the latest branch")

        if not self.svn_cfg__allow_autobranches:
            logging.info("Autobranches are not allowed for %s", self.name)
            return False

        rm_client = kwargs.pop("rm_client", client.RMClient())

        scope_result = rm_client.get_scopes(
            component_name=self.name,
            limit=1,
        )

        if not scope_result:
            logging.info("No scopes found for component %s: RM returned an empty result", self.name)
            return False

        latest_scope = scope_result.get("branchScopes", [None])[0]

        if not latest_scope:
            logging.info("Unable to find latest scope for %s: branch scope list is empty", self.name)
            return False

        latest_scope_number = latest_scope.get("scopeNumber")

        if not latest_scope_number:
            logging.info(
                "Unable to retrieve latest scope number of %s: RM returned the following result %s",
                self.name,
                scope_result,
            )
            return False

        logging.info("Latest scope number found: %s", latest_scope_number)

        event_limit = 10

        event_result = rm_client.get_events(
            component_name=self.name,
            event_types=["ReleaseButton"],
            length=event_limit,
        )

        if not event_result:
            logging.info("No ReleaseButton events found for %s: RM returned an empty result", self.name)
            return False

        for event_item in event_result.get("events", []):

            logging.debug("Considering event %s", event_item)

            try:

                release_item_name = event_item["event"]["releaseButtonData"]["releaseItem"]

                if "stable" in release_item_name.lower():

                    logging.debug("Found it! \n%s\n Retrieving scope number", event_item)

                    latest_release_button_pressed_scope_number = event_item["event"]["releaseButtonData"]["scopeNumber"]

                    return int(latest_release_button_pressed_scope_number) == int(latest_scope_number)

                logging.debug("Skip (not a stable release)")

            except KeyError:
                logging.exception(
                    "ReleaseButton event critical failure: expected key cannot be found in event %s",
                    event_item,
                )
                return False

        logging.info(
            "No stable release attempts found among the %s latest ReleaseButton events",
            event_limit,
        )
        return False

    def stable_release_trigger_activated(self, *args, **kwargs):
        """
        Check if last stable released component version differs from last scope known to release machine.

        :param rm_client: sandbox.projects.release_machine.client.RMClient instance
        :rtype bool
        """
        if not self.svn_cfg__allow_autobranches:
            return False
        rm_client = kwargs.pop("rm_client", client.RMClient())

        current_scope_num = int(self.last_scope_num)
        next_scope_num = current_scope_num + 1

        last_stable_release = next(self.get_last_release(), None)
        if last_stable_release is None:
            logging.warning("[%s] Last stable release not found. Trigger not activated", self.name)
            return False
        last_stable_release_scope_num = int(last_stable_release.major_release)
        last_scope_was_released = current_scope_num == last_stable_release_scope_num
        logging.info(
            "current_scope_num = %d, last_stable_release_scope_num = %d",
            last_scope_was_released, last_stable_release_scope_num,
        )
        logging.info("last_scope_was_released = %s", last_scope_was_released)
        if not last_scope_was_released:
            return False

        pre_release_button_events = rm_client.get_events(self.name, ["PreReleaseButton"], length=1).get("events", [])
        loaded_scope_num = json.loads(pre_release_button_events[0]["info"])["scope_number"]
        new_release_created = pre_release_button_events and loaded_scope_num == next_scope_num
        logging.info("new_release_created = %s", new_release_created)

        return not new_release_created

    def new_release_requirements_satisfied(self, *args, **kwargs):
        """
        Check if all requirements satisfied for a new release to be launched.

        By default there is no restrictions.
        """
        return True

    def get_new_release_revision(self, *args, **kwargs):
        """Return revision to launch new release on. Trunk's HEAD by default."""
        # TODO: Use last revision that appeared in component's trunk base.
        arcadia_trunk_url = svn.Arcadia.trunk_url()
        return svn.Arcadia.info(arcadia_trunk_url)["commit_revision"]

    @staticmethod
    def release_diff(prod, candidate):
        diff = super(Branched, Branched).release_diff(prod, candidate)
        if diff.position is not rm_core.ReleaseDiff.Position.same:
            return diff

        prod_minor_number = int(prod.get("tag_number", 0))
        candidate_minor_number = int(candidate.minor_release or 0)

        if candidate_minor_number > prod_minor_number:
            diff.position = rm_core.ReleaseDiff.Position.new
            diff.release_type = rm_core.ReleaseDiff.Type.minor
        elif candidate_minor_number < prod_minor_number:
            diff.position = rm_core.ReleaseDiff.Position.old
            diff.release_type = rm_core.ReleaseDiff.Type.minor

        return diff

    def path_for_released_resource(self, resource):
        logging.debug("Getting release path for resource: %s", resource)
        if resource and resource.major_release and resource.minor_release is not None:
            branch_num, tag_num = int(resource.major_release), int(resource.minor_release)
            if tag_num == 0:
                while branch_num > 0:
                    release_path = self.full_branch_path(branch_num)
                    if Arcadia.check(release_path):
                        return release_path
                    branch_num -= 1
            if tag_num > 1000:
                logging.warning(
                    "Tag num: %s > 1000. Probably, it was found wrong. Will use tag num = %s",
                    tag_num,
                    self.first_tag_num,
                )
                tag_num = self.first_tag_num
        else:
            branch_num = self.last_scope_num - 1
            tag_num = self.first_tag_num
        while branch_num > 0:
            while tag_num > 0:
                release_path = self.full_tag_path(tag_num, branch_num)
                if Arcadia.check(release_path):
                    return release_path
                logging.debug(
                    "Skipping #%s-%s since there's no such tag in Arcadia (%s does not exist)",
                    branch_num, tag_num, release_path,
                )
                tag_num -= 1
            branch_num -= 1
            tag_num = self.first_tag_num

    @staticmethod
    def get_release_numbers_from_attrs(resource):
        major = rm_th.get_resource_attr(resource, "major_release_num")
        minor = rm_th.get_resource_attr(resource, "minor_release_num")
        logging.debug("Got release numbers attrs: %s, %s", major, minor)
        return major, minor

    def min_allowed_release_branch_num(self):
        if self.svn_cfg__max_active_branches:
            return self.last_branch_num + 2 - self.svn_cfg__max_active_branches
        else:
            return self.last_branch_num

    def stable_release_notes(self, **kwargs):
        message = []
        item = kwargs.get("item", None)
        st_issue = kwargs.get("st_issue", None)
        add_info = kwargs.get("add_info", None)

        if isinstance(self, mixin.Changelogged):
            minor_release_num = kwargs.get("minor_release_num", None)
            if not minor_release_num and item:
                _, minor_release_num = self.get_release_numbers(item)
            release_num = kwargs.get("release_num", None)
            if self.changelog_cfg__wiki_page:
                message.append(
                    "Changes in release: {}".format(
                        self.changelog_minor_url(release_num, minor_release_num)
                    )
                )
                message.append("Changes in branch: {}".format(
                    "".join([rm_const.Urls.WIKI, self.changelog_major_url(release_num)]),
                ))
        rest_message = super(Branched, self).stable_release_notes(
            st_issue=st_issue, item=item, add_info=add_info
        )
        if rest_message and message:
            rest_message = "\n" + rest_message

        return "\n".join(message) + rest_message

    def get_branch_id_from_path(self, path):
        branch_names = re.findall(self.svn_cfg__branch_folder_pattern, path)
        eh.ensure(branch_names, "Cannot find correct branch name from path: {}".format(path))
        branch_id = branch_names[0]
        logging.info("Detected branch id = %s", branch_id)
        if not branch_id.isdigit():
            raise ValueError("Expected a numerical branch id, but got {}".format(branch_id))
        return int(branch_id)

    @property
    def source_branch_url(self):
        """Source branch for the branching mode branch->branch."""
        return None

    @property
    def prev_major_release_path(self):
        try:
            last_released_res = self.get_active_releases()[0]
            branch_n = last_released_res.major_release
            return self.full_branch_path(branch_n)
        except Exception as exc:
            eh.log_exception("Smth goes wrong %s", exc)
            logging.debug("Return prev_branch_path")
            return self.prev_branch_path

    @property
    def last_scope_path(self):
        return self.last_branch_path

    def full_scope_path(self, release_num):
        return self.full_branch_path(release_num)

    @property
    def last_scope_num(self):
        return self.last_branch_num

    def scope_path_short(self, num):
        return os.path.join(self.svn_cfg__branch_name, self.scope_folder_name(num))

    def scope_folder_name(self, num):
        return self.svn_cfg__branch_folder_name(num)

    __last_tag_num = {}

    @property
    def prev_branch_path(self):
        return self.full_branch_path(self.prev_branch_num)

    @property
    def last_branch_path(self):
        return self.full_branch_path(self.last_branch_num)

    @property
    def next_branch_path(self):
        return self.full_branch_path(self.next_branch_num)

    def full_branch_path(self, branch_num):
        return self.svn_cfg__branch_path(branch_num)

    @property
    def prev_branch_num(self):
        return self.last_branch_num - 1

    @property
    def next_branch_num(self):
        return self.last_branch_num + 1

    def component_branch_name(self, branch_num):
        return "{}/{}".format(self.svn_cfg__branch_name, self.svn_cfg__branch_folder_name(branch_num))

    @decorators.memoized_property
    def last_branch_num(self):
        return int(rm_svn.SvnHelper.get_highest_folder(
            self.svn_cfg__branch_dir, "^{}/".format(self.svn_cfg__branch_folder_pattern), only_num=True
        ))

    # tag methods:
    def prev_tag_path(self, branch_num):
        return self.full_tag_path(self.last_tag_num(branch_num) - 1, branch_num)

    def last_tag_path(self, branch_num):
        return self.full_tag_path(self.last_tag_num(branch_num), branch_num)

    def next_tag_path(self, branch_num):
        return self.full_tag_path(self.last_tag_num(branch_num) + 1, branch_num)

    def tag_path(self, tag_num, tag_dir, branch_num):
        return os.path.join(tag_dir, self.svn_cfg__tag_folder_name(branch_num, tag_num))

    def full_tag_path(self, tag_num, branch_num):
        return self.tag_path(tag_num, self.svn_cfg__tag_dir, branch_num)

    def arc_full_tag_path(self, tag_num, branch_num):
        return self.tag_path(tag_num, self.svn_cfg__arc_tag_dir, branch_num)

    def relative_tag_path(self, tag_num, branch_num):
        return self.tag_path(
            tag_num,
            os.path.join(self.svn_cfg__REPO_NAME, self.svn_cfg__tag_folder, self.svn_cfg__tag_name),
            branch_num,
        )

    def last_tag_folder_name(self, branch_num=None):
        if branch_num is None:
            branch_num = self.last_branch_num
        return self.svn_cfg__tag_folder_name(branch_num, self.last_tag_num(branch_num))

    def tag_wiki_page_name(self, branch_num=None, tag_num=None):
        if tag_num is None:
            return self.last_tag_folder_name(branch_num)
        if branch_num is None:
            branch_num = self.last_branch_num
        return self.svn_cfg__tag_folder_name(branch_num, tag_num)

    def last_tag_num(self, branch_num):
        if branch_num not in self.__last_tag_num:
            logging.debug("Getting last tag number")
            regex = self.svn_cfg__tag_folder_template.format(
                tag_prefix=self.svn_cfg__tag_prefix,
                branch_num=branch_num,
                tag_num="([0-9]+)",
            )
            self.__last_tag_num[branch_num] = int(
                rm_svn.SvnHelper.get_highest_folder(self.svn_cfg__tag_dir, regex, only_num=True)
            )
        return self.__last_tag_num[branch_num]

    # Whether to ignore release jobs when looking for last good revision https://st.yandex-team.ru/TESTENV-2221
    testenv_last_good_revision_ignore_release_jobs = False

    @property
    def testenv_add_tests(self):
        """
        List of TE test names.

        Use to add only specified tests from databases in testenv_cfg__merge_on_clone.
        Use only with testenv_cfg__merge_on_clone.
        """
        return None

    def testenv_db_stop_names(self, branch_num):
        return [
            self.testenv_cfg__db_template.format(testenv_db_num=n) for n in self.testenv_cfg__db_stop_range(branch_num)
        ]

    def testenv_db_drop_names(self, branch_num):
        return [
            self.testenv_cfg__db_template.format(testenv_db_num=n) for n in self.testenv_cfg__db_drop_range(branch_num)
        ]

    def cleanup_on_post_deploy(self, major_release, st_helper):
        try:
            st_helper.close_prev_tickets(
                self, major_release,
                "[cleanup on post deploy] Closing previous tickets after deploy of {} version".format(major_release)
            )
            params = {
                "stop_database_names": [self.testenv_cfg__db_template.format(testenv_db_num=major_release - 1)]
            }
            success = TEClient.cleanup_db(params)
            eh.ensure(success, "Cleanup databases failed")
        except Exception as exc:
            eh.log_exception("Couldn't cleanup after deploy", exc)

    def check_testenv_db(self, task, add_db=()):
        """
        Check if task is allowed to run from particular testenv_database.

        :param task: Sandbox task
        :param add_db: list with additional allowed db regexps
        """
        te_db = rm_th.ctx_field(task, "testenv_database")
        if not te_db:
            return  # allow to run task from sandbox directly

        if isinstance(add_db, (list, tuple)):
            add_db = tuple(add_db)
        elif isinstance(add_db, (six.text_type, six.binary_type)):
            add_db = (add_db,)
        elif add_db:
            eh.check_failed("Unknown additional database input")

        if self.testenv_cfg__branch_db_regex.match(te_db, 0):
            return
        for db_regex in add_db:
            if re.match(db_regex, te_db, 0):
                return

        eh.check_failed(
            "Testenv database '{}' is not allowed to launch "
            "this release-machine test for '{}' component. "
            "Allowed db regexps are: {}".format(
                te_db,
                self.name,
                ", ".join([self.testenv_cfg__branch_db_regex.pattern] + list(add_db)),
            )
        )

    def check_merge_permissions(self, task, user):
        key = rm_sec.get_rm_token(task)
        permission_type, people_groups = self.merges_cfg__permissions
        if not people_groups:
            return True  # allow by default
        if permission_type == rm_const.PermissionType.ALLOWED:
            if people_groups.abc_services and self._user_in_abc_services(key, user, people_groups.abc_services):
                return True
            if people_groups.staff_groups and self._user_in_staff_groups(key, user, people_groups.staff_groups):
                return True
            if people_groups.logins and user in people_groups.logins:
                return True
            return False
        elif permission_type == rm_const.PermissionType.BANNED:
            if people_groups.abc_services and self._user_in_abc_services(key, user, people_groups.abc_services):
                return False
            if people_groups.staff_groups and self._user_in_staff_groups(key, user, people_groups.staff_groups):
                return False
            if people_groups.logins and user in people_groups.logins:
                return False
            return True
        raise RuntimeError("Unknown permission type: {}".format(permission_type))

    @staticmethod
    def _user_in_abc_services(rm_token, user, abc_services):
        abc_api = abc_client.AbcClient(rm_token)
        for abc_service in abc_services:
            abc_people = abc_api.get_people_from_service(abc_service.component_id, abc_service.role_id)
            if user in abc_people:
                return True
        return False

    @staticmethod
    def _user_in_staff_groups(rm_token, user, staff_groups):
        staff_api = rm_staff_helper.StaffApi(rm_token)
        user_groups = staff_api.get_user_groups(user)
        for user_group in user_groups:
            if user_group in staff_groups:
                return True
        return False

    def after_release(
        self,
        release_results, release_stage,
        major_release_num, minor_release_num,
        st_helper=None, st_issue=None, task=None,
    ):
        super(Branched, self).after_release(
            release_results, release_stage,
            major_release_num, minor_release_num,
            st_helper=st_helper, st_issue=st_issue, task=task,
        )
        if release_stage == rm_const.ReleaseStatus.stable and self.testenv_cfg:
            self._cleanup_testenv_dbs_after_release(major_release_num)

    def _cleanup_testenv_dbs_after_release(self, major_release_num):
        success = TEClient.cleanup_db({
            'stop_database_names': self.testenv_db_stop_names(major_release_num)
        })
        if not success:
            logging.error("Unable to stop previous TE database: %s", success)


class BranchedFromBranch(Branched):
    """Class for components which create branch from other component's branches."""

    @property
    def first_rev(self):
        if self._first_rev:
            return self._first_rev

        # probably, it's the first usage of release machine
        if self.prev_branch_num == 0:
            self._first_rev = int(self.last_rev) - 1
            logging.info("First revision detected: %s", self._first_rev)
            return self._first_rev

        logging.info("First revision is not specified, try to detect it")
        try:
            last_released_res = next(self.get_last_release())
            branch_n = last_released_res.major_release
            release_path = self.full_branch_path(branch_n)
        except Exception:
            release_path = self.prev_branch_path
            eh.log_exception(
                "Unable to get last stable released path. Falling back to previous branch path: {}".format(release_path)
            )
        # look for the first trunk revision before revision of branching
        curr_rev = "HEAD"
        for it in range(2):
            branching_rev = Arcadia.log(release_path, 'r0', curr_rev, limit=1, stop_on_copy=True)[0]["revision"]
            logging.info("branching point at r%s", branching_rev)
            curr_rev = int(Arcadia.log(release_path, branching_rev, 'r0', limit=2)[-1]["revision"])
            logging.info("parent point at r%s", curr_rev)
        self._first_rev = curr_rev + 1
        logging.info("First revision detected: %s", self._first_rev)
        return self._first_rev

    # Test2
    def find_parent_branch_path(self, full_path):
        """
        Check if one of dirs 'arcadia', 'arcadia_tests_data', 'data' is in full_path and return path prefix and flag.

        Example: full_path = "arc/branches/user_sessions/stable-29/arcadia/sup/stat/ya.make",
        we should return "arc/branches/user_sessions/stable-29".
        :param full_path: string
        :return: full_path prefix, True if one of root_dirs in full_path, False else
        """
        root_dirs = frozenset(['arcadia', 'arcadia_tests_data', 'data'])
        dirs = full_path.split('/')
        for i in range(len(dirs)):
            if dirs[i] in root_dirs:
                return '/'.join(dirs[:i]), True
        return '/'.join(dirs), False

    def fallback_for_main_url(self, last_rev):
        """
        Fallback for components, which don't use our automation, so they don't have.

        :param last_rev: revision in branch
        :return: full_path prefix, True if one of root_dirs in full_path, False else
        """
        curr_branch_revisions = Arcadia.log(rm_svn.BRANCH_PATH, last_rev, limit=10, stop_on_copy=True)
        if len(curr_branch_revisions) == 10:
            return "", False
        branch_rev = curr_branch_revisions[-1]['revision']
        path_in_branched_rev = curr_branch_revisions[-1]["paths"][0][1]
        _, parent_branch_rev_log = Arcadia.log(
            Arcadia.ARCADIA_BASE_URL + path_in_branched_rev, branch_rev, 'r0', limit=2,
        )
        path_in_prev_rev_log = parent_branch_rev_log["paths"][0][1]
        parent_branch_path, _ = self.find_parent_branch_path(path_in_prev_rev_log)
        logging.debug("Got parent branch path: %s", parent_branch_path)
        return parent_branch_path, True

    @property
    def svn_cfg__main_url(self):
        last_rev = self.last_rev
        if not self._svn_cfg__main_url:
            curr_branch_last_rev_log = Arcadia.log(rm_svn.BRANCH_PATH, last_rev, limit=1)[0]
            path_in_last_rev_log = curr_branch_last_rev_log["paths"][0][1]
            log_message = curr_branch_last_rev_log["msg"].replace(" ", "").lower()
            #  Check that last_rev is first rev in branch, so it's path_in_last_rev_log is just /branches/abt/stable-XXX
            branching_regexp = r"\[branch:" + re.escape(self.name) + r"\]"
            branching_rev = re.findall(branching_regexp, log_message)
            if branching_rev:
                _, parent_branch_rev_log = Arcadia.log(
                    Arcadia.ARCADIA_BASE_URL + path_in_last_rev_log, last_rev, 'r0',
                    limit=2,
                )
                path_in_prev_rev_log = parent_branch_rev_log["paths"][0][1]
                parent_branch_path, _ = self.find_parent_branch_path(path_in_prev_rev_log)
                logging.debug("Got parent branch path: %s", parent_branch_path)
                self._svn_cfg__main_url = Arcadia.ARCADIA_BASE_URL + parent_branch_path
                logging.debug("Main_url for this component is %s", self._svn_cfg__main_url)
                return self._svn_cfg__main_url
            parent_branch_path, has_root_dir = self.find_parent_branch_path(path_in_last_rev_log)
            if not has_root_dir:
                parent_branch_path, was_found = self.fallback_for_main_url(last_rev)
                if not was_found:
                    logging.error("Couldn't find one of root dirs in path %s", path_in_last_rev_log)
                    raise AssertionError
            self._svn_cfg__main_url = Arcadia.ARCADIA_BASE_URL + parent_branch_path
            logging.debug("Main_url for this component is %s", self._svn_cfg__main_url)
        return self._svn_cfg__main_url


class ReferenceComponent(Branched, mixin.Changelogged, mixin.Startreked):
    """
    This is the component, which we are strongly recommended to inherit from.

    You can also include other base components in your inheritance list,
    but this component includes most useful of them with most common parameters.
    """


class ReferenceComponentMetricsed(ReferenceComponent, mixin.Metricsed):
    pass


ComponentInfoGeneral = c_info_base.wrap_cls_attribute_errors(ComponentInfoGeneral)
NoTrunkReleased = c_info_base.wrap_cls_attribute_errors(NoTrunkReleased)
Tagged = c_info_base.wrap_cls_attribute_errors(Tagged)
Branched = c_info_base.wrap_cls_attribute_errors(Branched)
BranchedFromBranch = c_info_base.wrap_cls_attribute_errors(BranchedFromBranch)
ReferenceComponent = c_info_base.wrap_cls_attribute_errors(ReferenceComponent)
ReferenceComponentMetricsed = c_info_base.wrap_cls_attribute_errors(ReferenceComponentMetricsed)
