from __future__ import unicode_literals

import logging
import re

from collections import defaultdict

import sandbox.projects.abc.client as abc_client
import sandbox.projects.release_machine.helpers.svn_helper as rm_svn
import sandbox.projects.release_machine.helpers.merge_helper as rm_merge
import sandbox.projects.release_machine.helpers.responsibility_helper as rm_responsibility
import sandbox.projects.release_machine.helpers.staff_helper as rm_staff_helper
import sandbox.projects.common.metrics_launch_manager as metrics_launcher

from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import time_utils as tu
from sandbox.projects.common import decorators
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.sdk2.vcs.svn import Arcadia


class Changelogged(object):
    """Mixin class for components, which need changelogs."""

    wiki_head_template = "===={descr} //r:{r1}-{r2}//\nCreation date: {date}"

    def changelog_wiki_head(self, release_num, r1, r2):
        return self.wiki_head_template.format(
            descr=self.scope_path_short(release_num),
            r1=r1, r2=r2,
            date=tu.date_ymdhm()
        )

    def changelog_title(self, release_num=None):
        return "Release changelog for {}".format(
            self.scope_folder_name(release_num or self.last_scope_num)
        )

    def changelog_minor_title(self, release_num=None, minor_release_num=None):
        if hasattr(self, 'tag_wiki_page_name'):
            return "Minor changelog for {}".format(self.tag_wiki_page_name(release_num, minor_release_num))
        eh.check_failed("Component hasn't minor changelogs.")

    @staticmethod
    def _get_prev_branch_rollbacked_revs(path, revs_list_from_rollback_commit):
        """
        Get trunk revisions that were merged and then rollbacked from branch.

        If rev was not merged with MergeToStable task we put it into strange_revs list.
        If rev was merged with MergeToStable task, we get trunk revision for this rev and put it in rollbacked_revs.
        :param path: Path to branch
        :param revs_list_from_rollback_commit: List with revs from rollback commit message
        :return: Rollbacked_revs and strange_revs - lists
        """
        # todo: probably this function should be moved somewhere
        rollbacked_revs = []
        strange_revs = []
        for rev in revs_list_from_rollback_commit:
            rollbacked_rev_log = Arcadia.log(path, rev, limit=1)[0]
            merged_revs = rm_merge.get_merges_from_commit_message(rollbacked_rev_log["msg"])
            if merged_revs:
                rollbacked_revs.extend(merged_revs)
            else:
                logging.debug("Got strange revision %s in branch", rev)
                strange_revs.append(rev)
        return rollbacked_revs, strange_revs

    def _get_branch_logs(self, branch_path):
        branch_log = Arcadia.log(branch_path, 'r0', 'HEAD', stop_on_copy=True)
        return branch_log

    @staticmethod
    def _filter_merged_and_rollbacked(merged_trunk_revs, rollbacked_trunk_revs):
        """
        Find common revisions in merged_trunk_revs and in rollbacked_trunk_revs and remove them in these lists.

        :param merged_trunk_revs: merged revisions list
        :param rollbacked_trunk_revs: rollbacked revisions list
        :return: filtered merged and rollbacked lists
        """
        common_revs = []
        for rollbacked_rev in rollbacked_trunk_revs:
            if rollbacked_rev in merged_trunk_revs:
                common_revs.append(rollbacked_rev)
        merged_trunk_revs = [rev for rev in merged_trunk_revs if rev not in common_revs]
        rollbacked_trunk_revs = [rev for rev in rollbacked_trunk_revs if rev not in common_revs]
        return merged_trunk_revs, rollbacked_trunk_revs

    def get_prev_branch_revisions(self, branch_path):
        """
        Find merged and not rollbacked revisions in logs from branch_path.

        :param branch_path: Branch path, str
        :return: list of infos for every merged or strange revision in branch.
            Strange revision = revision that was merged
        or rollbacked not with RM tasks - MergeToStable and RollbackCommit.
        """
        prev_branch_log = self._get_branch_logs(branch_path)
        rollbacked_trunk_revs = []  # Revs in trunk, which were rollbacked in branch.
        merged_trunk_revs = []  # Revs in trunk, which were committed in branch.
        strange_revs = []  # Revs that came in branch not with MergeToStable or RollbackCommit tasks
        for log in prev_branch_log:
            log_message = log["msg"]
            branch_str = re.findall(r"\[branch:", log_message)
            if branch_str:
                logging.debug("We got revision for new branch, skipping")
                continue
            rollbacked_revs = rm_merge.get_rollbacks_from_commit_message(log_message)
            if rollbacked_revs:
                parsed_revs = self._get_prev_branch_rollbacked_revs(branch_path, rollbacked_revs)
                rollbacked_trunk_revs.extend(parsed_revs[0])
                strange_revs.extend(parsed_revs[1])
                continue
            merged_revs = rm_merge.get_merges_from_commit_message(log_message)
            if merged_revs:
                merged_trunk_revs.extend(merged_revs)
                continue
            # If we can't find any of our task's prefixes
            logging.debug(
                "We got revision %s that was done not with MergeToStable or RollbackCommit tasks",
                log["revision"],
            )
            strange_revs.append(log["revision"])

        merged_trunk_revs, rollbacked_trunk_revs = self._filter_merged_and_rollbacked(
            merged_trunk_revs,
            rollbacked_trunk_revs,
        )
        strange_revs.extend(rollbacked_trunk_revs)

        merged_rev_infos = []
        strange_rev_infos = []
        for rev in merged_trunk_revs:
            log = Arcadia.log(rm_svn.TRUNK_PATH, rev, limit=1)[0]
            merged_rev_infos.append(log)
        for rev in strange_revs:
            log = Arcadia.log(branch_path, rev, limit=1)[0]
            strange_rev_infos.append(log)
        logging.debug(
            "Previous branch merged revs info: %s\n"
            "Strange revs info: %s",
            merged_rev_infos,
            strange_rev_infos,
        )
        merged_rev_infos.extend(strange_rev_infos)
        return merged_rev_infos

    def changelog_minor_url(self, release_id=None, minor_release_num=None):
        if not self.changelog_cfg__wiki_page:
            return None
        if hasattr(self, 'tag_wiki_page_name'):
            return "{}{}/".format(
                self.changelog_major_url(release_id),
                self.tag_wiki_page_name(release_id, minor_release_num)
            )
        eh.check_failed("Component hasn't minor changelogs.")

    def changelog_major_url(self, release_id=None):
        if not self.changelog_cfg__wiki_page:
            return None
        return "".join([
            self.changelog_cfg__wiki_page.rstrip("/"),
            "/",
            self.scope_folder_name(release_id or self.last_scope_num).strip("/"),
            "/"
        ])


class Startreked(object):
    """Abstract methods for work with startrek."""

    def st_description(self, release_num=None):
        te_db_name = self.testenv_cfg__db_template.format(testenv_db_num=release_num)
        te_links = "Tests: (({manage} TE {title})), (({timeline} timeline))".format(
            manage=rm_const.Urls.te_db_screen(te_db_name),
            title=te_db_name,
            timeline=rm_const.Urls.te_db_screen(te_db_name, screen_name="timeline"),
        )
        if getattr(self, "changelog_cfg__wiki_page", None):
            changelog_wiki_link = "Changelogs: {}".format("".join([
                rm_const.Urls.WIKI, self.changelog_major_url(release_num)
            ]))
        else:
            changelog_wiki_link = None
        release_chat_link = self.notify_cfg__tg__invite_link or "((https://nda.ya.ru/3UZi4o not specified))"
        description = [
            self.notify_cfg__st__ticket_description_prefix,
            te_links,
            changelog_wiki_link,
            "Release chat: {}".format(release_chat_link),
            self.link_to_ui(
                major_release_num=release_num,
                link_type=lb.LinkType.wiki
            ),
            "Responsible for release: staff:{}\n".format(self.get_responsible_for_release()),
            self.notify_cfg__st__ticket_description_suffix,
        ]
        return "\n".join([str(i) for i in description if i is not None])

    def filter_tickets(self, changelog_entry):
        """
        Return filtered list with tickets.

        :param changelog_entry: changelog entry
        :return: filtered set
        """
        filtered_tickets = set()
        ticket_filter = self.notify_cfg__st__ticket_filter
        all_related_tickets = changelog_entry["startrek_tickets"]
        if ticket_filter.filter_type == rm_const.StartrekTicketFilterTypes.include:
            for ticket in all_related_tickets:
                if ticket_filter.queues and ticket.split('-')[0] in ticket_filter.queues:
                    filtered_tickets.add(ticket)
                if ticket_filter.tickets and ticket in ticket_filter.tickets:
                    filtered_tickets.add(ticket)
        else:  # exclude
            for ticket in all_related_tickets:
                if ticket_filter.queues and ticket.split('-')[0] in ticket_filter.queues:
                    logging.debug("Filter ticket %s by queues", ticket)
                    continue
                if ticket_filter.tickets and ticket in ticket_filter.tickets:
                    logging.debug("Filter ticket %s by tickets", ticket)
                    continue
                filtered_tickets.add(ticket)
        return filtered_tickets

    def st_summary(self, release_num):
        return self.notify_cfg__st__summary_template.format(release_num)

    def get_release_num_from_st_summary(self, st_summary):
        template = (
            self.notify_cfg__st__summary_template
                .replace("[", "\\[")
                .replace("]", "\\]")
                .replace("(", "\\(")
                .replace(")", "\\)")
                .replace(".", "\\.")
        )
        found = re.findall(template.format("([0-9]+)"), st_summary)
        if found:
            return found[0]

    @property
    def st_tags(self):
        return ["rm_main_ticket", self.name]

    @decorators.retries(3)
    def custom_st_update(self, auth_token, issue=None):
        return None

    @decorators.retries(3)
    def st_post_release(self, task, auth_token, release_num=None):
        return None

    @staticmethod
    def close_tickets_by_query(query, st_helper, close_msg):
        logging.info("Going to close tickets by query: %s", query)
        found_issues = st_helper.st_client.issues.find(query=query)
        logging.info("Found tickets: %s", ", ".join([str(i.key) for i in found_issues]))
        for i in found_issues:
            st_helper.close_issue(i, close_msg)

    @staticmethod
    def st_create_links(release_ticket, commit_ticket_keys):
        for ticket_key in commit_ticket_keys:
            try:
                link = release_ticket.links.create(issue=ticket_key, relationship="relates")
                logging.info("Link attached: %s", link)
            except Exception as exc:
                eh.log_exception("Cannot create link for ticket: {}".format(ticket_key), exc)

    def get_followers(self, token):
        try:
            people_groups = self.notify_cfg__st__followers
        except Exception as e:
            eh.log_exception("Unable to get followers", e)
            return []
        if isinstance(people_groups, list):
            return people_groups
        staff_api = rm_staff_helper.StaffApi(token)
        abc_api = abc_client.AbcClient(token)
        followers = []
        if people_groups.staff_groups is not None:
            for group in people_groups.staff_groups:
                followers.extend(staff_api.get_users_from_group(group))
        if people_groups.abc_services is not None:
            for service in people_groups.abc_services:
                followers.extend(abc_api.get_people_from_service(service.component_id, service.role_id))
        if people_groups.logins is not None:
            followers.extend(people_groups.logins)
        return list(set(followers))

    def change_ticket_on_post_deploy(self, major_release, st_helper, task):
        try:
            if self.notify_cfg__st__workflow:
                st_helper.execute_transition(self, major_release, rm_const.Workflow.PRODUCTION, task)
        except Exception as exc:
            eh.log_exception("Unable to make transition to 'production' status", exc)

    @decorators.memoized_property
    def st_assignee(self):
        return rm_responsibility.get_responsible_user_login(self.notify_cfg__st__assignee)


class Metricsed(object):
    """Methods for work with Metrics launches inside Release Machine."""

    template_dir = "search/metrics_templates/{search_subtype}"
    template_name = None
    metrics_cfg__default_launch_id = ""

    STATUS_COLORS = {
        "CANCELED": "gray",
        "FAILED": "red",
        "CRITICAL": "red",
        "SUCCESS": "green",
        "PRELIMINARY RESULTS": "yellow",
        "WARN": "yellow",
        "UNKNOWN": "gray",
        "UNDEFINED": "gray",
    }
    ICONS = defaultdict(str, {
        "web": "**WEB**",
        "geo": "**GEO**",
        "images": "**IMG**",
        "video": "**VID**",
        "news": "**NEWS**",
        "music": "**MUSIC**",
    })

    def get_title_for_comment(self, task, launch_status, triggered_sla=False):
        if triggered_sla:
            launch_status += ", SLA VIOLATION"

        st_report_status = launch_status
        final_status = task.Context.launch_info.get("final_status")

        if final_status and launch_status == "COMPLETED":
            st_report_status = final_status

        title = "{} {} !!vs!! {} **!!({}){}!!**".format(
            self.ICONS[task.Parameters.search_subtype],
            task.Context.checked_tag or "priemka",
            task.Context.sample_tag or "production",
            self.STATUS_COLORS.get(st_report_status, "blue"),
            st_report_status,
        )
        return title

    @staticmethod
    def _metrics_short_message(task):
        launch_id = task.Context.launch_info["launch_id"]
        short_message = "(({m_link} {link_name}))\n Sandbox task: {s_task}\n".format(
            m_link=metrics_launcher.get_mlm_link(launch_id),
            link_name="Metrics launch",
            s_task=lb.task_wiki_link(task.id)
        )
        return short_message

    def launch_result_on_start(self, task):
        message = self._metrics_short_message(task)
        launch_status = task.Context.launch_info["status"]
        return self.get_title_for_comment(task, launch_status), message

    def launch_result_in_middle(self, task, report):
        message = [self._metrics_short_message(task)]
        launch_status = "PRELIMINARY RESULTS"
        message.append(report)
        return self.get_title_for_comment(task, launch_status), "".join(message)

    def launch_result_on_end(self, task, report, metrics_json_response, sla_response):
        triggered_sla = metrics_json_response.get("triggeredSla")
        checked_beta_name = str(task.Parameters.checked_beta).split(".")[0]
        findurl_result_site = task.run_findurl(self, metrics_json_response)
        unanswer_problems = task.check_scraper_problems(self, metrics_json_response, checked_beta_name)
        try:
            task.send_tm_notify()
        except Exception as exc:
            eh.log_exception('Telegram notify error', exc)
        message = [self._metrics_short_message(task)]
        if sla_response:
            mlm_ticket = sla_response.partition('#')[0]
            message.append("SLA Violation: {} ({})\n".format(mlm_ticket, sla_response))
        if unanswer_problems:
            message.append(unanswer_problems)
        if findurl_result_site:
            message.append("(({} FindUrlBucket result))\n".format(findurl_result_site))
        launch_status = task.Context.launch_info["status"]
        if launch_status != "CANCELED":
            message.append(report)
        return self.get_title_for_comment(task, launch_status, triggered_sla), "".join(message)
