import logging
import os
import re
import six
from collections import namedtuple

import sandbox.common.types.task as ctt
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_graph as jg
import sandbox.projects.release_machine.components.job_graph.stages.ci_stage as jg_ci
import sandbox.projects.release_machine.components.job_graph.stages.job_graph_element as jge  # noqa: UnusedImport
import sandbox.projects.release_machine.components.config_core.release_block as release_block
import sandbox.projects.release_machine.components.config_core.statistics_page as statistics_page
import sandbox.projects.release_machine.components.config_core.release_and_deploy as release_and_deploy
from sandbox.projects.common import decorators
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import paths
from sandbox.projects.common import time_utils as tu
from sandbox.projects.common.nanny import const as nanny_const
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.components.config_core import responsibility


NotifyOn = namedtuple("NotifyOn", ["task_success", "task_break", "task_failure", "custom"])
MergePermissions = namedtuple("MergePermissions", ["permission_type", "people_groups"])
PeopleGroups = namedtuple("PeopleGroups", ["staff_groups", "abc_services", "logins"])
LOGGER = logging.getLogger(__name__)


class RmTelegramNotifyConfig(object):
    DEFAULT = NotifyOn(
        task_success=[],
        task_break=[],
        task_failure=[],
        custom=[],
    )

    def __init__(self, chats=None, responsible=None, notify_tester=None, rm_maintainers=None):
        self.chats = chats or self.DEFAULT
        self.responsible = responsible or NotifyOn(
            task_success=[],
            task_break=[],
            task_failure=[rm_const.TaskTypes.RELEASE_RM_COMPONENT.name],
            custom=[],
        )
        self.notify_tester = notify_tester or NotifyOn(
            task_success=[],
            task_break=True,
            task_failure=True,
            custom=True,
        )
        self.rm_maintainers = rm_maintainers or NotifyOn(
            task_success=[],
            task_break=[
                rm_const.TaskTypes.ALL_RM_TASKS.name,
                rm_const.TaskTypes.KPI.name,
                rm_const.TaskTypes.RELEASE_RM_COMPONENT.name,
                rm_const.TaskTypes.LAUNCH_METRICS.name,
                rm_const.TaskTypes.CREATE_BRANCH_OR_TAG.name,
                rm_const.TaskTypes.BUILD.name,
                rm_const.TaskTypes.PRIEMKA.name,
                rm_const.TaskTypes.MERGE_TO_STABLE.name,
            ],
            task_failure=[
                rm_const.TaskTypes.RELEASE_RM_COMPONENT.name,
                rm_const.TaskTypes.ALL_RM_TASKS.name,
            ],
            custom=[],
        )


class RmSvnCfg(object):
    ARCADIA = "arcadia:/"
    REPO_NAME = "arc"

    use_arc = False

    # Use direct zipatch applying instead of careful SVN-based merging, see RMDEV-2173 for details
    # Use this option only with use_arc = True.
    use_zipatch_only = False

    @property
    def repo_base_url(self):
        return self.ARCADIA + self.REPO_NAME

    @property
    def trunk_url(self):
        """
            Trunk url for convenience with svn actions.
            Could be different if component is branching from folder inside of trunk.
            Ex. os.path.join(self.repo_base_url, "trunk", "sub_folder")
        """
        return os.path.join(self.repo_base_url, "trunk")

    @property
    def main_url(self):
        """
            Root url for branching and tagging operations
            Same as trunk url for components, copied from trunk directly.
            For branch-to-branch components it will be different.
        """
        return self.trunk_url

    def get_release_numbers(self, arcadia_url):
        """
            Get release numbers from arcadia url
            :return (major_num, minor_num) if we found both of them,
            (major_num, None) if we found just major_num, else None
        """
        # LOGGER.debug("Try to get release numbers from %s", arcadia_url)
        release_numbers = None
        if arcadia_url:
            for regexp in self._release_numbers_regexps:
                regexp_result = regexp.search(arcadia_url)
                if regexp_result:
                    release_numbers = self._get_release_numbers(regexp_result)
                    break
        # LOGGER.debug("Got release numbers: %s", release_numbers)
        return release_numbers

    def get_major_release_num(self, url):
        """
            Get major release number from arcadia url
            :return major_num if we found it, else 0
        """
        if url == "svn ssh url":  # that means testenv unit test
            return 0
        # LOGGER.debug("Try to get major release number from %s", url)
        release_number = self._get_major_release_num(url)
        if release_number:
            return release_number
        # LOGGER.debug("Fallback to get_release_numbers")
        release_numbers = self.get_release_numbers(url)
        if release_numbers:
            return int(release_numbers[0])
        return 0

    def _get_major_release_num(self, *args, **kwargs):
        """
        Get major release number from arcadia url.

        :return major_num if we found it, else 0
        """
        return

    def _get_release_numbers(self, regexp_result):
        """
            Get release numbers from arcadia url
            :return (major_num, minor_num) if we found both of them,
            (major_num, None) if we found just major_num, else None
        """
        return regexp_result.group(1), None

    @decorators.memoized_property
    def _release_numbers_regexps(self):
        return [re.compile("@([0-9]+)")]

    def __init__(self, name):
        self.name = name


class RmTestenvCfg(object):

    @property
    def db_template(self):
        raise NotImplementedError

    @property
    def testenv_db_owners(self):
        """
        List of logins, who should be added in testenv trunk db
        :return: list of strings
        """
        return []

    # Flags used in PrepareRMComponentEnvironment
    enable_notifications_on_task_warnings = True
    enable_notifications_on_db_events = True
    enable_all_notifications_for_db_owners = True

    @property
    def trunk_task_owner(self):
        """ Your team sandbox-group for trunk databases """
        raise NotImplementedError


class ReferenceConfig(object):
    component_group = None

    max_running_task_capacity = 2

    first_tag_num = 1

    class ReleaseViewer(object):
        kpi_alert = 0
        kpi_alerts_recipients = None
        kpi_alert_skip_weekends = False

        statistics_page_charts = statistics_page.DEFAULT
        job_timeline_filters = []

        @property
        def deploy(self):
            return {}

    class Releases(object):
        def __init__(self, root_cfg, name, responsible):
            self._root_cfg = root_cfg
            self.name = name
            self._responsible = responsible

        token_vault_owner = rm_const.COMMON_TOKEN_OWNER
        token_vault_name = rm_const.COMMON_TOKEN_NAME

        allow_old_releases = False
        allow_robots_to_release_stable = False
        release_followers_permanent = []

        deploy_system = rm_const.DeploySystem.nanny
        allow_to_add_resources = True  # for DeploySystem.nanny_push only
        allow_to_remove_resources = False  # for DeploySystem.nanny_push only
        wait_for_deploy_time_sec = 4 * 60 * 60
        default_stage = ctt.ReleaseStatus.STABLE
        use_release_task_as_ti_task_type = False

        block_on_test_results = [release_block.NEVER_BLOCK]
        block_on_test_results_look_behind = False

        infra_service_id = None
        infra_envs = {}
        infra_event_duration_sec = None

        @property
        def root_cfg(self):
            return self._root_cfg

        @property
        def approvement_required(self):
            return self.approvements is not None

        @property
        def approvements(self):
            return None

        @property
        def releasable_items(self):
            # type: () -> List[ri.AbstractReleasableItem]
            """
            List of releasable items descriptions.

            This property is needed for releases via RELEASE_RM_COMPONENT_2 and for release versions monitoring.
            Doc: https://wiki.yandex-team.ru/releasemachine/storedeployinfo/#xranenieopisanijavkonfigekomponenty
            """
            return []

        @property
        def resources_info(self):
            # type: () -> List[Any]
            """
            !!!DEPRECATED!!! Please, use releasable_items instead!

            If you do not release anything via common tools, like nanny etc, please specify resources_info = []
            """
            if self.releasable_items:
                return [
                    release_and_deploy.ReleasedResourceInfo(
                        name=i.name,
                        resource_type=i.data.resource_type,
                        deploy=[
                            release_and_deploy.DeployServicesInfo(
                                services=[deploy_service.name for deploy_service in deploy_info.services],
                                level=deploy_info.stage,
                                chooser=deploy_info.chooser,
                            ) for deploy_info in i.deploy_infos
                        ] if i.deploy_infos is not None else [],
                        attributes=i.data.attributes,
                        build_ctx_key=i.data.build_ctx_key,
                    )
                    for i in self.releasable_items
                    if isinstance(i, ri.ReleasableItem) and isinstance(i.data, ri.SandboxResourceData)
                ]
            raise NotImplementedError

        @property
        def responsible(self):
            """
            Person who is responsible for releases of the component.

            :rtype: Namedtuple with abc group and login or string with login"""
            return self._responsible

        def iter_over_deploy_info(self, release_stage=None):
            for res_info in self.resources_info:
                # LOGGER.debug("Processing resource info %s", res_info)
                if not res_info.deploy:
                    # LOGGER.debug("Deploy info not found, continue")
                    continue
                for deploy_info in self.to_deploy_infos(res_info.deploy):
                    # LOGGER.debug("Processing deploy info: %s", deploy_info)
                    if release_stage and str(deploy_info.level).lower() != str(release_stage).lower():
                        # LOGGER.debug(
                        #    "Deploy info filtered by release stage: %s != %s",
                        #    deploy_info.level, release_stage
                        # )
                        continue
                    yield res_info, deploy_info

        @staticmethod
        def to_deploy_infos(deploy_list):
            # type: (Any) -> List[release_and_deploy.DeployServicesInfo]
            result = {}
            for i in deploy_list:
                if not isinstance(i, release_and_deploy.DeployServicesInfo):
                    i = release_and_deploy.DeployServicesInfo(services=[i[1]], dashboards=[], level=i[0])

                if i.level in result:
                    result[i.level].services = list((set(result[i.level].services or [])) | set(i.services))
                    result[i.level].dashboards = list((set(result[i.level].dashboards or [])) | set(i.dashboards))
                else:
                    result[i.level] = i
            return result.values()

    class SvnCfg(RmSvnCfg):
        """ Only for non-trunk components """
        tag_folder = "tags"
        arc_tag_folder = "tags/releases"

        def __init__(self, name):
            super(ReferenceConfig.SvnCfg, self).__init__(name)
            self._tag_dir = None
            self._arc_tag_dir = None

        @property
        def tag_name(self):
            return self.name

        @property
        def tag_dir(self):
            if self._tag_dir is None:
                self._tag_dir = os.path.join(self.repo_base_url, self.tag_folder, self.tag_name)
            return self._tag_dir

        @property
        def arc_tag_dir(self):
            if self._arc_tag_dir is None:
                self._arc_tag_dir = os.path.join(self.arc_tag_folder, self.tag_name)
            return self._arc_tag_dir

        def tag_path(self, *args, **kwargs):
            return os.path.join(self.tag_dir, self.tag_folder_name(*args, **kwargs))

        def arc_tag_path(self, *args, **kwargs):
            return os.path.join(self.arc_tag_dir, self.tag_folder_name(*args, **kwargs))

        def tag_folder_name(self, *args, **kwargs):
            raise NotImplementedError

        @property
        def tag_folder_pattern(self):
            # type: () -> str
            raise NotImplementedError

        @property
        def tag_path_pattern(self):
            return os.path.join(
                self.tag_folder,
                self.tag_name,
                self.tag_folder_pattern,
            )

    class ChangelogCfg(object):
        def __init__(self, root_cfg, root_url, responsible):
            self._root_cfg = root_cfg
            self.root_url = root_url
            self._responsible = responsible

        @property
        def root_cfg(self):
            return self._root_cfg

        @property
        def ya_make_targets(self):
            # type: () -> List[str]
            """
            Include changes of build paths based on these directories to changelog.

            Also add commits to build paths based on these directories to CI project.
            Aka 'graph discovery' in CI
            """
            return []

        @property
        def observed_paths(self):
            # type: () -> List[str]
            """Include changes of these paths to changelog"""
            return self.dirs

        @property
        def dirs(self):
            # type: () -> List[str]
            """DEPRECATED! Please, use observed_paths or ya_make_targets!"""
            return []

        @property
        def testenv_dbs(self):
            # type: () -> Union[bool, List[str]]
            """
            Testenv dbs to search for diffs.
            :return
                True: use trunk database, specified in Testenv config part
                False: do not use any testenv db for changelogs
                List: means use specified testenv databases
            """
            return True

        @property
        def wiki_page(self):
            raise NotImplementedError

        @property
        def wiki_page_owner(self):
            return self._responsible

        @property
        def review_groups(self):
            """ Review groups to get commits for changelog from """
            return []

        @property
        def banned_authors(self):
            return []

        @staticmethod
        def calculate_importance(changelog_entry):
            """
            Calculate commit importance, which depends on commit paths.
            Default value is the mark of the given `changelog_entry`.
            """
            return changelog_entry.mark

        wiki_head_template = "===={descr} //r:{r1}-{r2}//\nCreation date: {date}"
        markers = None

        wiki_page_ttl_days = None

        add_to_release_notes = False

        use_previous_branch_as_baseline = False

        # rm_const.PermissionType.BANNED: skip revision if all paths from it are included here
        # rm_const.PermissionType.ALLOWED: don't skip revision if any path from it is included here
        # rm_const.PermissionType.CHANGED: don't skip revision and change its importance
        svn_paths_filter = None  # ChangelogPathsFilter(rm_const.PermissionType.ALLOWED, ["arcadia/path"])

        def set_paths_importance(self, path):
            path = "".join(path.partition("arcadia/")[1:])
            if self.svn_paths_filter is None:
                return 1, path
            in_path = any(paths.is_sub_path_of(p, path) for p in self.svn_paths_filter.paths)
            importance = 0
            if self.svn_paths_filter.permission_type == rm_const.PermissionType.ALLOWED:
                if in_path:  # found path in allowed paths: do not skip
                    importance = self.svn_paths_filter.importance
            elif self.svn_paths_filter.permission_type == rm_const.PermissionType.BANNED:
                if not in_path:  # found path not in banned paths: do not skip
                    importance = self.svn_paths_filter.importance
            elif self.svn_paths_filter.permission_type == rm_const.PermissionType.CHANGED:
                if in_path:  # found path in changed paths: set custom importance
                    importance = self.svn_paths_filter.importance
                else:  # default importance
                    importance = 1
            return int(importance), path

        def should_skip_revision(self, r_info):
            if (
                self.svn_paths_filter is None or
                self.svn_paths_filter.permission_type == rm_const.PermissionType.CHANGED
            ):
                return False
            filter_type = self.svn_paths_filter.permission_type
            for _, path in r_info["paths"]:
                # cut first part of path before arcadia folder
                # ex: "/branches/qq/stable-123/arcadia/my_file.py" -> "arcadia/my_file.py"
                path = "".join(path.partition("arcadia/")[1:])
                in_path = any(paths.is_sub_path_of(p, path) for p in self.svn_paths_filter.paths)
                # LOGGER.debug("Path %s is in paths: %s", path, in_path)
                if filter_type == rm_const.PermissionType.ALLOWED:
                    if in_path:  # found path in allowed paths: do not skip
                        return False
                elif filter_type == rm_const.PermissionType.BANNED:
                    if not in_path:  # found path not in banned paths: do not skip
                        return False
            return True

    class Notify(object):

        use_startrek = True

        notifications = None

        class Mail(object):
            def __init__(self, name):
                self.name = name

            mailing_list = ["search-components-releases"]

            def get_start_letter(
                self,
                major_release_num=None,
                revision=None,
                st_issue_key=None,
                changelog_wiki_page=None,
                **kwargs
            ):
                subj = [u"[{}] New release process started".format(self.name)]
                if major_release_num:
                    subj.append(u"Major release num: {}".format(major_release_num))
                content = subj[:]
                if revision:
                    subj.append(u"Revision: {}".format(revision))
                    content.append(u"Revision: {}".format(lb.revision_link(revision)))
                subj = u". ".join(subj)
                content = [
                    u". ".join(content),
                    u"Follow release process in UI: {}".format(lb.rm_ui_link(self.name, link_type=lb.LinkType.href)),
                ]
                if st_issue_key:
                    content.append(u"Ticket: {}".format(lb.st_link(st_issue_key)))
                if changelog_wiki_page:
                    content.append(u"Changelog: {}".format(lb.wiki_page_link(changelog_wiki_page, plain=True)))
                return {
                    "recipients": self.mailing_list,
                    "subject": subj,
                    "body": u"\n".join(content),
                }

        class Telegram(object):
            config = RmTelegramNotifyConfig()
            invite_link = None

            acceptance_delay = {
                rm_const.TaskTypes.LAUNCH_METRICS: 2 * 24 * 60 * 60,
                rm_const.TaskTypes.GENERATE_BETA: 2 * 60 * 60,
            }

            def get_acceptance_delay_message(self, task_id, task_type, exec_time_seconds, custom_msg=''):
                """
                Get acceptance delay message.
                Checks whether there is a delay in the given task and returns a message if any.
                Returns None if either no delay found or the component does not specify any
                delay notification configuration for the given task type.

                :param task_id: SB task id
                :param task_type: SB task type
                :param exec_time_seconds: current timing; it can be the entire execution time or some significant
                stage execution time or whatever makes sense for the task
                :param custom_msg: optional message, will be appended to the default message
                :return: Either a str message or None (meaning no notification is to be sent)
                """
                if not self.acceptance_delay or task_type not in self.acceptance_delay:
                    return
                if exec_time_seconds > self.acceptance_delay[task_type]:
                    return (
                        '{task_type} runs longer than expected (> {exec_time_seconds}). '
                        'You might want to check it up: {task_link}\n{custom_msg}'
                    ).format(
                        task_type=task_type,
                        exec_time_seconds=exec_time_seconds,
                        task_link=lb.task_link(task_id=task_id),
                        custom_msg=custom_msg,
                    )

            @property
            def chats(self):
                """ List of chat names from projects/release_machine/rm_notify.py """
                return []

        class Startrek(object):
            @property
            def assignee(self):
                """ Assignee of startrek ticket on its creation time """
                raise NotImplementedError

            @property
            def assignee_after_acceptance(self):
                """ Assignee of startrek ticket after release acceptance. Defaults to `assignee` """
                return self.assignee

            @property
            def queue(self):
                """ Name of queue in startrek """
                raise NotImplementedError

            @property
            def summary_template(self):
                """ Template of ticket summary """
                raise NotImplementedError

            @property
            def components(self):
                # type: () -> Optional[Text]
                return None

            @property
            def followers(self):
                # type: () -> List[str]
                """ People, who follow startrek ticket """
                return [self.assignee]

            @property
            def ticket_type(self):
                return None

            @property
            def tags(self):
                return ["rm_main_ticket", self.name]

            @property
            def workflow(self):
                # type: () -> Dict[str, str]
                return {}

            @property
            def dev_queue(self):
                """ Development queue. Used for complain tickets """
                return ''

            @property
            def abc_service(self):
                return None

            @property
            def abc_followers(self):
                # type: () -> Optional[Abc]
                """ Filter committers in startrek ticket by their role in abc component """
                return None

            @property
            def merged_revisions_order(self):
                """Stores order of merged revisions"""
                return rm_const.SortOrder.ASCENDING

            @property
            def notify_on_robot_comments_to_tickets(self):
                """Whether to send notifications when robot creates or updates comments in Startrek ticket."""
                return False

            def on_release_close_tickets_by_query(self, st_issue_key=None):
                return None

            def summary(self, release_scope_num):
                return self.summary_template.format(release_scope_num)

            @property
            def time_to_update_assignee(self):
                return False

            @property
            def banned_people(self):
                # type: () -> Set
                return {"ermolovd"}

            @property
            def deadline_date(self):
                # type: () -> Optional[str]
                if self.deadline:
                    return tu.date_ymd(delay_days=-self.deadline)

            def add_nanny_reports(self, where_to_release, release_params, st_issue_key):
                if (
                    self.nanny_reports and
                    where_to_release == rm_const.ReleaseStatus.stable and
                    st_issue_key is not None and
                    nanny_const.STARTREK_TICKET_IDS_KEY not in release_params
                ):
                    release_params[nanny_const.STARTREK_TICKET_IDS_KEY] = [st_issue_key]

            @property
            def notify_on_component_versions_change_to_feature_tickets(self):
                # type: () -> Union[bool, AbstractSet[str]]
                """
                Notify to feature tickets about component_versions_change.

                Used on backend during event ComponentVersionsChange processing.
                If True, notify to all feature tickets.
                If False, do not notify
                If AbstractSet: notify to tickets, which queues are in the set.
                Ex:
                    notify_on_component_versions_change_to_feature_tickets = {"RMDEV"}
                    means that there will be notifications to RMDEV feature tickets only
                """
                return False

            @property
            def notify_on_release_to_release_st_ticket(self):
                # type: () -> Union[bool, AbstractSet[str]]
                """
                Notify to release ticket about release status.

                Used in release task (RELEASE_RM_COMPONENT_2).
                If True: notify about releases to any release stage (stable, testing, ...)
                If False: do not notify
                If AbstractSet: notify about releases, specified in the set.
                Ex:
                    notify_on_release_to_release_st_ticket = frozenset(["stable"])
                    means that there will be notifications only on stable releases
                """
                return True

            @property
            def notify_on_deploy_to_feature_st_tickets(self):
                # type: () -> Union[bool, AbstractSet[str]]
                """
                Notify to feature tickets upon deploy (major)

                Possible values:
                - True - do notify
                - False - do not notify
                - set - notify if deploy stage belong to the set (e.g. frozenset(['stable']) means notify of stable
                deployment only, silent for all other stages)
                :return:
                """
                return False

            use_task_author_as_assignee = True
            hide_commits_under_cut = False
            commit_importance_threshold = 1
            important_changes_limit = 60
            ticket_filter = rm_const.TicketFilter(
                queues={"SOLOMON", "INFRA", "MLTOOLS", "FTB", "FTC", "IGNIETFERRO"},
                tickets={"DEVTOOLS-6598"},
            )
            following_mails = []  # ex: ["release-machine-cc@yandex-team.ru"]

            # Add commit authors as followers in release Startrek ticket
            add_commiters_as_followers = True
            nanny_reports = True
            deadline = 3
            banned_queues = {
                "CROWDFUNDING",
                "IGNIETFERRO",
                "SUBBOTNIK",
                "DEVTOOLSUP",
            }
            close_prev_tickets_stage = rm_const.PipelineStage.release
            close_linked_tickets = False

            # Sometimes users don't want to see one table for all merged commits, they want to see new comment for each
            # merge. In this case write_merges_in_table should be False.
            write_merges_in_table = True

            # Here are some functions to use to construct ticket description
            ticket_description_prefix = None  # custom info to add in front of common description
            ticket_description_suffix = None  # custom info to add at the end of common description

            def __init__(self, name):
                self.name = name

        def __init__(self, name):
            self.mail = self.Mail(name)
            self.tg = self.Telegram()
            self.st = self.Startrek(name) if self.use_startrek else None

    class MetricsCfg(object):
        limit_s = 50400  # 14 hours
        default_launch_id = ""
        run_bisect = False  # when True binary search is run in order to find a commit that influences changes
        # see sandbox.projects.common.search.findurl.FindUrl.compare_searches for more info.

    class Testenv(RmTestenvCfg):
        def __init__(self, name, use_startrek=True, releasable_items=None, release_approvement_required=False):
            self.name = name
            self.JT = rm_const.JobTypes
            self.job_graph = self.JobGraph(
                self.name,
                use_startrek=use_startrek,
                releasable_items=releasable_items,
                release_approvement_required=release_approvement_required,
            )

        class JobGraph(jg.JobGraph):
            def __init__(self, name, use_startrek=True, releasable_items=None, release_approvement_required=False):
                super(ReferenceConfig.Testenv.JobGraph, self).__init__(
                    name,
                    use_startrek=use_startrek,
                    releasable_items=releasable_items,
                    release_approvement_required=release_approvement_required,
                )
                self.graph += self._trunk_part

            @property
            def _trunk_part(self):
                """
                    Part of job graph for Testenv with jobs in trunk, i.e. tests, builds.
                    :return: list with jobs
                """
                return []

        @property
        def trunk_db(self):
            return self.name + "-trunk"

    class CI(object):
        FILTER_ARROW_TYPES = frozenset([
            rm_const.JobTypes.NEW_BRANCH,
            rm_const.JobTypes.CLONE_DB,
        ])

        class JobGraph(jg.JobGraph):
            release_auto = False

        def __init__(
            self,
            root_cfg,
            name,
            testenv_cfg=None,
            use_startrek=True,
            releasable_items=None,
            release_approvement_required=False,
        ):
            self._root_cfg = root_cfg
            self.name = name
            self._use_startrek = use_startrek
            if testenv_cfg:
                self.job_graph = testenv_cfg.job_graph
            else:
                self.job_graph = self.JobGraph(
                    self.name,
                    use_startrek=use_startrek,
                    releasable_items=releasable_items,
                    release_approvement_required=release_approvement_required,
                )

        @property
        def root_cfg(self):
            return self._root_cfg

        @property
        def secret(self):
            raise NotImplementedError

        @property
        def sb_owner_group(self):
            raise NotImplementedError

        @staticmethod
        def _filter_jobs(jobs, job_types):
            # type: (Iterable[jge.JobGraphElement], AbstractSet[str]) -> Generator[jge.JobGraphElement]
            for job in jobs:
                if job.job_type in job_types:
                    yield job

        @property
        def te_to_ci_jobs(self):
            convertible_types = frozenset([
                rm_const.JobTypes.BUILD,
                rm_const.JobTypes.RELEASE,
                rm_const.JobTypes.LAUNCH_METRICS,
            ])
            return list(self._filter_jobs(self.job_graph.graph, convertible_types))

        @property
        def ci_only_jobs(self):
            jobs = [
                jg_ci.JobGraphElementNewTagCI(),
                jg_ci.JobGraphElementChangelogCI(),
            ]
            if self._use_startrek:
                jobs.extend([
                    jg_ci.JobGraphElementStartrekCI(),
                    jg_ci.JobGraphElementLinkStartrekTicketsCI(),
                    jg_ci.JobGraphElementChangelogFormatCI(),
                    jg_ci.JobGraphElementPostChangelogToStartrekCI(),
                ])
            return jobs

        def get_needs(self, job):
            # type: (jge.JobGraphElement) -> List[str]
            needs = set()
            for arrow in job.job_arrows:
                if not isinstance(arrow, jg_arrows.JobTrigger):
                    continue
                if arrow.job_type in self.FILTER_ARROW_TYPES:
                    continue
                needs.add(arrow.this_job_name(self.name))
            if not needs and job.job_type != rm_const.JobTypes.NEW_TAG:
                # there should be only one entry point: CI-1860
                needs.add(rm_const.JobTypes.rm_job_name(rm_const.JobTypes.NEW_TAG, self.name))
            return [jg_utils.job_name_for_ci(i) for i in sorted(needs)]

        @property
        def ya_make_abs_paths_glob(self):
            # type: () -> List[str]
            """
            Add commits to build paths based on these directories (in glob format) to CI project.
            Aka 'graph discovery' in CI
            """
            return []

        @property
        def observed_sub_paths(self):
            # type: () -> List[str]
            """
            Add commits to these paths only to CI project.
            Aka 'sub-paths' in CI
            """
            return []

        @property
        def grant_config_update_permissions_to_release_machine_robot(self):  # type: () -> bool
            """
            The `True` value allows RM's robot robot-srch-releaser to update the component's a.yaml with
            no token re-delegation requirement. If set to `False` each robot's config update would require a
            confirmation from one of the members of the respective ABC service (CI token request)
            """
            return True

        @property
        def config_edit_approvals(self):
            """https://docs.yandex-team.ru/ci/permissions#config-edit-approvals"""

            if not self.root_cfg.responsible.abc.service_name:
                return []

            result = [
                responsibility.ABCSelection(service=self.root_cfg.responsible.abc.service_name),
            ]

            if (
                self.grant_config_update_permissions_to_release_machine_robot and
                self.root_cfg.responsible.abc.service_name != rm_const.RM_ABC_SERVICE_NAME
            ):
                result.append(
                    responsibility.ABCSelection(
                        service=rm_const.RM_ABC_SERVICE_NAME,
                        scope=rm_const.ABC_SCOPE_VIRTUAL,
                    ),
                )

            return result

    @property
    def name(self):
        """ Main identifier of the component. Choose it wisely! """
        raise NotImplementedError

    @property
    def responsible(self):
        """Namedtuple with abc group and login of person who is responsible for releases of the component
            or string with person, who is responsible for releases of the component"""
        raise NotImplementedError

    @property
    def release_cycle_type(self):
        raise NotImplementedError

    def link_to_ui(self, major_release_num, minor_release_num, link_name, link_type):
        return lb.rm_ui_link(
            self.name,
            link_name=link_name or "{} RM UI".format(self.display_name),
            link_type=link_type
        )

    @property
    def display_name(self):
        """ Name of the component (in English) to display in UI """
        if six.PY2:
            name = self.name.decode("utf-8")
        else:
            name = self.name
        return u" ".join(p.capitalize() for p in name.split("_"))

    @property
    def is_branched(self):
        """Return True if config is branched"""
        return False

    @property
    def is_branched_or_ci(self):
        """True if component is either branched or CI-driven"""
        return False

    @property
    def is_ci(self):
        return False

    @property
    def is_tagged(self):
        """Return True if config is tagged"""
        return False

    @property
    def is_trunk(self):
        """Return True if config is trunk"""
        return False

    def release_id(self, major_release_num=None, minor_release_num=None):

        if self.is_ci:
            release_id = [major_release_num or 0, minor_release_num or 0]
        else:
            release_id = filter(bool, [major_release_num, minor_release_num])

        return "-".join(map(str, release_id))

    def __init__(self):
        self.notify_cfg = self.Notify(self.name)
        self.releases_cfg = self.Releases(self, self.name, self.responsible)
        self.testenv_cfg = self.Testenv(
            self.name,
            use_startrek=self.notify_cfg.use_startrek,
            releasable_items=self.releases_cfg.releasable_items,
            release_approvement_required=self.releases_cfg.approvement_required,
        )
        self.svn_cfg = self.SvnCfg(self.name)
        self.release_viewer_cfg = self.ReleaseViewer()
        self.changelog_cfg = None
        self.metrics_cfg = None
        self.yappy_cfg = None
        self.ci_cfg = self.CI(
            self,
            self.name, self.testenv_cfg,
            use_startrek=self.notify_cfg.use_startrek,
            releasable_items=self.releases_cfg.releasable_items,
            release_approvement_required=self.releases_cfg.approvement_required,
        )
