import logging
import os
import re

from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import decorators
from sandbox.projects.release_machine.components.job_graph.stages import pre_release_stage as jg_prerelease
from sandbox.projects.release_machine.components.job_graph.stages import release_stage as jg_release
from sandbox.projects.release_machine.components.config_core import yappy
from sandbox.projects.release_machine.components.config_core.bases import base
from sandbox.projects.release_machine.core import const as rm_const


LOGGER = logging.getLogger(__name__)


class ReferenceBranchedConfig(base.ReferenceConfig):
    release_cycle_type = rm_const.ReleaseCycleType.BRANCH

    class MergesCfg(object):
        def __init__(self, name):
            self.name = name

        # Merge only into not released branches (RMDEV-283)
        never_merge_to_released_branches = False
        # Uses only in MergeToStable task
        number_last_branches_to_merge = 2

        # Used by ArcMergesCrawler
        @property
        def lts_branches(self):
            """
                A list of long-term support branches that require arc->svn sync
                in addition to the latest branches
            """
            return []

        permissions = base.MergePermissions(
            permission_type=rm_const.PermissionType.BANNED,
            people_groups=None,  # PeopleGroups("staff_groups", "abc_services", "logins"),
        )

        def not_granted_message(self, author, responsible):
            return "Merger {} is not allowed to merge into {}. Please, contact {} to get merge permission".format(
                author, self.name, responsible,
            )

    class SvnCfg(base.ReferenceConfig.SvnCfg):
        """
            Path to branches: /[REPO_NAME]/[branches_folder]/[branch_name]/[branch_folder_name]
            Path to tags: /[REPO_NAME]/[tag_folder]/[tag_name]/[tag_folder_name]
        """
        # means, no releases from any earlier branch
        max_active_branches = 2  # SEARCH-6016
        # overwrite `merge_to_old_branches` flag in MergeToStable when use mergeto marker (RMDEV-1737)
        merge_to_old_branches = False
        # Launch pre release process if new release was deployed from the last branch
        allow_autobranches = False
        branches_folder = "branches"
        # for given folder RM will create branch: releases/experimental/<component_name>/stable-NN
        # don't forget to grant robot-srch-releaser "Create" and "Fast-forward push" roles on ARC Vcs in IDM
        arc_branches_folder = "releases"
        branch_prefix = "stable"
        tag_prefix = branch_prefix

        branch_folder_template = "{branch_prefix}-{branch_num}"
        tag_folder_template = "{tag_prefix}-{branch_num}-{tag_num}"

        def __init__(self, name):
            super(ReferenceBranchedConfig.SvnCfg, self).__init__(name)
            self._branch_dir = None
            self._arc_branch_dir = None

        @property
        def branch_dir(self):
            if self._branch_dir is None:
                self._branch_dir = os.path.join(self.repo_base_url, self.branches_folder, self.branch_name)
            return self._branch_dir

        @property
        def arc_branch_dir(self):
            if self._arc_branch_dir is None:
                self._arc_branch_dir = os.path.join(self.arc_branches_folder, self.branch_name)
            return self._arc_branch_dir

        @property
        def branch_name(self):
            return self.name

        def branch_folder_name(self, branch_num):
            """ Example: stable-111 """
            return self.branch_folder_template.format(
                branch_prefix=self.branch_prefix,
                branch_num=branch_num,
            )

        def branch_path(self, branch_num):
            return os.path.join(self.branch_dir, self.branch_folder_name(branch_num))

        def arc_branch_path(self, branch_num):
            return os.path.join(self.arc_branch_dir, self.branch_folder_name(branch_num))

        @property
        def branch_folder_pattern(self):
            return self.branch_folder_template.format(
                branch_prefix=self.branch_prefix,
                branch_num="([0-9]+)",
            )

        @property
        def tag_folder_pattern(self):
            return self.tag_folder_template.format(
                tag_prefix=self.tag_prefix,
                branch_num="([0-9]+)",
                tag_num="([0-9]+)",
            )

        @property
        def branch_path_pattern(self):
            return os.path.join(
                self.branches_folder,
                self.branch_name,
                self.branch_folder_pattern,
            )

        @property
        def arc_branch_path_pattern(self):
            return os.path.join(
                self.arc_branches_folder,
                self.branch_name,
                self.branch_folder_pattern,
            )

        def tag_folder_name(self, branch_num, tag_num):
            """ Example: stable-111-2 """
            return self.tag_folder_template.format(
                tag_prefix=self.tag_prefix,
                branch_num=branch_num,
                tag_num=tag_num,
            )

        def _get_major_release_num(self, url):
            """
            Get major release number from arcadia url for branch components.
            Major release number for branch components is branch number.
            :param url: str, where we search major_num with branch_path_pattern
            :return major_num if we found it, else 0
            """
            pattern = self.branch_path_pattern
            patterns = [r".*/{}/arcadia".format(pattern), r".*/{}/?$".format(pattern)]
            for p in patterns:
                # LOGGER.debug("Try to use pattern: %s", p)
                found = re.search(p, url)
                if found:
                    # LOGGER.debug("FOUND: %s", found.group(0))
                    return int(found.group(1))
            return 0

        def _get_release_numbers(self, regexp_result):
            """
                Get release numbers from arcadia url
                :return (branch_num, tag_num) if we found both of them, else None
            """
            return regexp_result.group(1, 2)

        @decorators.memoized_property
        def _release_numbers_regexps(self):
            tag_pattern = self.tag_path_pattern
            branch_pattern = self.branch_path_pattern
            patterns = [
                r".*/{}/arcadia".format(tag_pattern),
                r".*/{}/?$".format(tag_pattern),
                r".*/{}/arcadia@([0-9]+)".format(branch_pattern),
                r".*/{}@([0-9]+)".format(branch_pattern),
            ]
            return [re.compile(i) for i in patterns]

    class Releases(base.ReferenceConfig.Releases):
        pass

    class Yappy(yappy.YappyBaseCfg):
        pass

    class Testenv(base.ReferenceConfig.Testenv):
        def __init__(
            self,
            name,
            use_startrek=True,
            use_arc=False,
            releasable_items=None,
            release_approvement_required=False,
        ):
            super(ReferenceBranchedConfig.Testenv, self).__init__(
                name, use_startrek=use_startrek, releasable_items=releasable_items,
            )
            self._use_arc = use_arc
            self.job_patch = self.JobPatch(self.name)
            self.job_graph = self.JobGraph(
                self.name,
                use_startrek=use_startrek,
                use_arc=use_arc,
                releasable_items=releasable_items,
                release_approvement_required=release_approvement_required,
            )
            self._branch_db_regex = None

        class JobGraph(base.ReferenceConfig.Testenv.JobGraph):
            def __init__(
                self,
                name,
                use_startrek=True,
                use_arc=False,
                releasable_items=None,
                release_approvement_required=False,
            ):
                super(ReferenceBranchedConfig.Testenv.JobGraph, self).__init__(
                    name,
                    use_startrek=use_startrek,
                    releasable_items=releasable_items,
                    release_approvement_required=release_approvement_required,
                )
                self._use_arc = use_arc
                self.graph += self._prerelease + self._branch_part

            @property
            def new_tag_should_wait_for_release_ticket(self):
                return False

            @property
            def _prerelease(self):
                """
                    Prerelease part of job graph for Testenv with arrows and dependencies
                    :return: list with jobs
                """
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementWiki(),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrekWiki(use_arc=self._use_arc),
                ]

            @property
            def _branch_part(self):
                """
                    Part of job graph for Testenv with jobs in branch, i.e. tests, builds.
                    :return: list with jobs
                """
                if self._use_startrek:
                    return [
                        jg_release.JobGraphElementLogMerge(),
                    ]

                return []

            @property
            def _release(self):

                result = [
                    jg_release.JobGraphElementNewTag(
                        wait_for_release_ticket=self.new_tag_should_wait_for_release_ticket,
                    ),
                    jg_release.JobGraphElementChangelogFinal(),
                ]

                if self._release_approvement_required:

                    result.append(
                        jg_release.JobGraphElementStartrekOkApprovement(),
                    )

                return result

        class JobPatch(object):
            def __init__(self, name):
                # type: (str) -> None
                self.name = name.upper()

            @property
            def change_frequency(self):
                freq = rm_const.TestFrequencies
                return {
                    "_{}_DAILY_PRIEMKA".format(self.name): freq.ONCE_A_DAY_TEST,
                    "_{}_UMBRELLA_ACCEPT_BY_MARKER".format(self.name): freq.EACH_REV_TEST,
                    "_{}_UMBRELLA_ACCEPT_SCHEDULED".format(self.name): freq.ONCE_A_DAY_TEST,
                    "_BUILD_RELEASE_{}".format(self.name): freq.EACH_REV_TEST,
                    "_BUILD_RELEASE_{}SEARCH".format(self.name): freq.EACH_REV_TEST,
                    rm_const.JobTypes.rm_job_name(rm_const.JobTypes.LOG_MERGE, self.name): freq.EACH_REV_TEST,
                    rm_const.JobTypes.rm_job_name(rm_const.JobTypes.NEW_TAG, self.name): freq.EACH_REV_TEST,
                    rm_const.JobTypes.rm_job_name(rm_const.JobTypes.BUILD, self.name): freq.EACH_REV_TEST,
                }

            @property
            def deactivate(self):
                return []

            @property
            def ignore_match(self):
                """ TODO: move some of these names out of default match """
                return [
                    "_{}_CHANGELOG".format(self.name),
                    # new standard jobs
                    rm_const.JobTypes.rm_job_name(rm_const.JobTypes.NEW_BRANCH, self.name),
                    rm_const.JobTypes.rm_job_name(rm_const.JobTypes.CLONE_DB, self.name),
                    rm_const.JobTypes.rm_job_name(rm_const.JobTypes.CHANGELOG, self.name),
                    rm_const.JobTypes.rm_job_name(rm_const.JobTypes.CHANGELOG_MAJOR, self.name),
                    rm_const.JobTypes.rm_job_name(rm_const.JobTypes.STARTREK, self.name),
                    rm_const.JobTypes.rm_job_name(rm_const.JobTypes.WIKI, self.name),
                    rm_const.JobTypes.rm_job_name(rm_const.JobTypes.ACTION_PRE_RELEASE, self.name),
                ]

            @property
            def ignore_prefix(self):
                """ These jobs MUST be ignored for all 'branched' components """
                return [
                    "_{}__".format(rm_const.JobTypes.NEW_BRANCH),
                    "_{}__".format(rm_const.JobTypes.CLONE_DB),
                    "_{}__".format(rm_const.JobTypes.CHANGELOG_MAJOR),
                    "_{}__".format(rm_const.JobTypes.STARTREK),
                    "_{}__".format(rm_const.JobTypes.WIKI),
                    "_{}__".format(rm_const.JobTypes.ACTION_PRE_RELEASE),
                ]

        last_good_revision_ignore_release_jobs = False  # ??? TODO: Understand what is it and rename it

        @property
        def branch_task_owner(self):
            """ Your team sandbox-group for branch databases """
            return self.trunk_task_owner

        @property
        def branch_db_template(self):
            """ Template for cloning db """
            return self.name + "-{testenv_db_num}"

        @property
        def db_template(self):
            return self.branch_db_template

        @property
        def branch_db_regex(self):
            """
                Regex for getting all branch databases for this component. This field is used in auto-add tests in dbs.
            """
            if self._branch_db_regex is None:
                self._branch_db_regex = re.compile(self.branch_db_template.format(testenv_db_num="([0-9]+)"))
            return self._branch_db_regex

        @property
        def merge_on_clone(self):
            """Databases to merge with main database for current component"""
            return None

        @staticmethod
        def db_stop_range(branch_number):
            # type: (int) -> Iterable[int]
            return range(max([branch_number - 4, 1]), max([branch_number - 2, 1]))

        @staticmethod
        def db_drop_range(branch_number):
            # type: (int) -> Iterable[int]
            if branch_number <= 7:
                drop_from = 1
            else:
                drop_from = branch_number - 7
            return range(drop_from, branch_number - 5)

    @property
    def is_branched(self):
        """Return True if config is branched"""
        return True

    @property
    def is_branched_or_ci(self):
        return True

    def link_to_ui(
        self,
        major_release_num=None,
        minor_release_num=None,
        link_name=None,
        link_type=lb.LinkType.plain
    ):
        screen = "manage"
        specific_screen = [
            "{}={}".format(i, j)
            for i, j in (("branch", major_release_num), ("tag", minor_release_num)) if j
        ]
        if specific_screen:
            screen = "{}?{}".format(screen, "&".join(specific_screen))

        return lb.rm_ui_link(
            self.name,
            screen=screen,
            link_name=link_name or "{} RM UI".format(self.display_name),
            link_type=link_type
        )

    def __init__(self):
        self.releases_cfg = self.Releases(self, self.name, self.responsible)
        self.notify_cfg = self.Notify(self.name)
        self.svn_cfg = self.SvnCfg(self.name)
        self.testenv_cfg = self.Testenv(
            self.name,
            use_startrek=self.notify_cfg.use_startrek,
            use_arc=self.svn_cfg.use_arc,
            releasable_items=self.releases_cfg.releasable_items,
            release_approvement_required=self.releases_cfg.approvement_required,
        )
        self.release_viewer_cfg = self.ReleaseViewer()
        self.changelog_cfg = self.ChangelogCfg(self, self.svn_cfg.main_url, self.responsible)
        self.yappy_cfg = self.Yappy()
        self.metrics_cfg = self.MetricsCfg()
        self.merges_cfg = self.MergesCfg(self.name)
        self.ci_cfg = self.CI(
            self,
            self.name, self.testenv_cfg,
            use_startrek=self.notify_cfg.use_startrek,
            releasable_items=self.releases_cfg.releasable_items,
            release_approvement_required=self.releases_cfg.approvement_required,
        )
