import os

from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import decorators
from sandbox.projects.release_machine.components.config_core import yappy
from sandbox.projects.release_machine.components.config_core.bases import base
from sandbox.projects.release_machine.core import const as rm_const


class ReferenceCIConfig(base.ReferenceConfig):
    release_cycle_type = rm_const.ReleaseCycleType.CI
    testenv_cfg = None  # Just to fix test. Get rid of it after moving from TE completely
    first_tag_num = 0

    @property
    def is_branched_or_ci(self):
        return True

    @property
    def is_ci(self):
        return True

    @property
    def robot(self):
        return rm_const.ROBOT_RELEASER_USER_NAME

    @property
    def uses_newstyle_jg(self):
        return self._uses_newstyle_jg

    def link_to_ui(
        self,
        major_release_num=None,
        minor_release_num=None,
        link_name=None,
        link_type=lb.LinkType.plain
    ):
        screen = "manage"
        specific_screen = [
            "{}={}".format(i, j)
            for i, j in (("branch", major_release_num), ("tag", minor_release_num)) if j
        ]
        if specific_screen:
            screen = "{}?{}".format(screen, "&".join(specific_screen))

        return lb.rm_ui_link(
            self.name,
            screen=screen,
            link_name=link_name or "{} RM UI".format(self.display_name),
            link_type=link_type
        )

    class MergesCfg(object):
        def __init__(self, name):
            self.name = name

        never_merge_to_released_branches = False  # Merge only into not released branches (RMDEV-283)
        number_last_branches_to_merge = 2  # Used only in MergeToStable task
        permissions = base.MergePermissions(
            permission_type=rm_const.PermissionType.BANNED,
            people_groups=None,  # PeopleGroups("staff_groups", "abc_services", "logins"),
        )

        def not_granted_message(self, author, responsible):
            return "Merger {} is not allowed to merge into {}. Please, contact {} to get merge permission".format(
                author, self.name, responsible,
            )

    class ChangelogCfg(base.ReferenceConfig.ChangelogCfg):
        testenv_dbs = False
        wiki_page = None

    class SvnCfg(base.ReferenceConfig.SvnCfg):
        """
        Path to branches: /[REPO_NAME]/[branches_folder]/[branch_name]/[branch_folder_name]
        Path to tags: /[REPO_NAME]/[tag_folder]/[tag_name]/[tag_folder_name]
        """
        # means, no releases from any earlier branch
        use_arc = True
        start_version = None  # CI-1815
        max_active_branches = 2  # SEARCH-6016
        # overwrite `merge_to_old_branches` flag in MergeToStable when use mergeto marker (RMDEV-1737)
        merge_to_old_branches = False
        # Launch pre release process if new release was deployed from the last branch
        allow_autobranches = False
        branches_folder = "branches"
        # for given folder RM will create branch: releases/<component_name>/stable-NN
        # don't forget to grant robot-srch-releaser "Create" and "Fast-forward push" roles on ARC Vcs in IDM
        arc_branches_folder = "releases"
        branch_prefix = "stable"
        tag_prefix = branch_prefix

        branch_folder_template = "{branch_prefix}-{branch_num}"
        tag_folder_template = "{tag_prefix}-{branch_num}-{tag_num}"
        tag_folder_pattern = ""  # this property is not relevant here, so, mock it

        @decorators.memoized_property
        def branch_dir(self):
            return os.path.join(self.repo_base_url, self.branches_folder, self.branch_name)

        @decorators.memoized_property
        def arc_branch_dir(self):
            return os.path.join(self.arc_branches_folder, self.branch_name)

        def branch_path(self, branch_num):
            return os.path.join(self.branch_dir, self.branch_folder_name(branch_num))

        def arc_branch_path(self, branch_num):
            return os.path.join(self.arc_branch_dir, self.branch_folder_name(branch_num))

        @property
        def branch_name(self):
            return self.name

        def branch_folder_name(self, branch_num):
            """Example: stable-111."""
            return self.branch_folder_template.format(
                branch_prefix=self.branch_prefix,
                branch_num=branch_num
            )

        def tag_folder_name(self, branch_num, tag_num):
            """Example: stable-111-2."""
            return self.tag_folder_template.format(
                tag_prefix=self.tag_prefix,
                branch_num=branch_num,
                tag_num=tag_num,
            )

        @property
        def branch_folder_pattern(self):
            return self.branch_folder_template.format(
                branch_prefix=self.branch_prefix,
                branch_num="([0-9]+)",
            )

        @property
        def tag_folder_pattern(self):
            return self.tag_folder_template.format(
                tag_prefix=self.tag_prefix,
                branch_num="([0-9]+)",
                tag_num="([0-9]+)",
            )

    class Releases(base.ReferenceConfig.Releases):
        allow_robots_to_release_stable = True
        main_release_flow_independent_stages = False  # https://docs.yandex-team.ru/ci/release#independent-stages
        main_release_flow_branch_auto_start = True  # https://docs.yandex-team.ru/ci/release#branches-auto-release

    class Yappy(yappy.YappyBaseCfg):
        pass

    def __init__(self):
        self.releases_cfg = self.Releases(self, self.name, self.responsible)
        self.notify_cfg = self.Notify(self.name)
        self.svn_cfg = self.SvnCfg(self.name)
        self.release_viewer_cfg = self.ReleaseViewer()
        self.changelog_cfg = self.ChangelogCfg(self, self.svn_cfg.main_url, self.responsible)
        self.yappy_cfg = self.Yappy()
        self.metrics_cfg = self.MetricsCfg()
        self.merges_cfg = self.MergesCfg(self.name)
        self.ci_cfg = self.CI(
            self,
            self.name,
            use_startrek=self.notify_cfg.use_startrek,
            releasable_items=self.releases_cfg.releasable_items,
            release_approvement_required=self.releases_cfg.approvement_required,
        )

        if hasattr(self, "JG"):
            self.jg = self.JG(root_cfg=self)
            self._uses_newstyle_jg = True
        else:
            self._uses_newstyle_jg = False
