import logging
import re

from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import decorators
from sandbox.projects.release_machine.components.config_core import yappy
from sandbox.projects.release_machine.components.config_core.bases import base
from sandbox.projects.release_machine.core import const as rm_const


LOGGER = logging.getLogger(__name__)


class ReferenceTaggedConfig(base.ReferenceConfig):
    """ For small components, with releases from tags without branches"""
    release_cycle_type = rm_const.ReleaseCycleType.TAG

    class SvnCfg(base.ReferenceConfig.SvnCfg):
        tag_prefix = 'r'
        tag_folder_template = "{tag_prefix}{tag_num}"

        @property
        def tag_folder_pattern(self):
            return self.tag_folder_template.format(
                tag_prefix=self.tag_prefix,
                tag_num="([0-9]+)",
            )

        def tag_folder_name(self, tag_num):
            """ Example: stable-111-2 """
            return self.tag_folder_template.format(
                tag_prefix=self.tag_prefix,
                tag_num=tag_num,
            )

        def _get_major_release_num(self, url):
            """
            Get major release number from arcadia url for tagged components.
            Major release number for tagged components is tag number.
            :param url: str, where we search major_num with tag_path_pattern
            :return major_num if we found it, else 0
            """
            pattern = self.tag_path_pattern
            patterns = [r".*/{}/arcadia".format(pattern), r".*/{}/?$".format(pattern)]
            for p in patterns:
                # LOGGER.debug("Try to use pattern: %s", p)
                found = re.search(p, url)
                if found:
                    return int(found.group(1))
            return 0

        @decorators.memoized_property
        def _release_numbers_regexps(self):
            tag_pattern = self.tag_path_pattern
            patterns = [
                r".*/{}/arcadia".format(tag_pattern),
                r".*/{}/?$".format(tag_pattern),
                r"@([0-9]+)",
            ]
            return [re.compile(i) for i in patterns]

    class Testenv(base.ReferenceConfig.Testenv):
        @property
        def db_template(self):
            return self.trunk_db

        class JobGraph(base.ReferenceConfig.Testenv.JobGraph):
            def __init__(self, name, use_startrek=True, releasable_items=None, release_approvement_required=False):
                super(ReferenceTaggedConfig.Testenv.JobGraph, self).__init__(
                    name,
                    use_startrek=use_startrek,
                    releasable_items=releasable_items,
                    release_approvement_required=release_approvement_required,
                )
                self.graph += self._tag_part

            @property
            def _release(self):
                return [
                    # JobGraphElementNewTagTagged(), todo: add after all tagged move to job_graph
                ]

            @property
            def _tag_part(self):
                """
                    Part of job graph for Testenv with jobs in tag, i.e. tests, builds.
                    :return: list with jobs
                """
                return []

    class Yappy(yappy.YappyBaseCfg):
        pass

    @property
    def is_tagged(self):
        """Return True if config is tagged"""
        return True

    def link_to_ui(
        self,
        major_release_num=None,
        minor_release_num=None,
        link_name=None,
        link_type=lb.LinkType.plain
    ):
        screen = "manage" if not major_release_num else "manage?tag={}".format(major_release_num)

        return lb.rm_ui_link(
            self.name,
            screen=screen,
            link_name=link_name or "{} RM UI".format(self.display_name),
            link_type=link_type
        )
