import inspect
import six

from sandbox.projects.release_machine.helpers import responsibility_helper
from sandbox.projects.release_machine.core import const as rm_const


def get_ci_action_list(cfg, component_config_pb2, main_release_action_name):

    if not hasattr(cfg, "jg"):

        # old-style graph (not a JGv2)

        return [
            component_config_pb2.RmCiAction(
                id=main_release_action_name,
                type="release",
                flow_id="{}-release-flow".format(cfg.name),
            ),
        ]

    result = []

    for action_type in cfg.jg.ci_actions:

        for action_id, action_data in six.iteritems(cfg.jg.ci_actions[action_type]):

            result.append(
                component_config_pb2.RmCiAction(
                    id=action_id,
                    type=action_type,
                    flow_id=cfg.jg.build_flow_name(action_id),
                ),
            )

    return result


def config2proto(cfg, token=None):
    """
    Give a config `cfg` construct the respective component_config_pb2.RmComponentConfigSerialized protobuf message

    :type cfg: sandbox.projects.release_machine.components.configs.ReferenceConfig
    :param cfg: RM config object
    :return: component_config_pb2.RmComponentConfigSerialized message
    """
    from release_machine.common_proto import component_config_pb2

    if hasattr(cfg, "jg"):
        main_release_action_name = cfg.jg.main_release_action_name
    else:
        main_release_action_name = "release_{}".format(cfg.name)

    return component_config_pb2.RmComponentConfigSerialized(
        name=cfg.name,
        display_name=cfg.display_name,
        responsible_user_login=responsibility_helper.get_responsible_user_login(cfg.responsible, token=token),
        responsible_abc_service_name=responsibility_helper.get_responsible_abc_service_name(
            cfg.responsible,
            token=token,
        ),
        release_cycle_type=rm_const.ReleaseCycleType.to_str(cfg.release_cycle_type),
        config_path=inspect.getfile(cfg.__class__),
        kpi=component_config_pb2.RmKpiSettings(
            kpi_alert=cfg.release_viewer_cfg.kpi_alert,
            kpi_alert_skip_weekend=cfg.release_viewer_cfg.kpi_alert_skip_weekends,
        ),
        statistics_page=component_config_pb2.RmStatisticsPageSettings(
            charts=cfg.release_viewer_cfg.statistics_page_charts,
        ),
        testenv=component_config_pb2.RmTestenvSettings(
            trunk_db=cfg.testenv_cfg.trunk_db,
            trunk_task_owner=cfg.testenv_cfg.trunk_task_owner,
        ) if hasattr(cfg, "testenv_cfg") and cfg.testenv_cfg is not None else component_config_pb2.RmTestenvSettings(),
        block_release=component_config_pb2.RmBlockReleaseSettings(
            settings=[block_setting.to_protobuf() for block_setting in cfg.releases_cfg.block_on_test_results],
            look_behind=cfg.releases_cfg.block_on_test_results_look_behind,
        ),
        job_timeline=component_config_pb2.RmJobTimelineSettings(
            filters=(cfg.release_viewer_cfg.job_timeline_filters or []),
        ),
        vcs=component_config_pb2.RmVcsSettings(
            use_arc=cfg.svn_cfg.use_arc,
            repo_base_url=cfg.svn_cfg.repo_base_url,
            trunk_url=cfg.svn_cfg.trunk_url,
            main_url=cfg.svn_cfg.main_url,
        ),
        a_yaml_dir=cfg.ci_cfg.a_yaml_dir if cfg.release_cycle_type == rm_const.ReleaseCycleType.CI else None,
        resources_info=component_config_pb2.RmResourceInfoSettings(
            resources_info=[
                component_config_pb2.RmResourceInfo(
                    name=ri.name,
                    resource_type=ri.resource_type,
                    resource_name=ri.resource_name,
                    build_ctx_key=ri.build_ctx_key,
                ) for ri in cfg.releases_cfg.resources_info
            ],
        ),
        main_release_action_name=main_release_action_name,
        ci_actions=get_ci_action_list(cfg, component_config_pb2, main_release_action_name),
    )
