import six
import collections
import logging
import typing  # noqa

from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.components.config_core.jg.graph import base as graph_base
from sandbox.projects.release_machine.components.config_core.jg.cube import base as cube_base
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import internal as internal_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import dummy as dummy_cubes
from sandbox.projects.release_machine.components.config_core.jg import flow


LOGGER = logging.getLogger(__name__)


class JGCfgMeta(type):

    def __new__(mcs, name, bases, namespace):
        cls = type.__new__(mcs, name, bases, namespace)

        cls.__flow_funcs = set()

        for name in dir(cls):

            cls_attr = getattr(cls, name)

            if not hasattr(cls_attr, "__defines_flow"):
                continue

            cls.__flow_funcs.add(name)

        return cls

    @property
    def flow_funcs(cls):
        return cls.__flow_funcs


class JGCfg(six.with_metaclass(JGCfgMeta, object)):

    def __init__(self, root_cfg=None):
        self._root_cfg = root_cfg
        self._ci_actions = collections.defaultdict(dict)  # type: typing.Dict[typing.Dict[flow.CiActionData]]
        self._build_ci_actions()

    def _build_ci_actions(self):
        for name in self.__class__.flow_funcs:
            flow_builder = getattr(self, name)
            flow_builder.register(self)

    def build_flow_name(self, action_name):
        return "{}-flow".format(action_name)

    @property
    def root_cfg(self):
        return self._root_cfg

    @property
    def component_name(self):
        return self._root_cfg.name

    @property
    def ci_actions(self):
        return self._ci_actions

    @property
    def release_actions(self):  # type: () -> typing.Dict[flow.CiActionData]
        return self.ci_actions[rm_const.CIActionKind.RELEASE]

    @property
    def non_release_actions(self):  # type: () -> typing.Dict[flow.CiActionData]
        return self.ci_actions[rm_const.CIActionKind.ACTION]

    @property
    def all_actions(self):  # type: () -> typing.Dict[flow.CiActionData]
        result = {}
        result.update(self.release_actions)
        result.update(self.non_release_actions)
        return result

    @property
    def main_release_action_name(self):
        main_release_action_name = "release_{}".format(self.component_name)

        release_action_key_list = list(self.release_actions.keys())

        if main_release_action_name not in release_action_key_list:
            main_release_action_name = release_action_key_list[0] if release_action_key_list else ""

        return main_release_action_name


class BaseReleaseMachineJobGraph(JGCfg):
    """
        new tag --> main_graph_entry --+----> changelog
                                       |          |                   [?]
                                       |          +--------> post changelog to release ticket
                                       |          |                   [?]
                                       |          +--------> create wiki page with changelog
                                       |          | [?]                         [?]
                                       +----> create st ticket -----> link feature tickets
    """

    def _add_startrek_jobs_if_required(self, graph, entry_cube_name=dummy_cubes.RMMainGraphEntry.NAME):

        if not self.root_cfg.notify_cfg.use_startrek:
            return []

        create_ticket = internal_cubes.CreateStartrekTicket(
            self.component_name,
            needs=[graph.get(entry_cube_name)],
        )

        link_tickets = internal_cubes.LinkFeatureTickets(
            self.component_name,
            input=cube_base.CubeInput(
                config={
                    "ticket_key": create_ticket.output.st_ticket.key,
                    "changelog_resource": graph.get(
                        internal_cubes.CreateChangelog.NAME,
                    ).output.resources["RELEASE_MACHINE_CHANGELOG"].first(),
                },
            ),
        )

        format_changelog = internal_cubes.FormatChangelog(
            self.component_name,
            input=cube_base.CubeInput(
                config={
                    "changelog_resource": graph.get(
                        internal_cubes.CreateChangelog.NAME,
                    ).output.resources["RELEASE_MACHINE_CHANGELOG"].first(),
                },
            ),
        )

        post_cl_to_startrek = internal_cubes.PostChangelogToStartrek(
            self.component_name,
            input=cube_base.CubeInput(
                comment_marker=rm_const.CHANGELOG_MARKER,
                place_comment="ticket_description",
                issue_key=create_ticket.output.st_ticket.key,
                comment_text=format_changelog.output.formatted_changelog.wiki,
            ),
        )

        graph.add(create_ticket)
        graph.add(link_tickets)
        graph.add(format_changelog)
        graph.add(post_cl_to_startrek)

    def _add_wiki_changelog_job_if_required(self, graph):

        if not self.root_cfg.changelog_cfg.wiki_page:
            # LOGGER.info("Component %s does not require changelogs on wiki", self.component_name)
            return

        create_wiki_page = internal_cubes.CreateWikiPageWithChangelog(
            self.component_name,
            input=cube_base.CubeInput(
                changelog=graph.get(
                    internal_cubes.CreateChangelog.NAME,
                ).output.resources["RELEASE_MACHINE_CHANGELOG"].first().id,
                startrek_issue=graph.get(
                    internal_cubes.CreateStartrekTicket.NAME,
                ).output.st_ticket.key,
            ),
        )

        graph.add(create_wiki_page)

    def _get_changelog_cube(self, candidate_path, needs, use_previous_branch_as_baseline=False):
        return internal_cubes.CreateChangelog(
            self.component_name,
            input=cube_base.CubeInput(
                candidate_path=candidate_path,
                major_release_num="${context.version_info.major}",
                minor_release_num="${not_null(context.version_info.minor, `0`)}",
                use_previous_branch_as_baseline=use_previous_branch_as_baseline,
            ),
            needs=needs,
        )

    @flow.release_flow()
    def release(self):

        new_tag = internal_cubes.NewTagCube(self.component_name)

        after_new_tag_dummy = dummy_cubes.RMMainGraphEntry(
            needs=[new_tag],
        )

        changelog = self._get_changelog_cube(
            candidate_path=new_tag.output.svn_data.svn_paths.tag,
            needs=[after_new_tag_dummy],
            use_previous_branch_as_baseline=self.root_cfg.changelog_cfg.use_previous_branch_as_baseline,
        )

        graph = graph_base.Graph([new_tag, after_new_tag_dummy, changelog])

        self._add_startrek_jobs_if_required(graph)
        self._add_wiki_changelog_job_if_required(graph)

        return graph
