# -*- coding: utf-8 -*-

import logging
import re
import six
import typing  # noqa
import enum

from sandbox.projects.release_machine.components.config_core.jg.lib import call_chain
from sandbox.projects.release_machine.components.config_core.jg.cube import exceptions
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.components.config_core import responsibility


LOGGER = logging.getLogger(__name__)

# RecursionError was introduced in Python 3.5. Was a plain RuntimeError before that
RECURSION_ERROR = __builtins__.get("RecursionError", RuntimeError)

if six.PY2:
    import collections
    MAPPING_TYPE = collections.Mapping
else:
    import collections.abc
    MAPPING_TYPE = collections.abc.Mapping


class CubeOutput(object):
    """
    Cube output objects. The main feature of these objects is to store a call chain
    which can later be transformed into something else (e.g. JMESPath string)

    >>> c = Cube(task="some/task")
    >>> co = CubeOutput(c, call_chain.CallChainMem())
    >>> co2 = co.a.b.c
    >>> [item.name for item in co2.call_chain.call_chain]
    ['a', 'b', 'c']
    >>> co2.call_chain.jmespath
    'a.b.c'
    >>> co3 = co.container['something'][0]
    >>> co3.call_chain.jmespath
    'a.b.c.container | [something] | [0]'

    """

    def __init__(self, cube, cc):  # type: ('Cube', call_chain.CallChainMem) -> None
        self._cube = cube
        self._call_chain = cc

    @property
    def cube(self):  # type: () -> 'Cube'
        return self._cube

    @property
    def call_chain(self):  # type: () -> call_chain.CallChainMem
        return self._call_chain

    def first(self):  # type: () -> 'CubeOutput'
        return self[0]

    def __getattr__(self, item):  # type: (six.string_types) -> 'CubeOutput'
        self._call_chain = getattr(self._call_chain, item)
        return self

    def __getitem__(self, item):  # type: (typing.Any) -> 'CubeOutput'
        self._call_chain = self._call_chain[item]
        return self


class CubeOutputTransformed(object):
    """
    Used to transform and aggregate cube outputs

    Usage

    >>> c1 = Cube(task="some/task1")
    >>> c2 = Cube(task="some/task2")
    >>> c3 = Cube(
    ...     task="some/task3",
    ...     input=CubeInput(
    ...         field3=CubeOutputTransformed(
    ...             [c1.output.field1, c2.output.field1],
    ...             lambda l: ",".join(l),
    ...         ),
    ...     ),
    ... )

    In this example `c3` will receive the values from `c1`'s output field `field1` and `c2`'s output field `field2`
    joined with ",". E.g., if `c1`'s `field1` equals "1" and `c2`'s `field2` equals "2" then `c3`'s `field3` is
    going to recieve the value "1,2".

    !NOTE! the transform_function should be applicable to a list of strings
    """

    def __init__(self, cube_output_list, transform_function):
        # type: (typing.List['CubeOutput'], typing.Callable) -> None
        """
        :param cube_output_list: a list of CubeOutput objects
        :param transform_function: a function applicable to a list of strings
        """

        self._cube_output_list = cube_output_list
        self._transform_function = transform_function

        self._check_transform_function()

    def _check_transform_function(self):
        n = len(self.cube_output_list)

        try:
            self.transform_function(["test"] * n)
        except TypeError:
            raise TypeError(
                "The transform function {f_repr} passed to {cls_name} is not applicable to a list of {n} strings, "
                "hence cannot be used for cube output transformation".format(
                    f_repr=self._transform_function,
                    cls_name=self.__class__.__name__,
                    n=n,
                ),
            )

    @property
    def cube_output_list(self):
        return self._cube_output_list

    @property
    def transform_function(self):
        return self._transform_function


class CubeInput(object):
    """
    Cube input objects. This is basically a wrapper over a dict which also incapsulates some useful logic such as
    input merges, updates and dereference
    """

    def __init__(self, **kwargs):
        self._input_dict = kwargs

    @property
    def input_dict(self):
        return dict(self._input_dict)

    def to_dict(self):
        result = {}

        for key in self._input_dict:
            result[key] = self.format_input_dict_item_value(self._input_dict[key])

        return result

    def get(self, key, default=None):
        return self._input_dict.get(key, default)

    def update(self, **kwargs):
        self._input_dict = self.update_recursive(self._input_dict, kwargs)

    def merge(self, other):

        if not isinstance(other, self.__class__):
            raise TypeError("Expected {}, got {}".format(self.__class__.__name__, type(other)))

        self.update(**other.input_dict)

    def get_references_from_value(self, value):

        # LOGGER.info("Get references from value %s (%s)", value, type(value))

        if isinstance(value, CubeOutput):
            return [value.cube]

        if isinstance(value, dict):

            result = []

            for k, v in six.iteritems(value):
                # LOGGER.info("Considering key %s of %s", k, value)
                result.extend(self.get_references_from_value(v))

            return result

        if (
            isinstance(value, list) or
            isinstance(value, tuple) or
            isinstance(value, set) or
            isinstance(value, frozenset)
        ):
            result = []

            for item in value:
                # LOGGER.info("Considering item %s of %s", item, value)
                result.extend(self.get_references_from_value(item))

            return result

        return []

    def get_references(self):

        references = []

        # LOGGER.info("Input: %s", self._input_dict)

        for key, value in six.iteritems(self._input_dict):
            # LOGGER.info("Searching for requirements in %s = %s (%s)", key, value, type(value))
            references.extend(self.get_references_from_value(value))

        return references

    @classmethod
    def format_cube_output_value(cls, value, enclose_in_braces=True):

        template = "tasks.{job_name}.{call_chain}"

        if enclose_in_braces:
            template = "${{" + template + "}}"

        return template.format(
            job_name=value.cube.name,
            call_chain=value.call_chain.jmespath,
        )

    @classmethod
    def format_input_dict_item_value(cls, value):

        if isinstance(value, CubeOutput):
            return cls.format_cube_output_value(value)

        if isinstance(value, CubeOutputTransformed):
            return value.transform_function(
                [cls.format_cube_output_value(cube_output) for cube_output in value.cube_output_list]
            )

        if isinstance(value, dict):
            return {
                k: cls.format_input_dict_item_value(v) for k, v in six.iteritems(value)
            }

        if (
            isinstance(value, list) or
            isinstance(value, tuple) or
            isinstance(value, set) or
            isinstance(value, frozenset)
        ):
            container_class = value.__class__
            return container_class([cls.format_input_dict_item_value(v) for v in value])

        return value

    @staticmethod
    def update_recursive(orig, upd):
        """
        :param orig: original input
        :param upd: upd dict
        :return: updated input

        >>> c = CubeInput(a=1)
        >>> c.update(b=2)
        >>> c.get("a")
        1
        >>> c.get("b")
        2

        >>> c = CubeInput(d={"a": 1})
        >>> c.update(d={"b": 2})
        >>> c.get("d")
        {'a': 1, 'b': 2}
        """

        result = dict(orig)

        for key, value in six.iteritems(upd):

            if isinstance(value, MAPPING_TYPE):
                result[key] = CubeInput.update_recursive(orig.get(key, {}), value)
            else:
                result[key] = value

        return result


class ICubeCondition(object):

    def to_dict(self):
        raise NotImplementedError

    def __and__(self, other):
        raise NotImplementedError

    def __or__(self, other):
        raise NotImplementedError

    def __xor__(self, other):
        return (self & other.negate()) | (self.negate() & other)

    def __eq__(self, other):
        raise NotImplementedError

    def __ne__(self, other):
        raise NotImplementedError

    def __gt__(self, other):
        raise NotImplementedError

    def __lt__(self, other):
        raise NotImplementedError

    def __ge__(self, other):
        raise NotImplementedError

    def __le__(self, other):
        raise NotImplementedError

    def negate(self):
        raise NotImplementedError


class CubeNeedsType(enum.Enum):
    ALL = "all"
    ANY = "any"
    FAIL = "fail"


class CubeManual(object):
    """
    Cube manual step. https://docs.yandex-team.ru/ci/flow#manual-confirmation
    """
    @staticmethod
    def _check_approver(approver):
        if approver is not None and not isinstance(approver, responsibility.ABCSelection):
            raise TypeError(
                "approver must be instance of responsibility.ABCSelection, but {} found".format(type(approver)),
            )

    def __init__(self, enabled=True, prompt=None, approvers=None):
        """
        :type enabled: bool
        :param enabled:
            True if need manual approve else False

        :type prompt: six.string_types
        :param prompt:
            Prompt for start cube

        :type approvers: typing.Union[responsibility.ABCSelection, typing.List[responsibility.ABCSelection]]
        :param approvers:
            Approvers start cube
        """
        if isinstance(approvers, list):
            for a in approvers:
                CubeManual._check_approver(a)
            self._approvers = approvers if approvers else None
        elif approvers is not None:
            CubeManual._check_approver(approvers)
            self._approvers = [approvers]
        else:
            self._approvers = None
        self._prompt = prompt
        self._enabled = enabled

    def __bool__(self):
        return self._enabled

    def to_dict(self):
        result = {
            "enabled": self._enabled
        }
        if self._prompt:
            result["prompt"] = self._prompt
        if self._approvers:
            result["approvers"] = [a.to_dict() for a in self._approvers]
        if len(result) == 1:
            return self._enabled
        return result


class Cube(object):
    """
    A JG node. A set of cubes forms a directed graph where cubes are interconnected according to their `requirements`.
    The requirement can be either direct (see the :param needs: parameter) or can be formed via input requirements
    (see the :param input: parameter)

    Example:
    >>> c1 = Cube(task="some/task_1")
    >>> c2 = Cube(task="some/task_2", needs=[c1])
    >>> c3 = Cube(task="some/task_3", input=CubeInput(some_field=c2.output.some_other_field))

    The above forms the following graph:

        +----+        +----+        +----+
        | c1 |------->| c2 |------->| c3 |
        +----+        +----+        +----+

    Note: despite c3 depends on c2 just like c2 depends on c1 there is a sifnificant difference between these
    two relationships: c3 also requires a particular output ('some_other_field') of c2 and claims to put the
    respective value into its own input field 'some_field', while c1->c2 relationship implies no input-output
    dependency
    """

    TYPE = "default"

    NAME_RE_SUBS__FORBIDDEN_CHARACTERS = re.compile(r"[^a-z0-9_]]")
    NAME_RE_SUBS__LONG_UNDERSCORES = re.compile(r"_[_]+")

    def __init__(
        self,
        name=None,
        title=None,
        task=None,
        input=None,
        condition=None,
        needs=None,
        attributes=None,
        manual=None,
        needs_type=CubeNeedsType.ALL,
    ):
        """
        :type name: six.string_types
        :param name:
            Cube ID

        :type title: six.string_types
        :param title:
            Cube title

        :type task: six.string_types
        :param task:
            Task or tasklet name.
            "SCREAMING_SNAKE_CASE" -- sandbox task (one of `rm_const.TASK_CI_REG_LOCATIONS` keys)
            "path/to/ci/registry/location" -- tasklet or SB task path relative to arcadia/ci/registry

        :type input: CubeInput
        :param input:
            Cube input

        :type condition: ICubeCondition
        :param condition:
            Cube run condition

        :type needs: typing.Iterable['Cube']
        :param needs:
            Additional needs

        :type manual: CubeManual or bool
        :param manual:
            Additional needs

        :type needs_type: CubeNeedsType
        :param needs_type:
            Needs type. Default is CubeNeedsType.ALL - require all of the cubes listed in :param needs: (as opposed to
            CubeNeedsType.ANY which means 'require any of the cubes listed in :param needs:')

        :raises exceptions.CubeInitializationError:
            Raised if cube initialization fails
        """

        """
        # LOGGER.info(
            "Initializing cube: "
            "name=%s "
            "title=%s "
            "task=%s "
            "input=%s "
            "condition=%s "
            "needs=%s "
            "manual=%s ",
            name,
            title,
            task,
            input,
            condition,
            needs,
            manual,
        )
        """

        self._name = name
        self._title = title
        self._task_path = None
        self._task_name = None
        self._needs = []
        self._needs_type = needs_type
        self._manual = None
        self.manual = manual
        self._attributes = attributes or {}

        self._set_task(task)
        self._set_needs(needs)

        self._input = None
        self._init_input(input)

        self._condition = None
        self.condition = condition

        self._cached_requirements = []

        # LOGGER.info("New cube initialized: %s", self)

    def _init_input(self, initial_input):

        self._input = CubeInput(**self.input_defaults)

        if initial_input:
            self._input.merge(initial_input)

        self._input.update(**self.input_override)

    @property
    def input_defaults(self):
        return {}

    @property
    def input_override(self):
        return {}

    @property
    def name(self):  # type: () -> six.string_types
        return self._name

    @property
    def title(self):  # type: () -> six.string_types
        return self._title or "".join(" ".join(self.name.split("__")).title().split("_"))

    @title.setter
    def title(self, value):
        self._title = value

    @property
    def task_name(self):  # type: () -> six.string_types
        return self._task_name

    @property
    def task_path(self):  # type: () -> six.string_types
        return self._task_path

    @property
    def attributes(self):  # type: () -> typing.Dict[six.string_types, typing.Any]
        return self._attributes

    @property
    def input(self):  # type: () -> CubeInput
        return self._input

    @property
    def output(self):  # type: () -> CubeOutput
        return CubeOutput(
            cube=self,
            cc=call_chain.CallChainMem(),
        )

    @property
    def output_params(self):  # type: () -> CubeOutput
        return self.output.output_params

    @property
    def requirements(self):  # type: () -> typing.List['Cube']

        input_refs = self.input.get_references() if self.input else []
        needs_refs = self._needs or []

        result = list(set(input_refs + needs_refs))

        req_diff = set(result) - set(self._cached_requirements)

        if req_diff:
            for req in req_diff:
                self._check_requirement(req)

        self._cached_requirements = result

        return result

    @property
    def manual(self):  # type: () -> CubeManual
        return self._manual

    @manual.setter
    def manual(self, value):
        if value is None:
            self._manual = CubeManual(enabled=False)
        elif isinstance(value, CubeManual):
            self._manual = value
        elif isinstance(value, bool):
            self._manual = CubeManual(enabled=value)
        else:
            raise TypeError("The 'manual' property value has to be a bool or CubeManual (got {})".format(type(value)))

    @property
    def condition(self):
        return self._condition

    @condition.setter
    def condition(self, value):
        if not isinstance(value, ICubeCondition) and value is not None:
            raise TypeError("ICubeCondition instance expected, got {}".format(type(value)))

        self._condition = value

    def to_dict(self):  # type: () -> typing.Dict[six.string_types, typing.Any]

        value = {
            key: CubeInput.format_input_dict_item_value(self.attributes[key])
            for key in self.attributes
        }

        value.update(
            title=self.title,
            task=self.task_path,
            needs=[cube.name for cube in self.requirements],
            input=self._input.to_dict(),
            needs_type=self._needs_type.value,
        )

        if self.manual:
            value["manual"] = self.manual.to_dict()

        if self._condition:
            value.update(self._condition.to_dict())

        return {
            self.name: value,
        }

    def set_name(self, all_names):  # type: (typing.Set[six.string_types]) -> None
        """
        Set the cube name. :param all_names: can be used to ensure uniqueness among a set of other cubes.
        If the cube already has a name and it's already present in :param all_names: the method fails.
        Otherwise if the cube name is empty the method sets it based on some of the cube's attributes
        and makes sure that it differs from any other name found in :param all_names:

        :param all_names:
            A set of names of rest of the graph cubes.

        :raises exceptions.CubeNamingError:
            Raised if the name of the cube is already set and at the same time it appears among :param all_names:
        """

        if self.name in all_names:
            raise exceptions.CubeNamingError(
                "The given cube name '{}' is not unique "
                "(there is at least one cube in graph with the exact same name)".format(
                    self.name,
                ),
            )

        if self.name:
            return

        base_name = self._construct_name()
        base_name = self._prepare_name(base_name)

        index = 0
        name = base_name

        while name in all_names:
            index += 1
            name = "{}_{}".format(base_name, index)

        self._name = name

    def _prepare_name(self, name):  # type: (six.string_types) -> six.string_types
        """
        When the cube's name is being set, this method makes sure that it meets some certain requirements

        :param name: name candidate
        :return: prepared name (slightly altered :param name:)
        """

        name = name.lower()
        name = self.NAME_RE_SUBS__FORBIDDEN_CHARACTERS.sub("", name)
        name = self.NAME_RE_SUBS__LONG_UNDERSCORES.sub("__", name)

        return name

    def _construct_name(self):  # type: () -> six.string_types
        return "{cube_type}_{task}".format(
            cube_type=self.TYPE,
            task=self.task_name,
        )

    def _set_task(self, task):  # type: (six.string_types) -> None

        if not task:
            raise exceptions.CubeInitializationError("task cannot be blank")

        if not isinstance(task, six.string_types):
            raise TypeError("Unexpected task type: {}".format(type(task)))

        if task.isupper():
            try:
                task_name = task
                task_path = rm_const.TASK_CI_REG_LOCATIONS[task]
            except KeyError:
                raise exceptions.CubeInitializationError(
                    "Sandbox task {} not found among the list of known CI-registered Sandbox tasks. "
                    "Either the type is wrong or the task is not registered. "
                    "More info at "
                    "https://wiki.yandex-team.ru/releasemachine/rm-over-ci/#ispolzovaniesandbox-zadach".format(
                        task,
                    ),
                )
        else:
            task_name = task.split("/")[-1]
            task_path = task

        self._task_name = task_name
        self._task_path = task_path

    def _set_needs(self, needs):

        if needs is None:
            return

        for index, item in enumerate(needs):

            if not isinstance(item, Cube):
                raise exceptions.CubeInitializationError(
                    "Item #{index} of {class_name}'s needs is not a Cube object. "
                    "Expected Cube, got {unexpected_type} instead".format(
                        index=index,
                        class_name=self.__class__.__name__,
                        unexpected_type=type(item),
                    ),
                )

        self._needs = needs

    def add_requirement(self, cube):

        if not isinstance(cube, Cube):
            raise exceptions.CubeUsageError(
                "A cube can only depend on cube (caught an attempt to add {} to cube needs)".format(type(cube)),
            )

        self._check_requirement(cube)

        self._needs.append(cube)

        self._needs = list(set(self._needs))

    def clone(self, **kwargs):
        """Return a completely new cube cloned from this one, with arguments updated with **kwargs"""

        new_cube_kwargs = dict(
            name=None,
            title=self.title,
            task=self.task_path,
            input=self.input,
            condition=self._condition,
            needs=self._needs,
            manual=self.manual,
        )

        new_cube_kwargs.update(kwargs)

        return self.__class__(**new_cube_kwargs)

    def _check_requirement(self, cube):  # type:  ('Cube') -> None
        """
        Make sure that neither :param cube: nor any of it's recursive requirements include this cube

        :type cube:
            'Cube'
        :param cube:
            A cube object

        :raises exceptions.CubeCircularDependencyError:
            Raised if `self` is found among :param cube: recursive requirements
        """

        error_msg = "Circular dependency detected between {} and {}".format(
            self.name,
            cube.name,
        )

        try:

            if self in cube.requirements:
                raise exceptions.CubeCircularDependencyError(error_msg)

            for item in cube.requirements:
                self._check_requirement(item)

        except RECURSION_ERROR:
            raise exceptions.CubeCircularDependencyError(error_msg)
