import six

from sandbox.projects.common.constants import constants as common_const

from sandbox.projects.release_machine.components.config_core.jg.cube import base as cube_base
from sandbox.projects.release_machine.components.config_core.jg.cube import exceptions as cube_exceptions

from sandbox.projects.release_machine.core import const as rm_const


TYPE_BUILD = "build"


class YaBuildCubeBase(cube_base.Cube):
    """
    Base cube class for YaMake/YaPackage-like tasks
    """

    TYPE = TYPE_BUILD
    DEFAULT_TASK = None

    def __init__(
        self,
        name=None,
        title=None,
        task=None,
        input=None,
        condition=None,
        needs=None,
        **kwargs
    ):

        task = task or self.DEFAULT_TASK

        super(YaBuildCubeBase, self).__init__(
            name=name,
            title=title,
            task=task,
            input=input,
            condition=condition,
            needs=needs,
            **kwargs
        )

    @property
    def input_defaults(self):
        return {}

    @property
    def input_override(self):
        return {}

    @classmethod
    def prepare_ya_make_input_iterable(cls, value):

        if isinstance(value, six.string_types):
            return value

        if hasattr(value, "__iter__"):
            return ";".join([target.strip() for target in value])

        raise TypeError(
            "Unexpected type of 'value' in {}. Expected string or list of strings, got {}".format(
                cls.__name__,
                type(value),
            ),
        )


class YaMakeBuildCubeBase(YaBuildCubeBase):
    """
    Base cube class for YaMake-like tasks

    Examples:
        YaMakeBuildCubeBase("search/metrics_templates", "search/metrics_templates=search/", "METRICS_TEMPLATE_RESOURCE")

        YaMakeBuildCubeBase(
            "path/to/target/1;path/to/target/2",
            "path/to/target/1=t1;path/to/target/2=t2",
            "TARGET_1;TARGET_2",
        )

        equivalent to

        YaMakeBuildCubeBase(
            ["path/to/target/1", "path/to/target/2"],
            ["path/to/target/1=t1", "path/to/target/2=t2"],
            ["TARGET_1", "TARGET_2"],
        )
    """

    def __init__(
        self,
        targets,
        artifacts,
        resource_types,
        source_artifacts=None,
        **kwargs
    ):

        self._init_targets = targets
        self._init_artifacts = artifacts
        self._init_resource_types = resource_types
        self._init_source_artifacts = source_artifacts

        super(YaMakeBuildCubeBase, self).__init__(**kwargs)

    @property
    def input_defaults(self):
        return {
            common_const.BUILD_SYSTEM_KEY: common_const.SEMI_DISTBUILD_BUILD_SYSTEM,
            common_const.ALLOW_AAPI_FALLBACK: True,
            common_const.USE_AAPI_FUSE: True,
            common_const.USE_ARC_INSTEAD_OF_AAPI: True,
        }

    @property
    def input_override(self):
        result = {
            "targets": self.prepare_ya_make_input_iterable(self._init_targets),
            "result_resources_types": self.prepare_ya_make_input_iterable(self._init_resource_types),
        }

        if self._init_artifacts:
            result.update({
                "arts": self.prepare_ya_make_input_iterable(self._init_artifacts),
            })

        if self._init_source_artifacts:
            result.update({
                "arts_source": self.prepare_ya_make_input_iterable(self._init_source_artifacts),
            })

        return result


class YaMake2(YaMakeBuildCubeBase):
    DEFAULT_TASK = "YA_MAKE_2"


class KosherYaMake(YaMakeBuildCubeBase):
    DEFAULT_TASK = "KOSHER_YA_MAKE"


class YaMakeTemplate(YaMakeBuildCubeBase):
    DEFAULT_TASK = None

    def __init__(self, target_resources=None, **kwargs):
        if not kwargs.get("task", None):
            raise cube_exceptions.CubeInitializationError(
                "{} requires a particular task to be specified, but none provided".format(
                    self.__class__.__name__,
                )
            )

        self._target_resources = target_resources

        super(YaMakeTemplate, self).__init__(None, None, None, **kwargs)

    @property
    def input_override(self):

        result = {}

        if self._target_resources:
            result["target_resources"] = self._target_resources

        return result


class YaPackageBuildCubeBase(YaBuildCubeBase):
    """
    Base cube class for YaPackage-like tasks
    """

    def __init__(self, packages, resource_types, use_compression=True, **kwargs):
        self._packages = packages
        self._resource_types = resource_types
        self._use_compression = use_compression

        super(YaPackageBuildCubeBase, self).__init__(**kwargs)

    @property
    def input_defaults(self):
        return {
            common_const.CLEAR_BUILD_KEY: False,
            common_const.BUILD_SYSTEM_KEY: common_const.YMAKE_BUILD_SYSTEM,
            common_const.ALLOW_AAPI_FALLBACK: True,
            common_const.USE_AAPI_FUSE: True,
            common_const.USE_ARC_INSTEAD_OF_AAPI: True,
            "publish_package": False,
            "requirements": {
                "sandbox": {
                    "platform": "linux",
                },
            },
        }

    @property
    def input_override(self):
        result = {
            "resource_type": ";".join(self._resource_types),
            "package_type": "tarball",
            "compress_package_archive": self._use_compression,
        }

        if self._packages:
            result["packages"] = ";".join(self._packages)

        return result


class YaPackage2(YaPackageBuildCubeBase):
    DEFAULT_TASK = "YA_PACKAGE_2"


class KosherYaPackage(YaPackageBuildCubeBase):
    DEFAULT_TASK = "KOSHER_YA_PACKAGE"


class OldYaPackage(YaPackageBuildCubeBase):
    DEFAULT_TASK = "YA_PACKAGE"


class DeployBinaryTask(cube_base.Cube):

    TYPE = TYPE_BUILD
    DEFAULT_TASK = "common/sandbox/deploy_binary_task"

    def __init__(self, target, target_task_type, attrs=None, **kwargs):

        self._target = target
        self._target_task_type = target_task_type
        self._attrs = attrs

        super_kwargs = dict(kwargs)
        super_kwargs.update({
            "task": self.DEFAULT_TASK,
        })

        super(DeployBinaryTask, self).__init__(**super_kwargs)

    @property
    def input_defaults(self):
        return {
            "arcadia_url": rm_const.CIJMESPathCommon.ARCADIA_URL,
            "binary_executor_release_type": "stable",
            "release_ttl": 14,
            "use_yt_cache": True,
            "yt_token_vault": "YT_STORE_TOKEN",
        }

    @property
    def input_override(self):

        attrs = dict(self._attrs or {})

        attrs.update(task_type=self.target_task_type)

        return{
            "target": self.target,
            "attrs": attrs,
        }

    @property
    def target(self):
        return self._target

    @property
    def target_task_type(self):
        return self._target_task_type


class AfishaBuildDocker(cube_base.Cube):
    TYPE = TYPE_BUILD
    DEFAULT_TASK = "projects/afisha/afisha_build_docker"

    def __init__(self, component_name, package_path, build_system=None, docker_build_args=None, ya_package_env_vars=None, release=True, **kwargs):

        self._component_name = component_name
        self._package_path = package_path
        self._release = release
        self._build_system = build_system
        self._docker_build_args = docker_build_args
        self._ya_package_env_vars = ya_package_env_vars

        super_kwargs = dict(kwargs)
        super_kwargs.update({
            "task": self.DEFAULT_TASK,
        })

        super(AfishaBuildDocker, self).__init__(**super_kwargs)

    @property
    def input_defaults(self):
        return {
            "number": rm_const.CIJMESPathCommon.MAJOR_RELEASE_NUM,
        }

    @property
    def input_override(self):
        result = {
            "package_path": self._package_path,
            "release": self._release,
            "component_name": self._component_name,
        }
        if self._build_system:
            result["BuildSystem"] = self._build_system
        if self._docker_build_args:
            result["docker_build_args"] = self._docker_build_args
        if self._ya_package_env_vars:
            result["ya_package_env_vars"] = self._ya_package_env_vars

        return result


class RCBuildTasklet(cube_base.Cube):
    """
    Build tasklet (and upload to Sandbox with schema) via RunCommand
    """

    TASK = "common/misc/run_command"
    TYPE = TYPE_BUILD

    def __init__(
        self,
        tasklet_path,
        tasklet_binary_path,
        sandbox_owner,
        sandbox_token_secret_uuid="sec-01desry8fbgvnkbeybem81ferv",
        sandbox_token_secret_key="common_release_token",
        **kwargs
    ):

        self._tasklet_path = tasklet_path
        self._tasklet_binary_path = tasklet_binary_path
        self._sandbox_owner = sandbox_owner
        self._sandbox_token_secret_uuid = sandbox_token_secret_uuid
        self._sandbox_token_secret_key = sandbox_token_secret_key

        kwargs["task"] = self.TASK

        super(RCBuildTasklet, self).__init__(**kwargs)

    RESULT_RESOURCE_ID_FILE_NAME = "result_resource_id"

    @property
    def cmd_line(self):
        make_cmd = "ya make --build=release {tasklet_path}".format(
            tasklet_path=self._tasklet_path,
        )

        upload_cmd = "{tasklet_bin_path} sb-upload --sb-owner {owner} --sb-schema".format(
            tasklet_bin_path=self._tasklet_binary_path,
            owner=self._sandbox_owner,
        )

        return "{make} && {upload} | tail -n1 > $RESULT_RESOURCES_PATH/{outname}".format(
            make=make_cmd,
            upload=upload_cmd,
            outname=self.RESULT_RESOURCE_ID_FILE_NAME,
        )

    @property
    def secret_environment_variables(self):
        return [
            {
                "key": "GLYCINE_TOKEN",
                "secret_spec": {
                    "uuid": self._sandbox_token_secret_uuid,
                    "key": self._sandbox_token_secret_key,
                },
            }
        ]

    @property
    def input_defaults(self):
        return {
            "config": {
                "cmd_line": self.cmd_line,
                "arc_mount_config": {
                    "enabled": True,
                },
                "secret_environment_variables": self.secret_environment_variables,
                "result_output": [
                    {
                        "path": self.RESULT_RESOURCE_ID_FILE_NAME,
                    }
                ],
            }
        }


class BuildReleaseEngineUi(cube_base.Cube):

    TYPE = TYPE_BUILD
    TASK = "projects/release_machine/build_release_engine_ui"

    def __init__(self, **kwargs):
        kwargs["task"] = self.TASK
        super(BuildReleaseEngineUi, self).__init__(**kwargs)


class BuildHorizonAgentConfig(cube_base.Cube):

    TYPE = TYPE_BUILD
    TASK = "projects/apphost/build_horizon_agent_config"

    def __init__(self, vertical, arc_tag, **kwargs):

        self._vertical = vertical
        self._arc_tag = arc_tag

        kwargs["task"] = self.TASK

        super(BuildHorizonAgentConfig, self).__init__(**kwargs)

    @property
    def input_override(self):
        return {
            "vertical": self._vertical.upper(),
            "arc_tag": self._arc_tag,
        }
