from sandbox.projects.release_machine.components.config_core.jg.cube import base as cube_base
from sandbox.projects.release_machine.core import const as rm_const


TYPE_RELEASE = "release"
TYPE_STABLE_RELEASE = "release_stable"
TYPE_PRESTABLE_RELEASE = "release_prestable"


class ReleaseCubeBase(cube_base.Cube):
    TYPE = TYPE_RELEASE
    TASK = None

    def __init__(self, component_name, where_to_release, **kwargs):

        self._component_name = component_name
        self._where_to_release = where_to_release

        kwargs["task"] = self.TASK

        if "manual" not in kwargs:
            kwargs["manual"] = True

        super(ReleaseCubeBase, self).__init__(**kwargs)

    def _construct_name(self):
        return "{cube_type}__{component_name}__{where}".format(
            cube_type=self.TYPE,
            component_name=self._component_name,
            where=self._where_to_release,
        )

    @property
    def input_override(self):
        return {
            "component_name": self._component_name,
            "where_to_release": self._where_to_release,
        }

    def clone(self, **kwargs):
        new_cube_kwargs = dict(
            component_name=self._component_name,
            where_to_release=self._where_to_release,
        )

        new_cube_kwargs.update(kwargs)
        return super(ReleaseCubeBase, self).clone(**new_cube_kwargs)


class ReleaseRmComponent2(ReleaseCubeBase):
    TASK = "common/releases/release_rm_component_2"
    MAJOR_RELEASE_NUM = rm_const.CIJMESPathCommon.MAJOR_RELEASE_NUM
    MINOR_RELEASE_NUM = rm_const.CIJMESPathCommon.MINOR_RELEASE_NUM

    @property
    def input_override(self):
        res = super(ReleaseRmComponent2, self).input_override
        res.update(
            major_release_num=self.MAJOR_RELEASE_NUM,
            minor_release_num=self.MINOR_RELEASE_NUM,
        )
        return res


class ReleaseRmComponent2Trunk(ReleaseRmComponent2):
    MAJOR_RELEASE_NUM = "0"
    MINOR_RELEASE_NUM = "0"


class AfishaRelease(ReleaseCubeBase):
    TASK = "projects/afisha/afisha_release"

    @property
    def input_override(self):
        return {
            "component_name": self._component_name,
            "flow": self._where_to_release,
            "target_state": "DEPLOYED",
            "version": "{}.{}".format(
                rm_const.CIJMESPathCommon.MAJOR_RELEASE_NUM,
                rm_const.CIJMESPathCommon.MINOR_RELEASE_NUM,
            ),
        }


class AfishaRelease2(AfishaRelease):
    TASK = "projects/afisha/afisha_release_2"

    def __init__(
        self,
        release_number_from_version=False,
        release_build_resource=True,
        allow_downgrade=True,
        allow_minor_downgrade=True,
        **kwargs
    ):

        self._release_number_from_version = release_number_from_version
        self._release_build_resource = release_build_resource
        self._allow_downgrade = allow_downgrade
        self._allow_minor_downgrade = allow_minor_downgrade

        super(AfishaRelease2, self).__init__(**kwargs)

    @property
    def input_override(self):
        return {
            "component_name": self._component_name,
            "flow": self._where_to_release,
            "target_state": "DEPLOYED",
            "version": "{}.{}".format(
                "${not_null(context.rollback_to_version.major, context.version_info.major)}",
                "${context.rollback_to_version && not_null(context.rollback_to_version.minor, `0`) || not_null(context.version_info.minor, `0`)}",
            ),
            "allow_downgrade": self._allow_downgrade,
            "allow_minor_downgrade": self._allow_minor_downgrade,
            "release_number_from_version": self._release_number_from_version,
            "release_build_resource": self._release_build_resource,
        }


class TaskletRelease(ReleaseCubeBase):
    TASK = "projects/release_machine/commit_tasklet_version"

    def __init__(
        self,
        component_name,
        where_to_release,
        tasklet_version_infos,
        do_commit=True,
        commit_message="",
        arc_token_secret_uuid="sec-01desry8fbgvnkbeybem81ferv",
        arc_token_secret_key="common_release_token",
        **kwargs
    ):
        super(TaskletRelease, self).__init__(
            component_name=component_name,
            where_to_release=where_to_release,
            **kwargs
        )

        self.input.update(config={
            "do_commit": do_commit,
            "commit_message": commit_message,
            "tasklet_version_infos": tasklet_version_infos,
            "arc_token": {
                "uuid": arc_token_secret_uuid,
                "key": arc_token_secret_key,
            },
        })

    @property
    def input_override(self):
        return {}


class ReleaseZephyrServiceUi(ReleaseCubeBase):
    TASK = "projects/release_machine/release_zephyr_service_ui"

    def __init__(
        self,
        component_name,
        where_to_release,
        resource,
        version="${context.target_revision.number}",
        dest_folder="dist",
        access_secret_key="zephyr_access_key",
        **kwargs
    ):
        self._resource = resource
        self._version = version
        self._dest_folder = dest_folder
        self._access_secret_key = access_secret_key

        super(ReleaseZephyrServiceUi, self).__init__(component_name, where_to_release, **kwargs)

    @property
    def input_defaults(self):
        default = super(ReleaseZephyrServiceUi, self).input_defaults

        default.update(
            resource=self._resource,
            version=self._version,
            dest_folder=self._dest_folder,
            access_secret_key=self._access_secret_key,
        )

        return default


class ScheduleRelease(ReleaseCubeBase):
    TASK = "projects/release_machine/schedule_release"

    def __init__(
        self,
        component_name,
        where_to_release,
        flows,
        warden_component_name,
        warden_parent_component_name,
        deploy_system,
        schedule_mode,
        **kwargs
    ):
        self._flows = flows
        self._warden_component_name = warden_component_name
        self._warden_parent_component_name = warden_parent_component_name
        self._deploy_system = deploy_system
        self._schedule_mode = schedule_mode

        super(ScheduleRelease, self).__init__(component_name, where_to_release, **kwargs)

    @property
    def input_override(self):
        result = {}

        result.update(super(ScheduleRelease, self).input_override)

        result.update(
            flows=self._flows,
            warden_component_name=self._warden_component_name,
            warden_parent_component_name=self._warden_parent_component_name,
            deploy_system=self._deploy_system,
            schedule_mode=self._schedule_mode,
            major_release_num=rm_const.CIJMESPathCommon.MAJOR_RELEASE_NUM,
            minor_release_num=rm_const.CIJMESPathCommon.MINOR_RELEASE_NUM,
        )

        return result


class AltaySetReleaseMeta(ReleaseCubeBase):
    TASK = "projects/altay/altay_set_release_meta"

    @property
    def input_defaults(self):
        return {
            "tag": "altay/stable-{}-{}".format(
                rm_const.CIJMESPathCommon.MAJOR_RELEASE_NUM,
                rm_const.CIJMESPathCommon.MINOR_RELEASE_NUM,
            ),
        }
