from sandbox.projects.release_machine.components.config_core.jg.cube import base as cube_base
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import internal as internal_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import build as build_cubes
from sandbox.projects.quasar.platform import get_platform_path_key


class CreateGerritBranchCube(cube_base.Cube):
    TASK = "projects/smart_devices/create_gerrit_branch"
    TYPE = "create_gerrit_branch"
    REPOSITORY_JSON_PATH = 'arcadia:/arc/trunk/arcadia/smart_devices/platforms/{}/firmware'

    def __init__(
        self,
        platform,
        arcadia_release_branch_number,
        rm_branch_name=None,
        is_factory_rm_branch=False,
        ssh_user=None,
        ssh_key_vault_owner=None,
        ssh_key_vault_name=None,
        resource_owner=None,
        yav_token_vault=None,
        repository_json_path=None,
        **kwargs
    ):

        self._platform = platform
        self._arcadia_release_branch_number = arcadia_release_branch_number
        self._rm_branch_name = rm_branch_name
        self._is_factory_rm_branch = is_factory_rm_branch
        self._repository_path = repository_json_path or self.REPOSITORY_JSON_PATH.format(
            get_platform_path_key(platform)
        )
        self._ssh_user = ssh_user or "robot-quasar"
        self._ssh_key_vault_owner = ssh_key_vault_owner or "QUASAR"
        self._ssh_key_vault_name = ssh_key_vault_name or "robot-quasar-ssh-private-key"
        self._resource_owner = resource_owner or "QUASAR"
        self._yav_token_vault = yav_token_vault or "QUASAR:robot-quasar-yav-token"

        kwargs["task"] = self.TASK

        super(CreateGerritBranchCube, self).__init__(**kwargs)

    @property
    def input_defaults(self):
        return dict(
            resource_owner=self._resource_owner,
            yav_token_vault=self._yav_token_vault,
            platform=self._platform,
            repository_config_url=self._repository_path,
            ssh_user=self._ssh_user,
            ssh_key_vault_owner=self._ssh_key_vault_owner,
            ssh_key_vault_name=self._ssh_key_vault_name,
        )

    @property
    def input_override(self):

        result = {
            "arcadia_release_branch_number": self._arcadia_release_branch_number,
        }

        if self._rm_branch_name is not None:
            result["arcadia_release_branch_name"] = self._rm_branch_name

        if self._is_factory_rm_branch:
            result["gerrit_release_branch_prefix"] = 'factory-'
            result["keep_last_x_branches"] = 3
        else:
            result["keep_last_x_branches"] = 5

        return result

    def _construct_name(self):
        return "{}__{}".format(
            self.TYPE,
            self._platform,
        )


class MergeSystemBranchesCube(internal_cubes.MergeToStable):
    def __init__(
        self,
        component_name,
        new_branch_number,
        revisions,
        resource_owner=None,
        yav_token_vault=None,
        skip_svn_branch=False,
        developers_options=False,
        **kwargs
    ):
        self._new_branch_number = new_branch_number
        self._revisions = revisions
        self._resource_owner = resource_owner or "QUASAR"
        self._yav_token_vault = yav_token_vault or "QUASAR:robot-quasar-yav-token"
        self._skip_svn_branch = skip_svn_branch
        self._developers_options = developers_options
        super(MergeSystemBranchesCube, self).__init__(component_name=component_name, **kwargs)

    @property
    def input_defaults(self):
        return {
            "resource_owner": self._resource_owner,
            "yav_token_vault": self._yav_token_vault,
            "developers_options": self._developers_options,
            "branch_nums_to_merge": self._new_branch_number,
            "skip_svn_branch": self._skip_svn_branch,
            "revs": cube_base.CubeOutputTransformed(self._revisions, lambda l: ",".join(l)),
        }


class DaemonsBuildCube(build_cubes.KosherYaPackage):
    def __init__(self, platform, **kwargs):

        super(DaemonsBuildCube, self).__init__(
            name="build_{}_daemons".format(platform),
            title="Build {} daemons package".format(platform),
            packages=["smart_devices/platforms/{}/packages/daemons.json".format(platform)],
            resource_types=["QUASAR_DAEMONS"],
            input=cube_base.CubeInput(
                build_type="minsizerel",
                use_arc_instead_of_aapi=True,
                ignore_recurses=True,
                raw_package=True,
                overwrite_read_only_files=True,
                package_resource_attrs={"quasar_platform": platform},
            ),
            **kwargs
        )


class FactoryDaemonsBuildCube(build_cubes.KosherYaPackage):
    def __init__(self, platform, **kwargs):

        super(FactoryDaemonsBuildCube, self).__init__(
            name="build_{}_factory_daemons".format(platform),
            title="Build {} factory daemons package".format(platform),
            packages=["smart_devices/platforms/{}/factory/packages/daemons.json".format(platform)],
            resource_types=["QUASAR_FACTORY_DAEMONS"],
            input=cube_base.CubeInput(
                build_type="minsizerel",
                use_arc_instead_of_aapi=True,
                ignore_recurses=True,
                raw_package=True,
                overwrite_read_only_files=True,
                package_resource_attrs={"quasar_platform": platform},
            ),
            **kwargs
        )


class PublishImageToS3Cube(cube_base.Cube):
    def __init__(self, image_build_cube_name, resourse_type, type="", **kwargs):

        super(PublishImageToS3Cube, self).__init__(
            name="publish_{}_image_to_s3".format(type),
            title="Publish {} image to s3".format(type),
            task="common/sandbox/tasklet_executor",
            input=cube_base.CubeInput(
                binary_executor_release_type="none",
                resource_owner="QUASAR",
                resource_type="SANDBOX_TASKS_BINARY",
                yav_token_vault="QUASAR:robot-quasar-yav-token",
                tasklet_name="PublishImageToS3",
                resource_attrs={"default_tasklet_name": "PublishImageToS3"},
                tasklet_input={
                    "sandbox_token": {"uuid": "sec-01d2ffwrdbwyj37zkj4r8zegsn", "key": "robot-quasar-sandbox-token"},
                    "resources_ids": [
                        "${{tasks.{}.resources[? type == '{}' && attributes.signed == 'True'].id | single(@)}}".format(
                            image_build_cube_name, resourse_type
                        )
                    ],
                },
            ),
            **kwargs
        )


class PublishImageToQdCube(cube_base.Cube):
    def __init__(self, publish_to_s3_cube_name, type="", **kwargs):
        self._image_type = type

        super(PublishImageToQdCube, self).__init__(
            name="publish_{}_image_to_qd".format(type),
            title="Publish {} image to qd".format(type),
            task="common/sandbox/tasklet_executor",
            input=cube_base.CubeInput(
                binary_executor_release_type="none",
                resource_owner="QUASAR",
                resource_type="SANDBOX_TASKS_BINARY",
                yav_token_vault="QUASAR:robot-quasar-yav-token",
                tasklet_name="PublishImageToQuasmodrom",
                resource_attrs={"default_tasklet_name": "PublishImageToQuasmodrom"},
                tasklet_input={
                    "sandbox_token": {"uuid": "sec-01d2ffwrdbwyj37zkj4r8zegsn", "key": "robot-quasar-sandbox-token"},
                    "resources_ids": "${{keys(tasks.{}.output_params.tasklet_output.resourcesLinks)}}".format(
                        publish_to_s3_cube_name
                    ),
                    "comment": "Image from branch ${context.branch} revision ${context.target_revision.number}",
                },
            ),
            **kwargs
        )

    @property
    def image_type(self):
        self._image_type

    @property
    def is_prod_build(self):
        return self._image_type == "" or self._image_type == "user"


class UpdateToQdCube(cube_base.Cube):
    def __init__(self, publish_to_qd_cube_name, group, **kwargs):
        super(UpdateToQdCube, self).__init__(
            name="publish_update_to_{}".format(group),
            title="Publish update to {}".format(group),
            task="common/sandbox/tasklet_executor",
            input=cube_base.CubeInput(
                binary_executor_release_type="none",
                resource_owner="QUASAR",
                resource_type="SANDBOX_TASKS_BINARY",
                yav_token_vault="QUASAR:robot-quasar-yav-token",
                tasklet_name="UpdateToQuasmodrom",
                resource_attrs={"default_tasklet_name": "UpdateToQuasmodrom"},
                tasklet_input={
                    "sandbox_token": {"uuid": "sec-01d2ffwrdbwyj37zkj4r8zegsn", "key": "robot-quasar-sandbox-token"},
                    "group_name": group,
                    "firmware_id": "${{tasks.{}.output_params.tasklet_output.firmwares.* | [?imageType=='ota'].id | single(@)}}".format(
                        publish_to_qd_cube_name
                    ),
                    "comment": "Image from branch ${context.branch} revision ${context.target_revision.number}",
                },
            ),
            **kwargs
        )
