import copy
import enum
import functools
import logging
import six  # noqa
import typing  # noqa

from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.components.config_core.jg.graph import base as graph_base
from sandbox.projects.release_machine.components.config_core.jg import exceptions


LOGGER = logging.getLogger(__name__)

STR = six.string_types[0]
L_DICT_STR_KEYS = typing.List[typing.Dict[STR, typing.Any]]
OPTIONAL_STR = typing.Optional[STR]
OPTIONAL_INT = typing.Optional[int]


class ActionTriggerOn(enum.Enum):
    PR = "pr"
    COMMIT = "commit"

    def __str__(self):
        return self.value


class ActionTrigger(object):

    def __init__(self, on, **kwargs):  # type: (ActionTriggerOn, **typing.Any) -> None

        if str(on) not in list(map(str, ActionTriggerOn)):
            raise TypeError("'on' argument should be one of ActionTriggerOn values")

        self._on = on
        self._options = kwargs

    def to_dict(self):
        result = {
            "on": str(self._on),
        }

        result.update(self._options)

        return result


class ActionBranches(object):

    def __init__(
        self,
        pattern,
        auto_create=False,
        forbid_trunk_releases=False,
        independent_stages=False,
        auto=False,
    ):  # type: (STR, bool, bool, bool, bool) -> None
        """
        :param pattern:                                                  https://docs.yandex-team.ru/ci/release#branches
        :param auto_create:                                    https://docs.yandex-team.ru/ci/release#auto-create-branch
        :param forbid_trunk_releases:                       https://docs.yandex-team.ru/ci/release#forbid-trunk-releases
        :param independent_stages:                             https://docs.yandex-team.ru/ci/release#independent-stages
        :param auto:                                        https://docs.yandex-team.ru/ci/release#branches-auto-release
        """

        self._pattern = pattern
        self._auto_create = auto_create
        self._forbid_trunk_releases = forbid_trunk_releases
        self._independent_stages = independent_stages
        self._auto = auto

    @property
    def pattern(self):
        return self._pattern

    @property
    def auto_create(self):
        return self._auto_create

    @property
    def forbid_trunk_releases(self):
        return self._forbid_trunk_releases

    @property
    def independent_stages(self):
        return self._independent_stages

    @property
    def auto(self):
        return self._auto

    def to_dict(self):
        return {
            "pattern": self.pattern,
            "auto-create": self.auto_create,
            "forbid-trunk-releases": self.forbid_trunk_releases,
            "independent-stages": self.independent_stages,
            "auto": self.auto,
        }


ReleaseStageDisplaceVerbose = typing.TypedDict(
    "ReleaseStageDisplaceVerbose",
    {
        "on-status": typing.List[STR],
    },
)

DISPLACE_ON_FAILURE = {"on-status": "FAILURE"}  # type: ReleaseStageDisplaceVerbose
DISPLACE_ON_FAILURE_OR_WAITING = {
    "on-status": ["FAILURE", "WAITING_FOR_MANUAL_TRIGGER", "WAITING_FOR_STAGE"],
}  # type: ReleaseStageDisplaceVerbose
DISPLACE_ON_RUNNING_OR_FAILURE_OR_WAITING = {
    "on-status": ["RUNNING", "RUNNING_WITH_ERRORS", "FAILURE", "WAITING_FOR_MANUAL_TRIGGER", "WAITING_FOR_STAGE"],
}  # type: ReleaseStageDisplaceVerbose


class ReleaseActionStageData(object):

    def __init__(self, name, title=None, cube_names=None, cube_types=None, displace=False, rollback=False):
        # type: (STR, OPTIONAL_STR, typing.List[STR], typing.List[STR], typing.Union[bool, ReleaseStageDisplaceVerbose], bool) -> None

        self._name = name
        self._title = title or name.title()
        self._cube_names = cube_names or []
        self._cube_types = cube_types or []
        self._displace = displace
        self._rollback = rollback

    @property
    def name(self):
        return self._name

    @property
    def title(self):
        return self._title

    @property
    def cube_names(self):
        return self._cube_names

    def add_cube_name(self, cube_name):
        release_action_stage_copy = copy.copy(self)
        release_action_stage_copy._cube_names.append(cube_name)
        return release_action_stage_copy

    def set_rollback(self, value):
        # type: (bool) -> 'ReleaseActionStageData'

        if not isinstance(value, bool):
            raise TypeError("set_rollback: Expected bool, got %s", type(value))

        release_action_stage_copy = copy.copy(self)
        release_action_stage_copy._rollback = value
        return release_action_stage_copy

    @property
    def cube_types(self):
        return self._cube_types

    @property
    def displace(self):
        return self._displace

    @property
    def rollback(self):
        return self._rollback

    def to_dict(self):
        return {
            "name": self.name,
            "title": self.title,
            "cubes": self._cube_names,
            "displace": self._displace,
        }

    def __copy__(self):
        return type(self)(
            name=self.name,
            title=self.title,
            cube_names=self.cube_names,
            cube_types=self.cube_types,
            displace=self.displace,
            rollback=self.rollback,
        )


L_STAGES = typing.List[ReleaseActionStageData]
L_TRIGGERS = typing.List[ActionTrigger]


class ActionParameters(object):
    """
    CI action parameters
    """

    def __init__(
        self,
        title=None,
        description=None,
        auto=False,
        triggers=None,
        branches=None,
        stages=None,
        start_version=None,
        filters=None,
    ):  # type: (STR, STR, bool, L_TRIGGERS, ActionBranches, L_STAGES, int, L_DICT_STR_KEYS) -> None

        self._auto = auto
        self._triggers = triggers
        self._title = title
        self._description = description
        self._branches = branches
        self._stages = stages
        self._start_version = start_version
        self._filters = filters

    @property
    def auto(self):
        return self._auto

    @property
    def triggers(self):
        return self._triggers

    @property
    def description(self):
        return self._description

    @property
    def title(self):
        return self._title

    @property
    def branches(self):
        return self._branches

    @property
    def stages(self):
        return self._stages

    @property
    def start_version(self):
        return self._start_version

    @property
    def filters(self):
        return self._filters

    def to_dict(self):
        return {
            "title": self._title or "",
            "description": self._description or "",
            "auto": self._auto,
            "branches": self._branches,
            "triggers": self._triggers,
        }


class CiActionData(object):

    def __init__(self, graph, action_parameters):  # type: ('CiActionData', graph_base.Graph, ActionParameters) -> None
        self._graph = graph
        self._action_parameters = action_parameters

    @property
    def graph(self):
        return self._graph

    @property
    def action_parameters(self):
        return self._action_parameters


RegisterFlowParams = typing.TypedDict(
    "RegisterFlowParams",
    {
        "kind": STR,
        "auto": bool,
        "triggers": typing.Optional[typing.List[ActionTrigger]],
        "title": OPTIONAL_STR,
        "description": OPTIONAL_STR,
        "branches": typing.Optional[ActionBranches],
        "stages": typing.Optional[ReleaseActionStageData],
        "parametrize": typing.Optional[list],
        "name_override": OPTIONAL_STR,
        "start_version": OPTIONAL_INT,
        "filters": typing.Optional[L_DICT_STR_KEYS],
    },
)

REGISTER_FLOW_PARAMS_DEFAULTS = RegisterFlowParams(
    kind=rm_const.CIActionKind.ACTION,
    auto=False,
    triggers=None,
    title=None,
    description=None,
    branches=None,
    stages=None,
    parametrize=None,
    name_override=None,
    start_version=None,
    filters=None,
)


class FlowDeclarationWrapper(object):
    """
    Flow declaration wrapper. Used by `register_flow` decorator to wrap the decorated function
    """

    def __init__(self, original_func, **kwargs):
        self._original_func = original_func

        self._register_params = dict(**REGISTER_FLOW_PARAMS_DEFAULTS)
        self._register_params.update(kwargs)

    @property
    def register_params(self):
        return dict(self._register_params)

    def _get_graph(self, instance, **kwargs):
        # LOGGER.info(
        #    "Building flow %s of kind %s with kwargs %s",
        #    self._original_func.__name__,
        #    self._register_params["kind"],
        #    kwargs,
        # )

        graph = self._original_func(instance, **kwargs)

        # LOGGER.info("Got graph: %s", graph)

        if graph is None:
            return None

        if not isinstance(graph, graph_base.Graph):
            raise exceptions.FlowRegistrationError(
                "Method {method_name} did not return a valid graph object: "
                "expected an instance of {graph_type}, got {incorrect_type}".format(
                    method_name=self._original_func.__name__,
                    graph_type=graph_base.Graph.__name__,
                    incorrect_type=type(graph).__name__,
                ),
            )

        return graph

    def _get_parametrize_names_and_values(self):

        parametrize = self._register_params["parametrize"]

        if parametrize:
            param_names = parametrize[0]
            param_values = parametrize[1]
        else:
            param_names = []
            param_values = [[]]

        return param_names, param_values

    def _iter_parametrized(self):

        param_names, param_values = self._get_parametrize_names_and_values()

        for value_list in param_values:
            yield {key: value for key, value in zip(param_names, value_list)}

    def _customize_register_params(self, instance, graph):

        return ActionParameters(
            auto=self._register_params["auto"],
            triggers=self._register_params["triggers"],
            title=self._register_params["title"],
            description=self._register_params["description"],
            branches=self._register_params["branches"],
            stages=self._register_params["stages"],
            start_version=self._register_params["start_version"],
            filters=self._register_params["filters"],
        )

    def _register_graph(self, instance, graph):

        reg_name = self._register_params.get("name_override") or self._original_func.__name__
        reg_name = reg_name.format(component_name=instance.root_cfg.name)

        instance.ci_actions[self._register_params["kind"]][reg_name] = CiActionData(
            graph=graph,
            action_parameters=self._customize_register_params(instance, graph),
        )

        # LOGGER.info("Added to %s flows", self._register_params["kind"])

    def __call__(self, instance, **kwargs):
        return self._get_graph(instance, **kwargs)

    def register(self, instance):

        for call_kwargs in self._iter_parametrized():

            graph = self._get_graph(instance, **call_kwargs)

            if graph is None:
                continue

            self._register_graph(instance, graph)


class FlowDeclarationWithBranchesWrapper(FlowDeclarationWrapper):

    def _branches(self, instance):
        return ActionBranches(
            pattern=instance.root_cfg.svn_cfg.arc_branch_path("${version}"),
            auto_create=True,
            forbid_trunk_releases=False,
            independent_stages=instance.root_cfg.releases_cfg.main_release_flow_independent_stages,
            auto=instance.root_cfg.releases_cfg.main_release_flow_branch_auto_start,
        )

    def _customize_register_params(self, instance, graph):

        filters = []

        if instance.root_cfg.ci_cfg.ya_make_abs_paths_glob:
            filters.append({
                "discovery": "any",
                "abs-paths": instance.root_cfg.ci_cfg.ya_make_abs_paths_glob,
            })

        if instance.root_cfg.ci_cfg.observed_sub_paths:
            filters.append({
                "sub-paths": instance.root_cfg.ci_cfg.observed_sub_paths,
            })

        return ActionParameters(
            auto=self._register_params["auto"],
            triggers=self._register_params["triggers"],
            title=self._register_params["title"],
            description=self._register_params["description"],
            branches=self._branches(instance),
            filters=filters,
            stages=self._register_params["stages"],
            start_version=instance.root_cfg.svn_cfg.start_version or None,
        )


def register_flow__decorate_with(f, registration_kwargs, wrapper_class=FlowDeclarationWrapper):
    # LOGGER.info("Registering new flow: %s", f.__name__)

    wrapper = wrapper_class(f, **registration_kwargs)
    wrapper = functools.wraps(f)(wrapper)
    wrapper.__defines_flow = True

    return wrapper


def register_flow(**kwargs):

    def decorator(f):
        return register_flow__decorate_with(f, kwargs)

    return decorator


def register_release_flow(with_branches=True, **kwargs):

    def decorator(f):
        if with_branches:
            return register_flow__decorate_with(f, kwargs, FlowDeclarationWithBranchesWrapper)
        else:
            return register_flow__decorate_with(f, kwargs)

    return decorator


release_flow = functools.partial(
    register_release_flow,
    kind="release",
    name_override="release_{component_name}",
)

main_release_flow = release_flow  # a better name for this decorator

action_with_branches = functools.partial(
    register_release_flow,
    kind="action",
)

supplementary_release_flow = functools.partial(
    register_release_flow,
    with_branches=False,
    kind="release",
)


def get_register_precommit_check_decorator(required):
    return functools.partial(
        register_flow,
        triggers=[ActionTrigger(on=ActionTriggerOn.PR, into="trunk", required=required)],
    )


precommit_check_flow = get_register_precommit_check_decorator(False)
precommit_check_flow_required = get_register_precommit_check_decorator(True)
