import six
import typing  # noqa

from sandbox.projects.release_machine.components.config_core.jg.cube import base as cube_base


class Graph(object):

    def __init__(self, cubes=None, resolve_dependencies=False):
        # type: ('Graph', typing.Iterable[cube_base.Cube], bool) -> None
        """
        Release Machine job graph base class

        :param cubes:
            An iterable of cubes

        :param resolve_dependencies:
            Whether or not to automatically resolve cube dependencies. If True then for each cube being added to
            the graph a dependency resolver is run. The resolver loops though the cube's dependencies, adds them
            to the graph and runs itself on each of the dependency recursively
        """

        self._cubes = {}
        self._resolve_dependencies = resolve_dependencies

        cubes = cubes or []

        for cube in cubes:
            self.add(cube)

    @property
    def all_cubes_iter(self):
        return six.itervalues(self._cubes)

    @property
    def all_cubes_list(self):
        return list(self.all_cubes_iter)

    @property
    def all_stages_set(self):

        result = set()

        for cube in self.all_cubes_iter:
            if cube.stage:
                result.add(cube.stage)

        return result

    def add(self, cube):  # type: ('Graph', cube_base.Cube) -> None
        if not isinstance(cube, cube_base.Cube):
            raise TypeError("Unexpected cube type {}".format(type(cube)))

        cube.set_name(all_names={c.name for c in self.all_cubes_iter})
        self._cubes[cube.name] = cube

        if self._resolve_dependencies:
            self.populate_with_cube_requirements(cube)

    def get(self, cube_name):  # type: ('Graph', six.string_types) -> cube_base.Cube
        try:
            return self._cubes[cube_name]
        except KeyError:
            raise KeyError(
                "Cube named '{}' cannot be found in the given graph. Available cubes: {}".format(
                    cube_name,
                    [cube.name for cube in self.all_cubes_iter],
                ),
            )

    def populate_with_requirements(self):
        """Populate self adding all cubes' requirements recursively"""

        current_cubes = self.all_cubes_list

        for cube in current_cubes:
            self.populate_with_cube_requirements(cube)

    def populate_with_cube_requirements(self, cube):
        """Populate self with all requirements of the given cube (and their requirements recursively)"""

        for needed_cube in cube.requirements:

            if needed_cube in self:
                continue

            self.add(needed_cube)
            self.populate_with_cube_requirements(needed_cube)

    def __contains__(self, item):

        if isinstance(item, six.string_types):
            return item in {cube.name for cube in self.all_cubes_iter}

        return item in self.all_cubes_list

    def __getitem__(self, item):
        return self.get(item)

    def __len__(self):
        return len(self._cubes)

    def __add__(self, other):
        if not isinstance(other, self.__class__):
            raise TypeError("Expected {} instance, got {}".format(
                self.__class__.__name__,
                type(other),
            ))
        return self.__class__(self.all_cubes_list + other.all_cubes_list)

    def to_dict(self):

        result = {}

        for cube in self.all_cubes_iter:
            result.update(cube.to_dict())

        return result
