import typing  # noqa
import collections
import re
import six


CallChainItem = collections.namedtuple("CallChainItem", ["name", "action"])
CALL_CHAIN_ACTION_GETATTR = "getattr"
CALL_CHAIN_ACTION_GETITEM = "getitem"


class CallChainMem(object):

    SPECIAL_NAME__RESOURCES = "resources"
    RESOURCE_TYPE_REGEX = re.compile("^[A-Z_]+$")

    def __init__(self, call_chain=None):
        self._call_chain = call_chain or []

    @property
    def call_chain(self):
        return self._call_chain

    @property
    def jmespath(self):

        if not self._call_chain:
            return ""

        prev_name, prev_action = self._call_chain[0]
        result = prev_name

        for item in self._call_chain[1:]:

            name, action = item

            if (
                prev_name == self.SPECIAL_NAME__RESOURCES and
                prev_action is CALL_CHAIN_ACTION_GETATTR and
                action is CALL_CHAIN_ACTION_GETITEM and
                isinstance(name, six.string_types) and
                self.RESOURCE_TYPE_REGEX.match(name)
            ):
                name = "?type == '{}'".format(name)

            if action == CALL_CHAIN_ACTION_GETATTR:
                result = "{}.{}".format(result, name)
            elif action == CALL_CHAIN_ACTION_GETITEM:
                result = "{} | [{}]".format(result, name)
            else:
                raise TypeError(
                    "Unexpected type {} found in JGE output call chain {}".format(
                        type(item),
                        self._call_chain,
                    ),
                )

            prev_name, prev_action = name, action

        return result

    def __getattr__(self, item):
        return self.__class__(self._call_chain + [CallChainItem(item, CALL_CHAIN_ACTION_GETATTR)])

    def __getitem__(self, item):
        return self.__class__(self._call_chain + [CallChainItem(item, CALL_CHAIN_ACTION_GETITEM)])

    def __dir__(self):
        return dir(super(self.__class__, self)) + ["_call_chain", "jmespath"]
