import six
import typing  # noqa

from sandbox.projects.release_machine.components.config_core.jg.cube import base as cube_base


class CICondition(cube_base.ICubeCondition):
    """
    Base class for CI conditions
    """

    IF = "if"

    def __init__(self, value):
        if isinstance(value, six.string_types):
            self._value = value
        elif isinstance(value, cube_base.CubeOutput):
            self._value = cube_base.CubeInput.format_cube_output_value(value, enclose_in_braces=False)
        elif isinstance(value, CICondition):
            self._value = value.value
        else:
            self._value = "{}".format(value)

    @property
    def value(self):
        return self._value

    def __str__(self):
        return "({})".format(self._value)

    def __repr__(self):
        return str(self)

    def join_with_operator(self, other, operator, quote_strings=True):
        """
        Join `self` with `other` using the given operator (:param operator:).
        The result is an instance of CICondition

        :type other: typing.Union[str, unicode, bytes, int, float, cube_base.CubeOutput, NoneType, CICondition]
        :param other:
            An object to join the condition with

        :type operator: six.string_types
        :param operator:
            Join operator (e.g., '==', '>', '&&', etc.)

        :type quote_strings: bool
        :param quote_strings:
            Whether to quote :param other: if :param other: is a string

        :return:
            Returns an instance of CICondition

        :raises TypeError:
            Raised if the type of :param other: is not supported
        """

        if isinstance(other, six.string_types):
            other = "'{}'".format(other) if quote_strings else "{}".format(other)
        elif isinstance(other, bool):
            # Note that `bool` is a subclass of `int`. This implies that we should treat boolean values separately
            # before the `isinstance(other, int)` check.
            raise TypeError(
                "Rhs of type 'bool' is not supported with {} objects. Use the object itself or object's "
                "`.negate()` method instead".format(
                    self.__class__.__name__,
                ),
            )
        elif isinstance(other, int) or isinstance(other, float):
            other = "`{}`".format(other)
        elif isinstance(other, CICondition):
            other = str(other)
        elif isinstance(other, cube_base.CubeOutput):
            other = cube_base.CubeInput.format_cube_output_value(other, enclose_in_braces=False)
        elif other is None:
            return self.join_with_operator("null", operator, quote_strings=False)
        else:
            raise TypeError("Type {} is not supported in {} rhs".format(type(other), self.__class__.__name__))

        return CICondition("{lhs} {op} {rhs}".format(lhs=self, op=operator, rhs=other))

    def __eq__(self, other):
        return self.join_with_operator(other, "==")

    def __ne__(self, other):
        return self.join_with_operator(other, "!=")

    def __gt__(self, other):
        return self.join_with_operator(other, ">")

    def __lt__(self, other):
        return self.join_with_operator(other, "<")

    def __ge__(self, other):
        return self.join_with_operator(other, ">=")

    def __le__(self, other):
        return self.join_with_operator(other, "<=")

    def __or__(self, other):
        return self.join_with_operator(other, "||")

    def __and__(self, other):
        return self.join_with_operator(other, "&&")

    def negate(self):
        return CICondition("!{}".format(self))

    def not_null(self, default):
        return CICondition("not_null({}, {})".format(self, default))

    def length(self):
        return CICondition("length({})".format(self))

    def to_dict(self):
        return {
            self.IF: "${{{}}}".format(self._value),
        }


CI_FIRST_TAG_ONLY = CICondition("not_null(context.version_info.minor, `0`)") == 0
