import logging
import collections

import six

from sandbox.projects.release_machine.components.config_core.jg import base as jg_base
from sandbox.projects.release_machine.components.config_core.jg.cube import base as jg_cube
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import build as build_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import release as release_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import internal as internal_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import yappy as yappy_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import dummy as dummy_cubes
from sandbox.projects.release_machine.components.config_core.jg import flow as jg_flow
from sandbox.projects.release_machine.components.config_core.jg.preset import exceptions as preset_exceptions

from sandbox.projects.release_machine.core import const as rm_const

LOGGER = logging.getLogger(__name__)


DEFAULT_STAGE_NEW_TAG = jg_flow.ReleaseActionStageData("new_tag", cube_names=[internal_cubes.NewTagCube.NAME])
DEFAULT_STAGE_BUILD = jg_flow.ReleaseActionStageData(
    "build",
    cube_names=[
        dummy_cubes.RMMainGraphEntry.NAME,
    ],
    cube_types=[
        build_cubes.TYPE_BUILD,
        internal_cubes.TYPE_CHANGELOG,
        internal_cubes.TYPE_STARTREK,
    ],
)
DEFAULT_STAGE_TEST = jg_flow.ReleaseActionStageData(
    "test",
    cube_names=[
        dummy_cubes.TestStageEntry.NAME,
    ],
    cube_types=[
        yappy_cubes.TYPE_GENERATE_BETA,
        "test",
    ],
)

DEFAULT_STAGE_RELEASE = jg_flow.ReleaseActionStageData(
    "release",
    cube_names=[
        dummy_cubes.ReleaseStageEntry.NAME,
    ],
    cube_types=[
        release_cubes.TYPE_RELEASE,
    ],
    rollback=True,
)

DEFAULT_STAGE_PRESTABLE_RELEASE = jg_flow.ReleaseActionStageData(
    "prestable",
    cube_names=[
        dummy_cubes.ReleasePrestableStageEntry.NAME,
    ],
    cube_types=[
        release_cubes.TYPE_PRESTABLE_RELEASE,
    ],
    rollback=True,
)

DEFAULT_STAGE_STABLE_RELEASE = jg_flow.ReleaseActionStageData(
    "stable",
    cube_names=[
        dummy_cubes.ReleaseStableStageEntry.NAME,
    ],
    cube_types=[
        release_cubes.TYPE_STABLE_RELEASE,
    ],
    rollback=True,
)


DEFAULT_RELEASE_FLOW_STAGES = [
    DEFAULT_STAGE_NEW_TAG,
    DEFAULT_STAGE_BUILD,
    DEFAULT_STAGE_TEST,
    DEFAULT_STAGE_RELEASE,
]


DEFAULT_STAGE_BUILD_AND_TEST = jg_flow.ReleaseActionStageData(
    "build_and_test",
    title="Build & Test",
    cube_names=[
        dummy_cubes.RMMainGraphEntry.NAME,
    ],
    cube_types=[
        build_cubes.TYPE_BUILD,
        internal_cubes.TYPE_CHANGELOG,
        internal_cubes.TYPE_STARTREK,
        yappy_cubes.TYPE_GENERATE_BETA,
        internal_cubes.TYPE_TEST,
    ],
)


JOINED_BUILD_RELEASE_FLOW_STAGES = [
    DEFAULT_STAGE_NEW_TAG,
    DEFAULT_STAGE_BUILD_AND_TEST,
    DEFAULT_STAGE_RELEASE,
]


class SingleBuildGeneralJGCfg(jg_base.BaseReleaseMachineJobGraph):
    """
                                                 [?]
    new tag --+---------> build ----------> generate beta ----------> release ...
              |                                                  |
              +----> changelog                                   +--> release  ...
              |          |                   [?]
              |          +--------> post changelog to release ticket
              |          |                   [?]
              |          +--------> create wiki page with changelog
              |          | [?]                         [?]
              +----> create st ticket -----> link feature tickets
    """

    def __init__(self, *args, **kwargs):
        self._suitable_releasable_items = None
        super(SingleBuildGeneralJGCfg, self).__init__(*args, **kwargs)

    @property
    def build_task(self):
        raise NotImplementedError

    @property
    def build_cube_name(self):
        return "build"

    @property
    def release_task(self):
        return "RELEASE_RM_COMPONENT_2"

    @property
    def add_beta_generator(self):
        return False

    @property
    def release_manually(self):
        """When True all release cubes are configured to require manual run"""
        return True

    @property
    def restrict_release_stages_to(self):
        """A set of release stages that should be used in the release graph. None means 'no restrictions'"""
        return None

    @property
    def suitable_releasable_items(self):

        if not self._suitable_releasable_items:
            self._suitable_releasable_items = self._get_suitable_releasable_items()

        return self._suitable_releasable_items

    def _check_releasable_item_is_suitable(self, ri):
        return hasattr(ri.data, "resource_type")

    def _get_suitable_releasable_items(self):
        if not self.root_cfg.releases_cfg.releasable_items:
            raise preset_exceptions.JGPresetUsageError(
                "{class_name} preset cannot be used without properly configured Releases.releasable_items".format(
                    class_name=self.__class__.__name__,
                ),
            )

        suitable_releasable_items = [
            ri for ri in self.root_cfg.releases_cfg.releasable_items
            if self._check_releasable_item_is_suitable(ri)
        ]

        # LOGGER.info("%s suitable releasable items found: %s", len(suitable_releasable_items), suitable_releasable_items)

        return suitable_releasable_items

    def _get_build_cube(self, graph):
        raise NotImplementedError

    def _get_generate_yappy_beta_cubes(self, graph, build):

        if not self.add_beta_generator:
            # LOGGER.info("add_beta_generator = False => no yappy generator is going to be added")
            return []

        if isinstance(self.add_beta_generator, bool):
            beta_keys = [key for key in self.root_cfg.yappy_cfg.betas]
        elif isinstance(self.add_beta_generator, six.string_types):
            beta_keys = [self.add_beta_generator]
        else:
            beta_keys = self.add_beta_generator

        if not self.root_cfg.yappy_cfg.betas:
            raise preset_exceptions.JGPresetUsageError(
                "add_beta_generator is set to True but no suitable Yappy configuration provided. Please provide one "
                "in the Yappy section of {}'s config file".format(self.root_cfg.name),
            )

        if not build:
            raise preset_exceptions.JGPresetUsageError(
                "In order to add a GenerateYappyBeta cube you should add a build cube first. No cube with name "
                "'build' can be found among graph cubes {}".format(
                    ", ".join([str(cube) for cube in graph.all_cubes_iter]),
                ),
            )

        result = []

        for beta_conf_type in beta_keys:

            result.append(
                yappy_cubes.GenerateYappyBeta(
                    component_name=self.component_name,
                    beta_conf_type=beta_conf_type,
                    input=jg_cube.CubeInput(
                        component_resources={
                            ri.name: build.output.resources[ri.data.resource_type].first().id
                            for ri in self.suitable_releasable_items
                        },
                    ),
                    needs=[build],
                ),
            )

        return result

    def _release_stage_allowed(self, stage):
        return self.restrict_release_stages_to is None or stage in self.restrict_release_stages_to

    def _iter_over_releasable_items_with_release_stages_and_deploy_systems(self, build):
        releasable_items_by_deploy_system = collections.defaultdict(lambda: collections.defaultdict(list))

        for ri in self.suitable_releasable_items:

            for di in ri.deploy_infos:

                if not self._release_stage_allowed(di.stage):
                    continue

                releasable_items_by_deploy_system[di.stage][di.deploy_system.name].append(ri)

        releasable_items_by_deploy_system = dict(releasable_items_by_deploy_system)

        for where in releasable_items_by_deploy_system:

            for deploy_system in releasable_items_by_deploy_system[where]:

                yield deploy_system, where, releasable_items_by_deploy_system[where][deploy_system]

    def _get_release_cubes(self, graph, build):

        result = []

        for deploy_system, where, releasable_items in (
            self._iter_over_releasable_items_with_release_stages_and_deploy_systems(build)
        ):
            result.extend(
                self._get_release_cubes_for_deploy_system(deploy_system, where, releasable_items, graph, build)
            )

        return result

    def _get_release_cubes_for_deploy_system(self, deploy_system, where, releasable_items, graph, build):
        return [
            release_cubes.ReleaseRmComponent2(
                name="release_{}_{}".format(where, deploy_system),
                component_name=self.component_name,
                task=self.release_task,
                where_to_release=where,
                input=jg_cube.CubeInput(
                    component_resources={
                        ri.name: build.output.resources[ri.data.resource_type].first().id
                        for ri in releasable_items
                    },
                    deploy_system=deploy_system,
                ),
                manual=self.release_manually,
            )
        ]

    def _get_release_stage_entry_dummy_cube(self, graph, build):

        return dummy_cubes.ReleaseStageEntry(
            needs=[
                build,
            ],
        )

    def _get_test_stage_entry_dummy_cube(self, graph, build):

        return dummy_cubes.TestStageEntry(
            needs=[
                build,
            ],
        )

    @jg_flow.release_flow(stages=DEFAULT_RELEASE_FLOW_STAGES)
    def release(self):
        graph = super(SingleBuildGeneralJGCfg, self).release(self)

        build = self._get_build_cube(graph)
        gyb_cubes = self._get_generate_yappy_beta_cubes(graph, build)
        release_entry = self._get_release_stage_entry_dummy_cube(graph, build)
        releases = self._get_release_cubes(graph, build)

        graph.add(build)

        if gyb_cubes:
            the_stage_after_build_entry = self._get_test_stage_entry_dummy_cube(graph, build)
            graph.add(the_stage_after_build_entry)
            graph.add(release_entry)
        else:
            the_stage_after_build_entry = release_entry
            graph.add(release_entry)

        for gyb_cube in gyb_cubes:
            gyb_cube.add_requirement(the_stage_after_build_entry)
            graph.add(gyb_cube)
            release_entry.add_requirement(gyb_cube)

        if internal_cubes.CreateChangelog.NAME in graph:
            the_stage_after_build_entry.add_requirement(
                graph.get(internal_cubes.CreateChangelog.NAME),
            )

        if internal_cubes.LinkFeatureTickets.NAME in graph:
            the_stage_after_build_entry.add_requirement(
                graph.get(internal_cubes.LinkFeatureTickets.NAME),
            )

        if internal_cubes.PostChangelogToStartrek.NAME in graph:
            the_stage_after_build_entry.add_requirement(
                graph.get(internal_cubes.PostChangelogToStartrek.NAME),
            )

        if internal_cubes.CreateWikiPageWithChangelog.NAME in graph:
            the_stage_after_build_entry.add_requirement(
                graph.get(internal_cubes.CreateWikiPageWithChangelog.NAME),
            )

        for release in releases:
            release.add_requirement(release_entry)
            graph.add(release)

        return graph


class SingleBuildYaMakeJGCfg(SingleBuildGeneralJGCfg):

    @property
    def build_task(self):
        return "YA_MAKE_2"

    def _get_build_cube(self, graph):

        targets = []
        artifacts = []
        source_artifacts = []
        resource_types = []

        for ri in self.suitable_releasable_items:
            targets.append(ri.build_data.target)
            if ri.build_data.is_source:
                source_artifacts.append(ri.build_data.artifact)
            else:
                artifacts.append(ri.build_data.artifact)
            resource_types.append(ri.data.resource_type)

        return build_cubes.YaMakeBuildCubeBase(
            name=self.build_cube_name,
            task=self.build_task,
            targets=targets,
            artifacts=artifacts,
            source_artifacts=source_artifacts,
            resource_types=resource_types,
            input=jg_cube.CubeInput(
                result_attrs={
                    "major_release_num": rm_const.CIJMESPathCommon.MAJOR_RELEASE_NUM,
                    "minor_release_num": rm_const.CIJMESPathCommon.MINOR_RELEASE_NUM,
                },
            ),
            needs=[
                graph.get(dummy_cubes.RMMainGraphEntry.NAME),
            ],
        )

    def _check_releasable_item_is_suitable(self, ri):
        return hasattr(ri, "build_data") and ri.build_data and hasattr(ri.data, "resource_type")


class SingleBuildKosherYaMakeJGCfg(SingleBuildYaMakeJGCfg):
    @property
    def build_task(self):
        return "KOSHER_YA_MAKE"


class SingleBuildYaMakeTemplateJGCfg(SingleBuildYaMakeJGCfg):

    @property
    def build_task(self):
        raise NotImplementedError

    def _check_releasable_item_is_suitable(self, ri):
        return hasattr(ri.data, "resource_type")

    def _get_build_cube(self, graph):
        return build_cubes.YaMakeTemplate(
            target_resources=[ri.data.resource_type for ri in self._get_suitable_releasable_items()],
            name="build",
            task=self.build_task,
            needs=[
                graph.get(dummy_cubes.RMMainGraphEntry.NAME),
            ],
        )


class SingleBuildYaPackageJGCfg(SingleBuildYaMakeJGCfg):

    @property
    def build_task(self):
        return "YA_PACKAGE_2"

    @property
    def build_use_compression(self):
        return True

    @property
    def set_custom_version(self):
        return False

    @property
    def custom_version_value(self):
        return "{}.{}".format(
            rm_const.CIJMESPathCommon.MAJOR_RELEASE_NUM,
            rm_const.CIJMESPathCommon.MINOR_RELEASE_NUM,
        )

    def _get_build_cube(self, graph):

        packages = []
        resource_types = []

        for ri in self.suitable_releasable_items:
            packages.append(ri.build_data.target)
            resource_types.append(ri.data.resource_type)

        result = build_cubes.YaPackageBuildCubeBase(
            packages=packages,
            resource_types=resource_types,
            use_compression=self.build_use_compression,
            task=self.build_task,
            needs=[
                graph.get(dummy_cubes.RMMainGraphEntry.NAME),
            ],
        )

        if self.set_custom_version:
            result.input.update(custom_version=self.custom_version_value)

        return result


class SingleBuildYaPackageInheritorJGCfg(SingleBuildYaPackageJGCfg):
    """
    Single-build preset for a release process with a custom build task derived from YaPackage.
    The build task should define default packages. This preset will not provide any package info.
    Requires `build_task` to be overridden
    """

    @property
    def build_task(self):
        raise NotImplementedError

    def _check_releasable_item_is_suitable(self, ri):
        return hasattr(ri.data, "resource_type")

    def _get_build_cube(self, graph):

        resource_types = []

        for ri in self.suitable_releasable_items:
            resource_types.append(ri.data.resource_type)

        return build_cubes.YaPackageBuildCubeBase(
            packages=[],
            resource_types=resource_types,
            use_compression=self.build_use_compression,
            task=self.build_task,
            needs=[
                graph.get(dummy_cubes.RMMainGraphEntry.NAME),
            ],
        )


class TaskletBuildPreset(SingleBuildGeneralJGCfg):
    """
    Build and release a single tasklet or tasklet bundle.
    """

    @property
    def build_task(self):
        return build_cubes.RCBuildTasklet.TASK

    @property
    def release_task(self):
        return release_cubes.TaskletRelease.TASK

    @property
    def the_only_releasable_item(self):
        return self.suitable_releasable_items[0]

    @property
    def the_only_deploy_info(self):
        return self.the_only_releasable_item.deploy_infos[0]

    @property
    def release_ticket(self):
        return ""

    def _check_releasable_item_is_suitable(self, ri):
        return bool(ri.build_data)

    def _get_build_cube(self, graph):

        return build_cubes.RCBuildTasklet(
            tasklet_path=self.the_only_releasable_item.build_data.target,
            tasklet_binary_path=self.the_only_releasable_item.build_data.artifact,
            sandbox_owner=self.root_cfg.ci_cfg.sb_owner_group,
            needs=[
                graph.get(dummy_cubes.RMMainGraphEntry.NAME),
            ],
        )

    def _get_release_cubes_for_deploy_system(self, deploy_system, where, releasable_items, graph, build):
        return [
            release_cubes.TaskletRelease(
                component_name=self.root_cfg.name,
                where_to_release=where,
                tasklet_version_infos=[
                    {
                        "registry_path": registry_path,
                        "stage": where,
                        "resource_id": build.output.result_output["?path == 'result_resource_id'"].string[0][0],
                    } for registry_path in self.the_only_deploy_info.registry_paths
                ],
                do_commit=True,
                commit_message="{component_name}: release tasklet to {where}. "
                               "Version: {version} ({commit}) {ticket}".format(
                    component_name=self.component_name,
                    where=where,
                    version="${context.version}",
                    commit="${context.target_revision.number}",
                    ticket=self.release_ticket,
                ),
                manual=False,
            ),
        ]

    def _get_generate_yappy_beta_cubes(self, graph, build):
        return []

    @jg_flow.release_flow(stages=DEFAULT_RELEASE_FLOW_STAGES, auto=True)
    def release(self):
        return super(TaskletBuildPreset, self).release(self)
