"""
This module provides classes for notification declarative configuration for the new RM notification system
(see https://wiki.yandex-team.ru/releasemachine/notifications/#notifications-2-1)

Declarative notifications live alongside notifications configured in RM UI and are NOT editable in RM UI (remaining
visible though). Note that declarative notifications are always active. Thus the only way to turn off a notification
declared in component's config file is to remove it from there and commit the changes. Remember also that there
might be a significant lag between a commit and the time the changes are actually applied to notification system.
"""

import sys


class Protobufable(object):
    """
    An abstract class with general `to_protobuf` implementation used as an interface for classes below
    """

    def get_protobuf_message(self, message_pb2, table_pb2):
        raise NotImplementedError

    def to_protobuf(self):
        if 'message_pb2' not in sys.modules:
            from release_machine.release_machine.proto.structures import message_pb2
        if 'table_pb2' not in sys.modules:
            from release_machine.release_machine.proto.structures import table_pb2
        return self.get_protobuf_message(message_pb2, table_pb2)


class NotificationConditionItem(Protobufable):
    """
    An layer class for `release_machine.release_machine.proto.structures.table_pb2.NotificationCondition`
    """

    def __init__(self, field, operator, value, negate=False):
        """
        :param field: str, field reference
        :param operator: str, `table_pb2.NotificationConditionItem.ConditionOperatorType`
        :param value: str, the `field`'s value is compared with this value using `operator` as a comparator
        :param negate: bool, negate condition
        """
        self._field = field
        self._operator = operator
        self._value = value
        self._negate = negate

    @property
    def field(self):
        return self._field

    @property
    def operator(self):
        return self._operator

    @property
    def value(self):
        return self._value

    @property
    def negate(self):
        return self._negate

    def get_protobuf_message(self, message_pb2, table_pb2):
        return table_pb2.NotificationConditionItem(
            operator=self.operator,
            field=self.field,
            value=self.value,
            negate=self._negate,
        )


class NotificationCondition(Protobufable):
    """
    An layer class for `release_machine.release_machine.proto.structures.table_pb2.NotificationCondition`
    """

    def __init__(self, conditions, join_strategy):
        """
        :param conditions: a list of `NotificationConditionItem` objects
        :param join_strategy: str, `table_pb2.NotificationCondition.ConditionJoinStrategy`
        """
        self._conditions = conditions
        self._join_strategy = join_strategy

    @property
    def conditions(self):
        return self._conditions

    @property
    def join_strategy(self):
        return self._join_strategy

    def get_protobuf_message(self, message_pb2, table_pb2):
        return table_pb2.NotificationCondition(
            join_strategy=self.join_strategy,
            conditions=[condition.get_protobuf_message(message_pb2, table_pb2) for condition in self.conditions]
        )


class Notification(Protobufable):
    """
    An layer class for `release_machine.release_machine.proto.structures.message_pb2.NotificationConfig`
    """

    def __init__(self, event_type, chat_name, conditions, message_template_file='', tags='', pin=False):
        """
        :param event_type:, str, `table_pb2.TypeOfEvent`
        :param chat_name: str, name of a chat to which notification should be sent to
        :param conditions: `NotificationCondition` object
        :param message_template_file: message template file path
        """
        self._event_type = event_type
        self._chat_name = chat_name
        self._conditions = conditions
        self._message_template_file = message_template_file
        self._tags = tags
        self._pin = pin

    @property
    def event_type(self):
        return self._event_type

    @property
    def chat_name(self):
        return self._chat_name

    @property
    def conditions(self):
        return self._conditions

    @property
    def message_template_file(self):
        return self._message_template_file

    @property
    def tags(self):
        return self._tags

    @property
    def pin(self):
        return self._pin

    def get_protobuf_message(self, message_pb2, table_pb2):
        return message_pb2.NotificationConfig(
            component_name="",
            event_type=self.event_type,
            chat=message_pb2.ComponentChat(
                name=self.chat_name,
            ),
            condition_new=self.conditions.get_protobuf_message(message_pb2, table_pb2),
            message_template_file=self.message_template_file,
            active=True,
            tags=self.tags,
            pin=self.pin,
        )


JOB_EVENTS = [
    "AcceptanceTest",
    "BuildTest",
    "GenericTest",
    "GenericDiff",
    "NewBetaGeneration",
]

TASK_BREAK_STATUSES = ["NO_RES", "EXCEPTION", "TIMEOUT", "EXPIRED", "STOPPED"]
TASK_PROBLEM_STATUSES = ["FAILURE"] + TASK_BREAK_STATUSES
TASK_SUCCESS_STATUSES = ["SUCCESS"]


TASK_BREAK_CONDITIONS = [
    NotificationConditionItem(
        field="task_data.status",
        operator="IN",
        value=",".join(TASK_BREAK_STATUSES),
    )
]

TASK_PROBLEM_CONDITIONS = [
    NotificationConditionItem(
        field="task_data.status",
        operator="IN",
        value=",".join(TASK_PROBLEM_STATUSES),
    )
]

TASK_SUCCESS_CONDITIONS = [
    NotificationConditionItem(
        field="task_data.status",
        operator="TEXT_EXACTLY_IS",
        value=status,
    ) for status in TASK_SUCCESS_STATUSES
]

TASK_SUCCESS_AND_PROBLEM_CONDITIONS = TASK_SUCCESS_CONDITIONS + TASK_PROBLEM_CONDITIONS

STARTREK_TICKET_CREATED_CONDITIONS = [
    NotificationConditionItem(
        field="ticket_history_data.ticket_history_latest_status",
        operator="TEXT_EXACTLY_IS",
        value="open",
    )
]

CONDITION_ALWAYS = NotificationCondition(
    conditions=[],
    join_strategy="OR",
)


def build_notifications_for_any_test_problems(chat_name):
    return [
        Notification(
            event_type=event_type,
            chat_name=chat_name,
            conditions=NotificationCondition(
                conditions=[
                    NotificationConditionItem(
                        field="task_data.status",
                        operator="IN",
                        value=",".join(TASK_PROBLEM_STATUSES),
                    ),
                ],
                join_strategy="OR",
            ),
        ) for event_type in JOB_EVENTS
    ]
