import logging
import typing

from six import string_types
from sandbox.projects.release_machine.components.config_core import responsibility


LOGGER = logging.getLogger(__name__)
APPROVEMENT_DESCRIPTION_DEFAULT = "Release {tag_name} approvement required"


class ReleaseApprovementStage(object):

    NEED_ALL_KEY = "need_all"
    STAGES_KEY = "stages"
    APPROVER_KEY = "approver"

    def __init__(self, stage, need_all=False):
        """
        A stage of release approvement

        :param stage: either an appropver login or a responsibility.Responsible object
        or a list of ReleaseApprovementStage objects
        :type stage: typing.Union[str,responsibility.Responsible,typing.List[typing.Union[ReleaseApprovementStage]]]

        :param need_all: need all of the approvers to approve or just any of them
        :type need_all: bool
        """
        self._stage = stage
        self._need_all = need_all

    @property
    def stage(self):
        return self._stage

    @property
    def need_all(self):
        return self._need_all

    def is_simple(self):
        return isinstance(self._stage, string_types) or isinstance(self._stage, responsibility.Responsible)

    def __repr__(self):

        if self.is_simple():
            return "<{cls_name} {approver}@>".format(
                cls_name=self.__class__.__name__,
                approver=str(self._stage),
            )

        return "<{cls_name} {stage} all={need_all}>".format(
            cls_name=self.__class__.__name__,
            stage=repr(self._stage),
            need_all=self._need_all,
        )

    def __str__(self):
        return repr(self)


STAGE_TYPE = typing.Union[
    str,
    responsibility.Responsible,
    ReleaseApprovementStage,
]


class ReleaseApprovementsSettings(object):

    def __init__(
        self,
        description=APPROVEMENT_DESCRIPTION_DEFAULT,
        stages=None,
        is_parallel=True,
        first_tag_only=False,
    ):
        """
        https://wiki.yandex-team.ru/Intranet/OK/

        https://wiki.yandex-team.ru/Intranet/OK/private-api/#sozdanieizapusksoglasovanija

        :param description: approvement description
        :type description: string_types

        :param stages: list of ReleaseApprovementStage
        :type stages: typing.List[STAGE_TYPE]

        :param is_parallel: whether the approvement should be parallel or not
        :type is_parallel: bool

        :param first_tag_only: (branched components only) create approvements only for the 1st tag of each branch
        :type first_tag_only: bool
        """

        self._description = description
        self._stages = stages
        self._is_parallel = is_parallel
        self._first_tag_only = first_tag_only

    @property
    def description(self):
        return self._description

    @property
    def stages(self):
        return self._stages

    @property
    def is_parallel(self):
        return self._is_parallel

    @property
    def first_tag_only(self):
        return self._first_tag_only

    def __repr__(self):
        return "<{cls_name} {stages} parallel={parallel}>".format(
            cls_name=self.__class__.__name__,
            stages=[repr(stage) for stage in self._stages if stage],
            parallel=self._is_parallel,
        )

    def __str__(self):
        return repr(self)
