# -*- coding: utf-8 -*-
import fnmatch
import re

from functools import total_ordering

import six
from six import string_types
from typing import Union, Optional, List, Type  # noqa: UnusedImport


REGEX_TYPE = type(re.compile(''))


@total_ordering
class TestStatus(object):

    ALL = dict()

    def __new__(cls, order=None, name=''):

        if name not in cls.ALL:
            cls.ALL[name] = super(TestStatus, cls).__new__(cls)

        return cls.ALL[name]

    def __init__(self, order=None, name=''):
        self._order = order
        self._name = name

    @property
    def order(self):
        return self._order

    @property
    def name(self):
        return self._name

    def __lt__(self, other):
        return self._order < other.order

    def __eq__(self, other):
        return self._order == other.order

    def __str__(self):
        return self._name

    def __repr__(self):
        return self._name


# The below should always be in sync with release_machine/common_proto/test_results.proto
ONGOING = TestStatus(order=0, name='ONGOING')
UB = TestStatus(order=1, name='UB')
CRIT = TestStatus(order=2, name='CRIT')
WARN = TestStatus(order=3, name='WARN')
OK = TestStatus(order=4, name='OK')

KNOWN_STATUSES = {
    ONGOING.name: ONGOING,
    UB.name: UB,
    CRIT.name: CRIT,
    WARN.name: WARN,
    OK.name: OK,
}


def convert_test_status(test_status):

    from release_machine.common_proto import test_results_pb2

    if isinstance(test_status, six.string_types):
        return KNOWN_STATUSES[test_status]

    if isinstance(test_status, int):
        return KNOWN_STATUSES[test_results_pb2.TestResult.TestStatus.Name(test_status)]


def get_test_status_min(*test_statuses):

    test_statuses = [convert_test_status(ts) for ts in test_statuses]

    return min(test_statuses)


class BlockConfBase(object):
    """

    >>> REQUIRE_ALL_OK.should_block({"T1": OK, "T2": OK})
    False
    >>> REQUIRE_ALL_OK.should_block({"T1": OK, "T2": CRIT})
    True
    >>> REQUIRE_ALL_OK.should_block({"T1": OK, "T2": WARN})
    True
    >>> block_conf(name_filter='TEST_*', accept_result_threshold=OK).should_block(
    ...     {"TEST_SOMETHING": OK, "BUILD_SOMETHING": CRIT},
    ... )
    False
    >>> block_conf(name_filter='TEST_*', accept_result_threshold=OK).should_block(
    ...     {"TEST_SOMETHING": OK, "TEST_SOMETHING_ELSE": CRIT},
    ... )
    True
    >>> block_conf(
    ...     name_filter=re.compile(r'TEST_[A-Z_]+_IMPORTANT'), accept_result_threshold=OK
    ... ).should_block(
    ...     {"TEST_SOMETHING_IMPORTANT": OK, "TEST_SOMETHING_ELSE": CRIT},
    ... )
    False
    >>> block_conf(
    ...     name_filter=re.compile(r'TEST_[A-Z_]+_IMPORTANT'), accept_result_threshold=OK
    ... ).should_block(
    ...     {"TEST_SOMETHING_IMPORTANT": OK, "TEST_SOMETHING_ELSE_IMPORTANT": CRIT},
    ... )
    True
    >>> block_conf(
    ...     name_filter=re.compile(r'TEST_[A-Z_]+_IMPORTANT'), accept_result_threshold=OK
    ... ).should_block(
    ...     {"TEST_SOMETHING_IMPORTANT": OK, "TEST_SOMETHING_ELSE_IMPORTANT": UB},
    ... )
    True
    >>> block_conf(
    ...     name_filter=re.compile(r'TEST_[A-Z_]+_IMPORTANT'), accept_result_threshold=OK
    ... ).should_block(
    ...     {"TEST_SOMETHING_IMPORTANT": OK, "TEST_SOMETHING_ELSE_IMPORTANT": None},
    ... )
    False
    >>> block_conf(
    ...     name_filter=re.compile(r'TEST_[A-Z_]+_IMPORTANT'), accept_result_threshold=OK, ignore_empty=False
    ... ).should_block(
    ...     {"TEST_SOMETHING_IMPORTANT": OK, "TEST_SOMETHING_ELSE_IMPORTANT": None},
    ... )
    True
    """

    def __init__(self, name_filter, accept_result_threshold, ignore_empty):
        self._name_filter = name_filter
        self._accept_result_threshold = accept_result_threshold
        self._ignore_empty = ignore_empty
        self._last_block_culprit = None

    def name_matches_filter(self, name):
        raise NotImplementedError

    def to_protobuf(self):
        raise NotImplementedError

    def should_block(self, test_result_dict):
        """
        Checks if test results statuses in `test_result_dict` meet the requirements
        """
        self._last_block_culprit = None
        for test_name in test_result_dict:
            test_result = test_result_dict[test_name]
            if not self.name_matches_filter(test_name):
                continue
            if test_result is None:
                if self._ignore_empty:
                    continue
                self._last_block_culprit = {test_name: test_result}
                return True
            if test_result >= self._accept_result_threshold:
                continue
            self._last_block_culprit = {test_name: test_result}
            return True
        return False

    @property
    def last_block_culprit(self):
        return self._last_block_culprit

    @property
    def name_filter_str(self):
        return self._name_filter

    @property
    def ignore_empty(self):
        return self._ignore_empty

    @property
    def accept_result_threshold(self):
        return self._accept_result_threshold

    def __str__(self):
        return u"{cls_name}: {name_filter} - {threshold}{ignore_empty}".format(
            cls_name=self.__class__.__name__,
            name_filter=self.name_filter_str,
            threshold=self._accept_result_threshold,
            ignore_empty=(" (ignores empty)" if self._ignore_empty else ""),
        )

    def __repr__(self):
        return str(self)


class BlockConfMatchStr(BlockConfBase):

    def name_matches_filter(self, name):
        return fnmatch.fnmatch(name, self._name_filter)

    def to_protobuf(self):
        from release_machine.common_proto import block_release_pb2, test_results_pb2

        return block_release_pb2.BlockRelease(
            accept_result_threshold=getattr(test_results_pb2.TestResult.TestStatus, self._accept_result_threshold.name),
            ignore_empty=self._ignore_empty,
            name_filter_str=self.name_filter_str,
        )


class BlockConfMatchRegex(BlockConfBase):

    def name_matches_filter(self, name):
        return bool(self._name_filter.match(name))

    @property
    def name_filter_str(self):
        return self._name_filter.pattern

    def to_protobuf(self):
        from release_machine.common_proto import block_release_pb2, test_results_pb2

        return block_release_pb2.BlockRelease(
            accept_result_threshold=getattr(test_results_pb2.TestResult.TestStatus, self._accept_result_threshold.name),
            ignore_empty=self._ignore_empty,
            name_filter_regex=self.name_filter_str,
        )


class BlockConfNever(BlockConfBase):

    def name_matches_filter(self, name):
        return False

    def should_block(self, test_result_dict):
        return False

    def to_protobuf(self):
        from release_machine.common_proto import block_release_pb2
        return block_release_pb2.BlockRelease()

    def __str__(self):
        return "{} - never blocks a thing".format(self.__class__.__name__)


def block_conf(name_filter, accept_result_threshold, ignore_empty=True):
    # type: (Union[string_types, REGEX_TYPE, None], Optional[TestStatus], bool) -> BlockConfBase

    if not name_filter or accept_result_threshold is None:
        return BlockConfNever(name_filter, accept_result_threshold, ignore_empty)
    if isinstance(name_filter, string_types):
        return BlockConfMatchStr(name_filter, accept_result_threshold, ignore_empty)
    if isinstance(name_filter, REGEX_TYPE):
        return BlockConfMatchRegex(name_filter, accept_result_threshold, ignore_empty)


def parse_block_conf_proto(block_conf_proto):
    # type: (List[release_machine.common_proto.block_release_pb2.BlockRelease]) -> List[BlockConfBase]

    from release_machine.common_proto import test_results_pb2

    block_on_test_results = []

    for block_option_proto in block_conf_proto:

        status_name = test_results_pb2.TestResult.TestStatus.Name(block_option_proto.accept_result_threshold)

        block_on_test_results.append(block_conf(
            name_filter=(
                re.compile(block_option_proto.name_filter_regex) if block_option_proto.name_filter_regex
                else block_option_proto.name_filter_str
            ),
            accept_result_threshold=KNOWN_STATUSES[status_name],
            ignore_empty=block_option_proto.ignore_empty,
        ))

    return block_on_test_results


REQUIRE_ALL_OK = block_conf(name_filter='*', accept_result_threshold=OK)
REQUIRE_NO_CRITS = block_conf(name_filter='*', accept_result_threshold=WARN)
REQUIRE_ALL_FINISHED = block_conf(name_filter='*', accept_result_threshold=CRIT)
NEVER_BLOCK = block_conf(name_filter=None, accept_result_threshold=None)
