import six

from collections import namedtuple


if six.PY34:
    from collections.abc import Iterable
else:
    from collections import Iterable


U = namedtuple("U", ["name"])


class Abc(namedtuple("Abc", ["component_id", "role_id", "schedule_slug", "service_name"])):

    def __new__(cls, component_id=None, role_id=None, schedule_slug=None, service_name=None):
        return super(Abc, cls).__new__(
            cls,
            component_id=component_id,
            role_id=role_id,
            schedule_slug=schedule_slug,
            service_name=service_name,
        )


class Responsible(namedtuple("Responsible", ["login", "abc", "u"])):

    def __new__(cls, login, abc=None, u=None):
        return super(Responsible, cls).__new__(cls, login, abc, u)


class ABCSelection(object):

    def __init__(self, service, scope=None, role=None, duty=None):
        """
        :type service: six.string_types
        :param service:
            Abc service name

        :type scope: typing.Union[six.string_types, typing.List[six.string_types]]
        :param scope:
            Abc service scope or list of them

        :type role: typing.Union[six.string_types, typing.List[six.string_types]]
        :param role:
            Abc service role or list of them

        :type duty: typing.Union[six.string_types, typing.List[six.string_types]]
        :param duty:
            Abc service duty or list of them
        """

        self._service = service
        self._scope = scope
        self._role = role
        self._duty = duty

    @property
    def service(self):
        return self._service

    @property
    def scope(self):
        return self._scope

    @property
    def role(self):
        return self._role

    @property
    def duty(self):
        return self._duty

    @staticmethod
    def convert_container_or_str(value):

        if isinstance(value, six.string_types):
            return value

        if not isinstance(value, Iterable):
            return str(value)

        return [str(item) for item in value]

    def to_dict(self):
        result = {
            "service": self._service,
        }

        if self.scope is not None:
            result["scope"] = self.convert_container_or_str(self.scope)

        if self.role is not None:
            result["role"] = self.convert_container_or_str(self.role)

        if self.duty is not None:
            result["duty"] = self.convert_container_or_str(self.duty)

        return result
