import logging


LOGGER = logging.getLogger(__name__)


class YappyBaseCfg(object):
    """ Config to generate yappy betas """
    working_betas_limit = 3
    wait_for_deploy_time_sec = 4 * 60 * 60

    @property
    def betas(self):
        # Dict with betas configuration: "beta_configuration name (for ex. web)": YappyBetaCfg or YappyTemplateCfg
        return {}

    def get_beta_name(self, beta_conf_type, patch_name=None):
        # patch name is required parameter for templated configs, for pure yappy betas it is ignored
        beta_conf = self.betas.get(beta_conf_type)
        if not beta_conf:
            raise KeyError(
                "Component does not have beta configuration with name '{}'. "
                "Specify correct beta_conf_type via task parameters".format(beta_conf_type)
            )
        # LOGGER.debug("Get beta name with parameters: beta_conf_type=%s, patch_name=%s", beta_conf_type, patch_name)
        beta_name = beta_conf.get_beta_name(patch_name=patch_name)
        # LOGGER.debug("Construct beta_name=%s", beta_name)
        return beta_name

    def get_working_betas_limit(self, beta_conf_type):
        return self.betas.get(beta_conf_type).get_working_betas_limit()

    def get_stop_betas_gap(self, beta_conf_type):
        """
        Stores number of layer from which all betas will be stopped
        Assume current branch number is N. Then `stop_betas_gap == 2` all betas in branches N-2, N-3 will be stopped.
        """
        return self.betas.get(beta_conf_type).get_stop_betas_gap()


class YappyBetaCfg(object):

    def __init__(self, beta_name, patches, new_yappy=False):
        self.beta_name = beta_name
        self.patches = patches
        self.new_yappy = new_yappy

    def get_beta_name(self, **kwargs):
        return self.beta_name

    def get_working_betas_limit(self):
        return 1

    def get_stop_betas_gap(self):
        return 2


class YappyTemplateCfg(object):

    def __init__(self, template_name, patches, working_betas_limit=3, new_yappy=False, stop_betas_gap=2):
        self.template_name = template_name
        self.patches = patches
        self.working_betas_limit = working_betas_limit
        self.new_yappy = new_yappy
        self.stop_betas_gap = stop_betas_gap

    def get_beta_name(self, **kwargs):
        patch_name = kwargs["patch_name"]
        return "{}-{}".format(self.template_name, patch_name)

    def get_working_betas_limit(self):
        return self.working_betas_limit

    def get_stop_betas_gap(self):
        return self.stop_betas_gap


class YappyBetaPatch(object):

    def __init__(
        self,
        patch_file, resources,
        parent_service=None, ignore_instance_spec=None, callisto_options=None, instance_spec=None,
        copy_coredump_policy=False,
    ):
        self.patch_file = patch_file
        self.resources = resources
        self.parent_service = parent_service
        self.ignore_instance_spec = ignore_instance_spec
        self.callisto_options = callisto_options
        self.instance_spec = instance_spec
        self.copy_coredump_policy = copy_coredump_policy

    def get_patch_name(self, **kwargs):
        return "{}.yaml".format(self.patch_file)


class YappyTemplatePatch(object):

    def __init__(
        self,
        patch_dir, resources,
        parent_service=None, ignore_instance_spec=None, callisto_options=None, instance_spec=None,
        copy_coredump_policy=False,
    ):
        self.patch_dir = patch_dir
        self.resources = resources
        self.parent_service = parent_service
        self.ignore_instance_spec = ignore_instance_spec
        self.callisto_options = callisto_options
        self.instance_spec = instance_spec
        self.copy_coredump_policy = copy_coredump_policy

    def get_patch_name(self, **kwargs):
        patch_name = kwargs.get("patch_name")
        return "{}/{}.yaml".format(self.patch_dir, patch_name)


class YappyParametrizedResource(object):

    manage_type = "SANDBOX_RESOURCE"

    def __init__(self, param_name, local_path, checkconfig_name=None, storage=None):
        self.local_path = local_path
        # Pop this element to use, it could not be in yappy patch
        self.param_name = param_name
        self.checkconfig_name = checkconfig_name
        self.storage = storage


class YappyStaticResource(object):
    def __init__(self, local_path, manage_type, resource_id=None, content=None):
        self.local_path = local_path
        self.manage_type = manage_type
        self.resource_id = resource_id
        self.content = content


class YappyLastReleasedResource(object):

    manage_type = "SANDBOX_RESOURCE"

    def __init__(self, local_path, res_type):
        self.local_path = local_path
        self.res_type = res_type


class YappyDeleteResource(YappyStaticResource):

    MANAGE_TYPE = "DELETE"

    def __init__(self, local_path, resource_id=None, content=None):

        super(YappyDeleteResource, self).__init__(
            local_path=local_path,
            manage_type=self.MANAGE_TYPE,
            resource_id=resource_id,
            content=content,
        )


class YappyEnvVar(object):

    def __init__(self, container, name, value):
        self.container = container
        self.name = name
        self.value = value
