# -*- coding: utf-8 -*-

import typing  # noqa

from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import ab_tests as ab_tests_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import run_command as run_command_cubes
# from sandbox.projects.release_machine.components.config_core.jg.cube.lib import dummy as dummy_cubes


KPI = (
    "https://nda.ya.ru/t/WD7eUt8H4w9qNq",
    "KPI",
)

LONGEST_RUNNING_EXPID = (
    "https://nda.ya.ru/t/sS6Zp6CB4w9owC",
    "Longest Running ExpID",
)

SOY_QUEUE = (
    "https://nda.ya.ru/t/oX-7nkha4w9mau",
    "SoY Queue - experiments_rm (number of running and enqueued operations)",
)

CI_ACTION_STATUSES = (
    "https://nda.ya.ru/t/Qmw-iioi4wBYFK",
    "CI Action Statuses",
)

APPROVEMENT_TEXT = (
    "Некоторые тесты упали. Требуется ручное подтверждение ответственного<br>"
    "<strong>Не пытайтесь согласовывать эксперимент, если не все тесты добежали</strong>"
)

EXPID_EXTRACT_FAILURE_NOTIFICATION_COMMAND = """curl -XPOST \
-H"Authorization: OAuth $JNS_TOKEN" \
-H"Content-type: application/json"\
 https://jns.yandex-team.ru/api/messages/send_to_channel \
--data-binary @- <<EOF
{
  "project": "ups",
  "template": "approvement_alert",
  "params": {
    "message": {"string_value": "FAILED to parse experiment ID"},
    "flow_url": {"string_value": "${context.ci_url}"}
  },
  "target_project": "ups",
  "channel": "ab_exp__approvement_alert"
}
EOF
"""


def get_statistics_page_charts():
    spc = []

    for link, description in [
        KPI,
        LONGEST_RUNNING_EXPID,
        SOY_QUEUE,
        CI_ACTION_STATUSES,
    ]:
        spc.append(
            "{}?description={}".format(
                link,
                description.replace(" ", "%20"),
            ),
        )

    spc.append(
        configs.statistics_page.PredefinedCharts.TIMESPECTRE_METRICS,
    )

    return spc


class AbExpCfg(configs.ReferenceCIConfig):

    name = "ab_exp"
    display_name = "AB Experiments Tests"
    responsible = configs.Responsible(
        abc=configs.Abc(service_name="ups", component_id=33085),
        login="ilyaturuntaev",
    )

    class CI(configs.ReferenceCIConfig.CI):
        a_yaml_dir = "quality/robots/ab_testing/experiments"
        secret = "sec-01desry8fbgvnkbeybem81ferv"
        sb_owner_group = "EXPERIMENTS"

        observed_sub_paths = [
            "*/**.yaml",
        ]

    class Releases(configs.ReferenceCIConfig.Releases):
        resources_info = []

        @property
        def block_on_test_results(self):
            try:
                from release_machine.tasklets.ab_tests.configure_ab_experiments_tests import utils as ab_conf_utils
            except ImportError:
                # A pure Sandbox build is not able to handle this import so we're just skipping it.
                # Additional logging here is omitted on purpose: we do not want to fill Sandbox logs
                # with useless reports
                return [configs.release_block.NEVER_BLOCK]

            block_settings = []

            for test_info in ab_conf_utils.test_settings_iter():

                block_settings.append(
                    configs.release_block.block_conf(
                        name_filter=test_info.test_name,
                        accept_result_threshold=(
                            configs.release_block.TestStatus(name=test_info.test_accept_result_threshold)
                            if test_info.test_accept_result_threshold
                            else configs.release_block.WARN
                        ),
                        ignore_empty=False,
                    ),
                )

            return block_settings

    class JG(configs.jg_base.JGCfg):

        def iter_tests(self, configuration_cube):

            try:
                from release_machine.tasklets.ab_tests.configure_ab_experiments_tests import utils as ab_conf_utils
            except ImportError:
                # A pure Sandbox build is not able to handle this import so we're just skipping it.
                # Additional logging here is omitted on purpose: we do not want to fill Sandbox logs
                # with useless reports
                return

            for test_info in ab_conf_utils.test_settings_iter():

                test_cube_input_dict = {
                    field.name: getattr(
                        getattr(
                            configuration_cube.output.tests_config,
                            test_info.test_name,
                        ),
                        field.name,
                    ) for field in test_info.test_config_message_descriptor.fields
                }

                test_cube_input = configs.jg_cube.CubeInput(**test_cube_input_dict)

                run_flag = getattr(configuration_cube.output.tests_config, test_info.test_should_run_field_name)

                yield configs.jg_cube.Cube(
                    name=test_info.test_name,
                    task=test_info.test_task,
                    input=test_cube_input,
                    condition=configs.ci_conditions.CICondition(run_flag).not_null("`false`"),
                )

        @configs.jg_flow.register_flow(
            triggers=[configs.jg_flow.ActionTrigger(
                on=configs.jg_flow.ActionTriggerOn.COMMIT,
                filters=[
                    {
                        "discovery": "dir",
                        "sub-paths": ["*/**.yaml"],
                    },
                ],
            )],
        )
        def testid_tests(self):

            configuration_cube = ab_tests_cubes.ConfigureAbExperimentsTests(
                component_name=self.component_name,
                testid_commit=configs.rm_const.CIJMESPathCommon.TARGET_COMMIT_HASH,
            )

            graph = configs.jg_graph.Graph([configuration_cube])

            for test_cube in self.iter_tests(configuration_cube):
                graph.add(test_cube)

            return graph

        @configs.jg_flow.register_flow(
            triggers=[configs.jg_flow.ActionTrigger(
                on=configs.jg_flow.ActionTriggerOn.COMMIT,
                filters=[
                    {
                        "discovery": "dir",
                        "abs-paths": ["quality/robots/ab_testing/experiments_publishing/**/**.json"],
                    },
                ],
            )],
        )
        def approvements(self):

            extract_experiment_id = run_command_cubes.RunCommand(
                name="extract_experiment_id",
                title="Extract Experiment ID",
                input=configs.jg_cube.CubeInput(
                    config={
                        "cmd_line": "cat $("
                                    "arc show r${context.target_revision.number} --name-only --json "
                                    "| ya tool jq '.[0].names[0].path' "
                                    "| sed -r 's/\"//g'"
                                    ") "
                                    "| ya tool jq '.experiment_id' "
                                    "| sed -r 's/\"//g' "
                                    "> $RESULT_RESOURCES_PATH/experiment_id",
                        "arc_mount_config": {
                            "enabled": True,
                            "arc_token": {
                                "key": "common_release_token",
                            },
                        },
                        "result_output": [
                            {
                                "path": "experiment_id",
                            }
                        ],
                    },
                ),
            )

            experiment_id_length_expression = configs.ci_conditions.CICondition(
                extract_experiment_id.output.result_output["?path == 'experiment_id'"].string[0],
            ).not_null("''").length()

            notify_extract_failed = run_command_cubes.RunCommand(
                name="notify_extract_failed",
                title="Notify experiment ID extraction failure",
                condition=experiment_id_length_expression == 0,
                input=configs.jg_cube.CubeInput(
                    config={
                        "cmd_line": EXPID_EXTRACT_FAILURE_NOTIFICATION_COMMAND,
                        "secret_environment_variables": [
                            {
                                "key": "JNS_TOKEN",
                                "secret_spec": {
                                    "key": "jns_token",
                                },
                            }
                        ],
                    },
                ),
                needs=[
                    extract_experiment_id,
                ],
            )

            approvement_creator = ab_tests_cubes.AbExperimentsApprovementCreator(
                component_name=self.root_cfg.name,
                name="approvement_creator",
                title="Approvement Creator",
                input=configs.jg_cube.CubeInput(
                    mode="experiment_approvements",
                    ticket_key=extract_experiment_id.output.result_output["?path == 'experiment_id'"].string[0][0],
                    approvement_text=APPROVEMENT_TEXT,
                ),
                condition=experiment_id_length_expression != 0,
                needs=[
                    extract_experiment_id,
                ],
                attributes={
                    "attempts": {
                        "max": 4,
                        "backoff": "exp",
                        "initial-backoff": "5m",
                    },
                },
                manual=False,  # enable (manual=False) or disable (manual=True) automatic approvement launch
            )

            return configs.jg_graph.Graph([
                extract_experiment_id,
                notify_extract_failed,
                approvement_creator,
            ])

    class Notify(configs.ReferenceCIConfig.Notify):
        use_startrek = False

    class ChangelogCfg(configs.ReferenceCIConfig.ChangelogCfg):
        wiki_page = None
        observed_paths = ["quality/robots/ab_testing/experiments"]

    class MetricsCfg(object):
        limit_s = 10800  # 3 hours

    class ReleaseViewer(configs.ReferenceCIConfig.ReleaseViewer):
        statistics_page_charts = get_statistics_page_charts()
