# -*- coding: utf-8 -*-
import os

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import releasable_items as ri
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils


class ABTCfg(configs.ReferenceBranchedConfig):
    name = "abt"
    responsible = "primorial"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_db = "abt_release_trunk"
        trunk_task_owner = "AB-TESTING"
        branch_db_template = "abt-release-stable-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            TARGET_Z2_MR_VELES = "MR_VELES"
            DEFAULT_KATI_ARCADIA_PATH = "quality/ab_testing/scripts/kati"

            TARGET_Z2_MR_VELES01 = "MR_VELES01"
            TARGET_Z2_MR_VELES02 = "MR_VELES02"
            YT_CLUSTER_ABT_METRICS_DIFF = "freud"
            YT_POOL_ABT_METRICS_DIFF = "abt"

            CALC_METRICS_DAEMON_PLATFORMS = [
                {
                    "name": "linux",
                    "hr_name": "Linux",
                },
                {
                    "name": "darwin",
                    "hr_name": "macOS",
                }
            ]

            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(branching_mode=rm_const.BranchingMode.BRANCH_TO_BRANCH),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementWiki(),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrekWiki(),
                ]

            @property
            def _branch_part(self):
                default_branch_part = super(self.__class__, self)._branch_part
                branch_part = [
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="BUILD_SESSION_MANAGER_RPC",
                        build_item="SEMA_RPC",
                        out={
                            "SESSION_MANAGER_RPC_EXECUTABLE": 200,
                        }
                    ),
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="BUILD_BLENDER_BINARIES",
                        build_item="BLENDER_BINARIES",
                        out={
                            "BLENDER_RANDOM_POOL_EXECUTABLE": 120,
                            "BLENDER_SURPLUS_SQUEEZE_EXECUTABLE": 120,
                            "BLENDER_WIZARDS_CLICKS_EXECUTABLE": 120,
                        },
                        ctx={
                            "target_resources": ["BLENDER_RANDOM_POOL_EXECUTABLE", "BLENDER_SURPLUS_SQUEEZE_EXECUTABLE", "BLENDER_WIZARDS_CLICKS_EXECUTABLE"],
                            "strip_binaries": True
                        }
                    )
                ]

                def package_name(name, prefix="packages/debian/mapreduce"):
                    return "{}/{}/{}.json".format(prefix, name, name)

                ABT_BUILD_PACKAGES = [
                    # Main yandex-search-ab-testing package must be the first
                    (package_name("yandex-search-ab-testing"), "AB_TESTING_YA_PACKAGE"),
                    (package_name("yandex-search-ab-testing-lsd"), "AB_TESTING_LSD_YA_PACKAGE"),
                    (package_name("yandex-search-ab-testing-prepare"), "AB_TESTING_PREPARE_YA_PACKAGE"),
                    (package_name("yandex-search-sessions-viewer-sessions-analysis"), "AB_TESTING_SESSION_ANALYSIS_YA_PACKAGE"),
                ]
                MR_VELES01_BUILD_PACKAGES = [
                    (package_name("yandex-search-blender-wizards-clicks"), "YA_PACKAGE"),
                ]
                MR_VELES02_BUILD_PACKAGES = [
                    (package_name("yandex-search-blender-metrics-calculator"), "YA_PACKAGE"),
                    (package_name("yandex-search-blender-surplus-squeeze"), "YA_PACKAGE"),
                    (package_name("yandex-search-blender-random-pool"), "YA_PACKAGE"),
                    (package_name("yandex-search-blender-misspell-pool"), "YA_PACKAGE"),
                ]
                for subname, target, release_to_nanny, (packages, resource_types) in [
                    ("ABT", self.TARGET_Z2_MR_VELES, True, zip(*ABT_BUILD_PACKAGES)),
                    ("BLENDER", self.TARGET_Z2_MR_VELES01, False, zip(*MR_VELES01_BUILD_PACKAGES)),
                    ("BLENDER", self.TARGET_Z2_MR_VELES02, False, zip(*MR_VELES02_BUILD_PACKAGES)),
                ]:
                    release_text = "{} component {} packages build using release machine".format(
                        self.name,
                        subname,
                    )
                    packages_text = ",".join(os.path.splitext(os.path.basename(package))[0] for package in packages)
                    description = "[KATI] {}\nPackages: {} release_machine\nRun by {}".format(
                        release_text,
                        packages_text,
                        self.DEFAULT_KATI_ARCADIA_PATH,
                    )

                    branch_part.append(
                        jg_build.JobGraphElementYaMakeBuildBranched(
                            task_name="KOSHER_YA_PACKAGE",
                            build_item="{}_PACKAGES_{}".format(subname, target),
                            job_params={
                                "apiargs": {
                                    "requirements": {
                                        "disk_space": 300 * (2 ** 30),  # 300 GB
                                    }
                                },
                            },
                            ctx={
                                "description": description,
                                "packages": ";".join(packages),
                                "package_type": "debian",
                                "build_type": "release",
                                "strip_binaries": True,
                                "compress_package_archive": True,
                                "key_user": "robot-eksperimentus",
                                "publish_package": True,
                                "publish_to": "search-precise",
                                "resource_type": ";".join(resource_types),
                                "notify_via": "email",
                                "notify_if_failed": ["primorial", "experiments-team-dev"],
                                "release_to_nanny": release_to_nanny,
                            }
                        )
                    )

                branch_part.append(
                    jg_build.JobGraphElementYaMakeBuildBranched(
                        task_name="KOSHER_YA_PACKAGE",
                        build_item="ABT_TESTS",
                        job_params={
                            "apiargs": {
                                "requirements": {
                                    "disk_space": 300 * (2 ** 30),  # 300 GB
                                }
                            },
                        },
                        ctx={
                            "description": "[TESTS] Run ABT tests",
                            "packages": package_name("yandex-search-ab-testing-abt-test"),
                            "package_type": "debian",
                            "build_type": "release",
                            "strip_binaries": True,
                            "compress_package_archive": True,
                            "key_user": "robot-eksperimentus",
                            "publish_package": False,
                            "debian_dont_store_package": True,
                            "notify_via": "email",
                            "notify_if_failed": ["sharamed", "primorial", "experiments-team-dev"],
                            "run_tests": True,
                            "run_medium_tests": True,
                            "run_long_tests": True,
                            "ignore_fail_tests": False,
                            "release_to_nanny": False,
                        }
                    )
                )

                for platform in self.CALC_METRICS_DAEMON_PLATFORMS:
                    description = "calc_metrics_daemon ABTREL ({})".format(
                        platform["hr_name"]
                    )

                    branch_part.append(
                        jg_build.JobGraphElementYaMakeBuildBranched(
                            task_name="KOSHER_YA_PACKAGE",
                            build_item="CALC_METRICS_DAEMON_{}".format(
                                platform["name"].upper()
                            ),
                            job_params={
                                "apiargs": {
                                    "requirements": {
                                        "disk_space": 300 * (2 ** 30),  # 300 GB
                                    }
                                },
                            },
                            ctx={
                                "description": description,
                                "target_platform": platform["name"],
                                "build_system": "ya",  # need this because reasons https://st.yandex-team.ru/RMDEV-1814#5eff547b8e522e4d6708320b
                                "packages": "quality/logs/calc_metrics/calc_metrics_daemon/pkg.json",
                                "package_type": "tarball",
                                "compress_package_archive": True,
                                "resource_type": "CALC_METRICS_DAEMON_EXECUTABLE",
                                "package_resource_description": {
                                    "quality/logs/calc_metrics/calc_metrics_daemon/pkg.json": description,
                                },
                                "do_not_remove_resources": True,
                            },
                            out={
                                "CALC_METRICS_DAEMON_EXECUTABLE": 200,
                            }
                        )
                    )

                branch_part.append(
                    jg_test.JobGraphElementActionRunAcceptanceBranched(
                        job_params={
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="SEMA_RPC",
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="BLENDER_BINARIES",
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="ABT_PACKAGES_{}".format(self.TARGET_Z2_MR_VELES),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="BLENDER_PACKAGES_{}".format(self.TARGET_Z2_MR_VELES01),
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="BLENDER_PACKAGES_{}".format(self.TARGET_Z2_MR_VELES02),
                            ),
                        ) +
                        tuple(
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="CALC_METRICS_DAEMON_{}".format(
                                    platform["name"].upper()
                                ),
                            )
                            for platform in self.CALC_METRICS_DAEMON_PLATFORMS
                        )
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="RUN_ABT_METRICS",
                        job_params={
                            "job_name_parameter": "RUN_CANDIDATE_ABT_METRICS",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="ABT_PACKAGES_{}".format(self.TARGET_Z2_MR_VELES),
                                parent_job_data=(
                                    jg_job_data.ParentDataId(
                                        input_key="package_task_id",
                                    ),
                                ),
                            ),
                            jg_arrows.GlobalResourceData(
                                input_key="sample_report",
                                resource_name="ABT_REGRESSION_SAMPLE_REPORT",
                            ),
                        ),
                        ctx={
                            "cluster": self.YT_CLUSTER_ABT_METRICS_DIFF,
                            "yt_pool": self.YT_POOL_ABT_METRICS_DIFF,
                        },
                        out={
                            "RUN_ABT_METRICS_REPORT": 200,
                        }
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="RUN_ABT_METRICS",
                        job_params={
                            "job_name_parameter": "RUN_RELEASED_ABT_METRICS",
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerNewTag(),
                            jg_arrows.GlobalResourceData(
                                input_key="sample_report",
                                resource_name="ABT_REGRESSION_SAMPLE_REPORT",
                            ),
                        ),
                        ctx={
                            "package_id": "stable",  # our task will find and use prod package_id
                            "cluster": self.YT_CLUSTER_ABT_METRICS_DIFF,
                            "yt_pool": self.YT_POOL_ABT_METRICS_DIFF,
                        },
                        out={
                            "RUN_ABT_METRICS_REPORT": 200,
                        }
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="REPORT_DIFF_METRICS_ABT",
                        job_params={
                            "job_name_parameter": "REPORT_DIFF_METRICS_ABT",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="RUN_CANDIDATE_ABT_METRICS",
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="metrics_next",
                                        resource_name="RUN_ABT_METRICS_REPORT",
                                    ),
                                ),
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="RUN_RELEASED_ABT_METRICS",
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key="metrics_prev",
                                        resource_name="RUN_ABT_METRICS_REPORT",
                                    ),
                                ),
                            ),
                            jg_arrows.GlobalResourceData(
                                input_key="sample_report",
                                resource_name="ABT_REGRESSION_SAMPLE_REPORT",
                            ),
                            jg_arrows.ParamsData(
                                input_key="release_number",
                                transform=jg_utils.get_major_release_number,
                            )
                        ),
                    )
                )
                for subname, target_server in [
                    ("ABT", self.TARGET_Z2_MR_VELES),
                    ("BLENDER", self.TARGET_Z2_MR_VELES01),
                    ("BLENDER", self.TARGET_Z2_MR_VELES02),
                ]:
                    branch_part.append(
                        jg_test.JobGraphElementTestBranchCommon(
                            task_name="RELEASE_SEARCH_COMPONENT_Z_2",
                            job_params={
                                "job_type": rm_const.JobTypes.RELEASE,
                                "job_name_parameter": "{}_PACKAGES_Z2_{}".format(subname, target_server),
                                "apiargs": {
                                    "kill_timeout": 600,
                                }
                            },
                            job_arrows=(
                                jg_arrows.ParentsData(
                                    input_key="package_tasks",
                                    triggers=(
                                        jg_job_triggers.JobTriggerBuild(
                                            job_name_parameter="{}_PACKAGES_{}".format(subname, target_server),
                                            parent_job_data=(
                                                jg_job_data.ParentDataId(
                                                    input_key="build_id",
                                                )
                                            )
                                        ),
                                    ),
                                    transform=lambda build_id: [build_id],
                                ),
                                jg_arrows.ParamsData(
                                    input_key="release_number",
                                    transform=jg_utils.get_major_release_number,
                                )
                            ),
                            ctx={
                                "roll_to": "stable",
                                "z2_config_id": target_server,
                            },
                        )
                    )
                return default_branch_part + branch_part

            @property
            def _release(self):
                default_release_part = super(self.__class__, self)._release
                release_part = [
                    jg_release.JobGraphElementWikiFinal(),
                    jg_release.JobGraphElementReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        release_item="SEMA_RPC",
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="SEMA_RPC",
                                parent_job_data=(
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "sema_rpc",
                                        "SESSION_MANAGER_RPC_EXECUTABLE",
                                    )
                                )
                            )
                        )
                    ),
                    jg_release.JobGraphElementReleaseBranched(
                        release_to=rm_const.ReleaseStatus.stable,
                        release_item="BLENDER_BINARIES",
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="BLENDER_BINARIES",
                                parent_job_data=[
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "blender_random_pool",
                                        "BLENDER_RANDOM_POOL_EXECUTABLE",
                                    ),
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "blender_surplus_squeeze",
                                        "BLENDER_SURPLUS_SQUEEZE_EXECUTABLE",
                                    ),
                                    jg_job_data.ParentDataDict(
                                        "component_resources",
                                        "blender_wizards_clicks",
                                        "BLENDER_WIZARDS_CLICKS_EXECUTABLE",
                                    )
                                ]
                            )
                        )
                    ),
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="UPDATE_METRICS_DAEMON_IN_PROJECTS",
                        job_params={
                            "job_name_parameter": "ABT_RELEASE_UPDATE_METRICS_DAEMON_IN_PROJECTS",
                        },
                        job_arrows=(
                            tuple(
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="CALC_METRICS_DAEMON_{}".format(
                                        platform["name"].upper()
                                    ),
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="{}_resource".format(
                                                platform["name"]
                                            ),
                                            resource_name="CALC_METRICS_DAEMON_EXECUTABLE"
                                        ),
                                    ),
                                )
                                for platform in self.CALC_METRICS_DAEMON_PLATFORMS
                            )
                        ),
                    ),
                    jg_test.JobGraphElementActionRunAcceptanceBranched(
                        job_params={
                            "job_type": rm_const.JobTypes.ACTION_RELEASE,
                            "job_name_parameter": rm_const.ReleaseStatus.stable,
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerRelease(
                                job_name_parameter="SEMA_RPC__{}".format(rm_const.ReleaseStatus.stable),
                            ),
                            jg_job_triggers.JobTriggerRelease(
                                job_name_parameter="BLENDER_BINARIES__{}".format(rm_const.ReleaseStatus.stable),
                            ),
                            jg_job_triggers.JobTriggerRelease(
                                job_name_parameter="ABT_PACKAGES_Z2_{}".format(self.TARGET_Z2_MR_VELES),
                            ),
                            jg_job_triggers.JobTriggerRelease(
                                job_name_parameter="BLENDER_PACKAGES_Z2_{}".format(self.TARGET_Z2_MR_VELES01),
                            ),
                            jg_job_triggers.JobTriggerRelease(
                                job_name_parameter="BLENDER_PACKAGES_Z2_{}".format(self.TARGET_Z2_MR_VELES02),
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="ABT_RELEASE_UPDATE_METRICS_DAEMON_IN_PROJECTS",
                            ),
                            jg_job_triggers.JobTriggerWikiFinal(),
                        )
                    )
                ]
                return default_release_part + release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            @property
            def change_frequency(self):
                uppercase_name = self.name.upper()
                tests = super(self.__class__, self).change_frequency

                tests.update({
                    "_BUILD_RELEASE_{}".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                    "_{}_REPORT_DIFF_METRICS_ABT".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                })
                return tests

    class ReleaseViewer(configs.ReferenceConfig.ReleaseViewer):
        kpi_alert = 7

    class Releases(configs.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name="sema_rpc",
                    data=ri.SandboxResourceData("SESSION_MANAGER_RPC_EXECUTABLE"),
                    deploy_infos=[ri.single_nanny_service("production_sema")]
                ),
                ri.ReleasableItem(
                    name="ab_testing_ya_package",
                    data=ri.SandboxResourceData("AB_TESTING_YA_PACKAGE"),
                    deploy_infos=[
                        ri.single_nanny_service("testing_ems", stage=rm_const.ReleaseStatus.stable),
                        ri.NannyDeployInfo(
                            [
                                ri.DeployService("production_ems{suffix}_{dc}".format(suffix=suffix, dc=datacenter))
                                for suffix in ("", "_dataset")
                                for datacenter in ("vla", "man", "sas")
                            ],
                            stage=rm_const.ReleaseStatus.stable
                        )
                    ]
                ),
                ri.ReleasableItem(
                    name="blender_random_pool",
                    data=ri.SandboxResourceData("BLENDER_RANDOM_POOL_EXECUTABLE"),
                    deploy_infos=[ri.SandboxInfo(stage=rm_const.ReleaseStatus.stable)]
                ),
                ri.ReleasableItem(
                    name="blender_surplus_squeeze",
                    data=ri.SandboxResourceData("BLENDER_SURPLUS_SQUEEZE_EXECUTABLE"),
                    deploy_infos=[ri.SandboxInfo(stage=rm_const.ReleaseStatus.stable)]
                ),
                ri.ReleasableItem(
                    name="blender_wizards_clicks",
                    data=ri.SandboxResourceData("BLENDER_WIZARDS_CLICKS_EXECUTABLE"),
                    deploy_infos=[ri.SandboxInfo(stage=rm_const.ReleaseStatus.stable)]
                ),
            ]
        release_followers_permanent = [
            "primorial",
        ]
        wait_for_deploy_time_sec = 900

    class Notify(configs.ReferenceBranchedConfig.Notify):
        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            chats = ["primorial"]
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            assignee = "primorial"
            queue = "ABTREL"
            dev_queue = "EXPERIMENTS"
            summary_template = u"Релиз АБТ из abt/stable-{}"
            components = "ABT"
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = [
                "ajkon",
                "alexfilatov",
                "buryakov",
                "ilyusha",
                "primorial",
                "rkam",
                "samhan",
                "timurguryev",
                "valgushev",
                "leftmain",
                "sharamed",
                "conterouz",
            ]
            add_commiters_as_followers = True

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = "jandekspoisk/kachestvopoiska/abt/releases/history/"
        dirs = [
            "arcadia/quality/ab_testing/abt_lib",
            "arcadia/quality/logs/baobab/tamus/rules",
            "arcadia/quality/ab_testing/calc_activity",
            "arcadia/quality/ab_testing/calc_app_activation",
            "arcadia/quality/ab_testing/calc_app_activity",
            "arcadia/quality/ab_testing/calc_app_push",
            "arcadia/quality/ab_testing/calc_app_push_for_metrika",
            "arcadia/quality/ab_testing/calc_app_uuid_to_testids",
            "arcadia/quality/ab_testing/calc_long_tail",
            "arcadia/quality/ab_testing/calc_news_metrics",
            "arcadia/quality/ab_testing/calc_spy_log_stat",
            "arcadia/quality/ab_testing/lib_calc_session_metrics",
            "arcadia/quality/ab_testing/lsd",
            "arcadia/quality/ab_testing/batch_fetcher",
            "arcadia/quality/ab_testing/bs_collector",
            "arcadia/quality/ab_testing/prepare_fast_video",
            "arcadia/quality/ab_testing/stat_cofe_lib",
            "arcadia/quality/ab_testing/stat_collector_util",
            "arcadia/quality/ab_testing/watch_collector",
            "arcadia/quality/ab_testing/yuid_testids",
            "arcadia/quality/ab_testing/abt_resources_lib",
            "arcadia/quality/ab_testing/paths_to_tables_lib",
            "arcadia/quality/ab_testing/sessions_analysis/sessions_viewer",
            "arcadia/quality/ab_testing/stat_collector_prepare_lib",
            "arcadia/quality/ab_testing/stat_collector_lib",
            "arcadia/quality/ab_testing/stat_collector",
            "arcadia/quality/ab_testing/stat_fetcher",
            "arcadia/quality/ab_testing/stat_fetcher_lib",
            "arcadia/quality/ab_testing/stat_long_fetcher",
            "arcadia/quality/ab_testing/stability/lib",
            "arcadia/quality/user_metrics/suggest_metrics",
            "arcadia/quality/user_metrics/surplus_metrics/lib",
            "arcadia/quality/user_metrics/vertical_metrics",
            "arcadia/quality/functionality/turbo/user_metrics_lib",
            "arcadia/extsearch/geo/kernel/similar_orgs/log_parsing",
            "arcadia/extsearch/geo/kernel/similar_orgs/preparation",
            "arcadia/velocity/rum/calculator",
            "arcadia/velocity/rum/parser_cpp",
            "arcadia/velocity/rum/velocity_index",
        ]
        svn_paths_filter = configs.ChangelogPathsFilter(
            rm_const.PermissionType.BANNED,
            [
                "arcadia/quality/ab_testing/abt_resources_lib/loaders/feature_custom/ya.make",
                "arcadia/quality/ab_testing/abt_resources_lib/loaders/metrics/ya.make",
                "arcadia/quality/ab_testing/abt_resources_lib/loaders/blockstat/ya.make",
            ],
        )
