import os
import sys
import logging
import inspect
from six import string_types, iteritems

from sandbox.projects.common.decorators import memoize
from sandbox.projects.release_machine.components.configs import ReferenceConfig
from sandbox.projects.release_machine.core.const import RMNames
from sandbox.projects.release_machine.core.const import ReleaseCycleType

logger = logging.getLogger(__name__)


class ReleaseMachineConfigModuleLoaderError(Exception):
    pass


class LazyNewConfigs:
    """Class performs self initialization (by configs scanning) on first access, after that it pretend to be a dict"""
    def __init__(self):
        self._is_loaded = False
        self._all_configs = {}

    def _load(self):
        if not self._is_loaded:
            _load_all_new_configs(self._all_configs)
            self._is_loaded = True
        return self._all_configs

    # Now duplicate all 'dict' interface
    def __len__(self):
        return self._load().__len__()

    def __iter__(self):
        return self._load().__iter__()

    def __contains__(self, key):
        return self._load().__contains__(key)

    def __getitem__(self, key):
        return self._load().__getitem__(key)

    def __getattr__(self, name):
        """Proxy for all ordinary members of dict"""
        def wrap(*args, **kwargs):
            return getattr(self._load(), name)(*args, **kwargs)
        return wrap


ALL_CONFIGS = LazyNewConfigs()


def get_all_names():
    return ALL_CONFIGS.keys()


def get_all_branched_names():
    return [
        conf_name
        for conf_name, config in iteritems(ALL_CONFIGS)
        if config.release_cycle_type is ReleaseCycleType.BRANCH
    ]


def get_all_tagged_names():
    return [
        conf_name
        for conf_name, config in iteritems(ALL_CONFIGS)
        if config.release_cycle_type is ReleaseCycleType.TAG
    ]


def get_all_trunk_names():
    return [
        conf_name
        for conf_name, config in iteritems(ALL_CONFIGS)
        if config.release_cycle_type is ReleaseCycleType.TRUNK
    ]


def get_all_ci_names():
    return [
        conf_name
        for conf_name, config in iteritems(ALL_CONFIGS)
        if config.release_cycle_type is ReleaseCycleType.CI
    ]


def _scan_files_for_modules():
    """
        Scan directory (hierarchically) from this module.
        All *.py files treated as loadable modules (except all started with '_')
        Yield modules found (as python module name)
    """
    root_dir = os.path.dirname(os.path.abspath(__file__))
    root_dir_len = len(root_dir) + 1
    base_module_name = os.path.splitext(__name__)[0] + "."
    for dirpath, _, filenames in os.walk(root_dir):
        if "__init__.py" not in filenames:
            logger.info(
                "Configs Loader: Skipping directory %s - no __init__.py file found", dirpath[root_dir_len:]
            )
            continue
        for file_name in filenames:
            if not file_name.startswith("_"):
                # FS path to file found (from this directory)
                module_file_path = os.path.join(dirpath, file_name)[root_dir_len:]
                # Cut file extension
                module_file_path, ext = os.path.splitext(module_file_path)
                if ext == ".py":
                    # Create Python module name (replace '/' with '.') and concatenate with this package name
                    yield base_module_name + module_file_path.replace(os.sep, ".")


def _scan_precompiled_for_modules():
    """
        Scan precompiled (in Arcadia binary Python make) modules.
        All *.py files, included via ya.make, nested in this module, will be returned
    """
    base_module_name = os.path.splitext(__name__)[0] + "."
    for module_name in sys.extra_modules:
        if module_name.startswith(base_module_name) and '._' not in module_name:
            yield module_name


def _scan_modules(raise_errors=False):
    """
        Get modules from _scan_files_for_modules() and try to load them all.
        Catch all possible exceptions - skipping module is better than whole crash :)
        Test if modules contains exactly one New Config (and optional one Old Config)
        If all ok - yield new configs and Python module names - for error reporting
    """
    for module_name in (_scan_precompiled_for_modules if hasattr(sys, 'extra_modules') else _scan_files_for_modules)():

        try:
            loaded_module = __import__(module_name, level=0, fromlist=["_"])
        except ImportError as error:
            logger.error("Configs Loader: Can't load module %s, exception caught: %s", module_name, error)
            if raise_errors:
                raise ReleaseMachineConfigModuleLoaderError("Cannot load module {}: {}".format(module_name, error))
            continue
        new_config = None

        for name in dir(loaded_module):
            cls = getattr(loaded_module, name)

            if not inspect.isclass(cls):
                continue

            if issubclass(cls, ReferenceConfig):

                if new_config:

                    if raise_errors:
                        raise ReleaseMachineConfigModuleLoaderError(
                            "More than one config class found in module {}".format(module_name),
                        )

                    logger.warning(
                        "Configs Loader: More than one config class defined in module %s (%s, %s)",
                        module_name,
                        cls.__name__,
                        new_config.__name__,
                    )
                    continue

                new_config = cls

        if not new_config:
            logger.warning("Configs Loader: config class not found in module %s", module_name)

            if raise_errors:
                raise ReleaseMachineConfigModuleLoaderError(
                    "No config class found in module {module_name}. If this module is not supposed to define any config"
                    " class (i.e. it is a supplementary module) then its name should start with an underscore: "
                    "'_{module_fixed_name}'".format(
                        module_name=module_name,
                        module_fixed_name=module_name.rsplit(".", 1)[-1],
                    ),
                )

            continue

        yield new_config, module_name


def _load_all_new_configs(all_configs):
    """
        Load and register all New Config modules
        Modules searched for in current (and nested) directory
        Can be called many times
    """
    for new_config, module_name in _scan_modules():
        name = new_config.name

        if not isinstance(name, string_types):
            continue
        if name in all_configs:
            logger.warning("Configs Loader: config %s already registered (module %s)", name, module_name)
            continue
        all_configs[name] = new_config
        if not hasattr(RMNames, name.upper()):
            setattr(RMNames, name.upper(), new_config.name)


@memoize
def get_config(component_name):
    """
    Get config instance by `component_name`

    :param component_name: component name
    :return: component config instance
    """
    return ALL_CONFIGS[component_name]()


def scan_all_modules_and_raise_on_errors():
    list(_scan_modules(raise_errors=True))
