# -*- coding: utf-8 -*-
from six import iteritems

from sandbox.projects.common import constants as sandbox_constants
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.core import const as rm_const

from sandbox.projects.release_machine.components.config_core.jg.cube.lib import build as build_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import dummy as dummy_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import release as release_cubes
from sandbox.projects.release_machine.components.config_core.jg import cube as jg_cube
from sandbox.projects.release_machine.components.config_core.jg import base as jg_base
from sandbox.projects.release_machine.components.config_core.jg import flow


YA_MAKE_RESOURCES = {
    'binaries': {
        'desc': 'Altay db cooking binaries',
        'suffix': 'binaries'
    },
    'cmpy': {
        'desc': 'Altay db cooking CM binaries and scripts',
        'suffix': 'cmpy'
    },
    'configs': {
        'desc': 'Altay db cooking configs',
        'suffix': 'configs'
    },
    'scripts': {
        'desc': 'Misc scripts for altay db cooking',
        'suffix': 'scripts'
    },
    'yql': {
        'desc': 'YQL scripts for altay db cooking',
        'suffix': 'yql_scripts'
    }
}

YA_PACKAGE_RESOURCES = {
    'java': 'java_binaries'
}

ALTAY_YT_CONFIG = {
    rm_const.ReleaseStatus.stable: {
        "yt_prefix": "//home/altay/db",
        "yt_server": "hahn.yt.yandex.net",
        "vault_yt_token": "altay_db_yt_token"
    },
    rm_const.ReleaseStatus.prestable: {
        "yt_prefix": "//home/altay-dev/nightly/db",
        "yt_server": "hahn.yt.yandex.net",
        "vault_yt_token": "altay_db_dev_yt_token"
    }
}


class AltayCfg(configs.ReferenceCIConfig):
    name = "altay"
    responsible = configs.Responsible(
        abc=configs.Abc(service_name="spravdata"),
        login="ar7is7",
    )
    robot = "robot-altay"

    class JG(jg_base.BaseReleaseMachineJobGraph):

        def _get_ya_make_build_cubes(self):

            result = []

            default_input_kwargs = {
                sandbox_constants.DEFINITION_FLAGS_KEY: '-DDEBUGINFO_LINES_ONLY=yes',
                sandbox_constants.BUILD_SYSTEM_KEY: sandbox_constants.YMAKE_BUILD_SYSTEM,
                sandbox_constants.YA_YT_DIR: '//home/altay/yamake_cache',
                sandbox_constants.YA_YT_PROXY: 'arnold.yt.yandex.net',
                sandbox_constants.YA_YT_PUT: True,
                sandbox_constants.YA_YT_STORE: True,
                sandbox_constants.YA_YT_TOKEN_VAULT_NAME: 'altay_db_yt_token',
                sandbox_constants.YA_YT_TOKEN_VAULT_OWNER: 'ALTAY',
            }

            for target, meta in iteritems(YA_MAKE_RESOURCES):

                actual_resource_type = "ALTAY_{}".format(meta['suffix'].upper())

                result.append(
                    build_cubes.KosherYaMake(
                        name="build_{}".format(meta["suffix"].lower()),
                        title="Build {} {}".format(
                            self.component_name.capitalize(),
                            meta["suffix"].capitalize(),
                        ),
                        targets="sprav/altay/packages/{}".format(target),
                        artifacts="sprav/altay/packages/{}".format(target),
                        resource_types="",  # use result_rt instead of result_resource_types
                        input=jg_cube.CubeInput(
                            result_rt=actual_resource_type,
                            result_rd=meta["desc"],
                            result_single_file=True,
                            result_attrs={
                                "arcadia_tag": "altay-{}-{}".format(
                                    rm_const.CIJMESPathCommon.MAJOR_RELEASE_NUM,
                                    rm_const.CIJMESPathCommon.MINOR_RELEASE_NUM
                                )
                            },
                            **default_input_kwargs
                        ),
                        attributes={
                            "requirements": {
                                "disk": "50GB",
                            },
                        },
                    ),
                )

            return result

        def _get_ya_package_build_cubes(self):

            result = []

            default_input_kwargs = {
                sandbox_constants.YA_YT_DIR: '//home/altay/yamake_cache',
                sandbox_constants.YA_YT_PROXY: 'arnold.yt.yandex.net',
                sandbox_constants.YA_YT_PUT: True,
                sandbox_constants.YA_YT_STORE: True,
                sandbox_constants.YA_YT_TOKEN_VAULT_NAME: 'altay_db_yt_token',
                sandbox_constants.YA_YT_TOKEN_VAULT_OWNER: 'ALTAY',
            }

            for target, resource_type in iteritems(YA_PACKAGE_RESOURCES):

                actual_resource_type = 'ALTAY_{}'.format(resource_type.upper())

                result.append(
                    build_cubes.KosherYaPackage(
                        name="build_{}".format(resource_type.lower()),
                        title="Build {}".format(resource_type.upper()),
                        packages=["sprav/altay/packages/{}/pkg.json".format(target)],
                        resource_types=[actual_resource_type],
                        use_compression=False,
                        input=jg_cube.CubeInput(
                            package_resource_attrs={
                                "arcadia_tag": "altay-{}-{}".format(
                                    rm_const.CIJMESPathCommon.MAJOR_RELEASE_NUM,
                                    rm_const.CIJMESPathCommon.MINOR_RELEASE_NUM
                                )
                            },
                            **default_input_kwargs),
                        attributes={
                            "requirements": {
                                "disk": "50GB",
                            },
                        },
                    ),
                )

            return result

        def _get_release_cubes(self, graph):
            result = []
            resources = {}

            for name, resource_type in YA_PACKAGE_RESOURCES.items():
                resource_name = "altay_{}".format(resource_type)
                resources[resource_name] = graph.get(
                    "build_{}".format(resource_type)
                ).output.resources[resource_name.upper()].first().id

            for name, resource_info in YA_MAKE_RESOURCES.items():
                resource_name = "altay_{}".format(resource_info["suffix"])
                resources[resource_name] = graph.get(
                    "build_{}".format(resource_info["suffix"])
                ).output.resources[resource_name.upper()].first().id

            for release_type, ctx in iteritems(ALTAY_YT_CONFIG):
                result.append(
                    release_cubes.AltaySetReleaseMeta(
                        component_name=self.component_name,
                        where_to_release=release_type,
                        input=jg_cube.CubeInput(component_resources=resources, **ctx),
                    ),
                )

            return result

        @flow.release_flow()
        def release(self):
            graph = super(AltayCfg.JG, self).release(self)

            ya_make_build_cubes = self._get_ya_make_build_cubes()
            ya_package_build_cubes = self._get_ya_package_build_cubes()
            all_builds = ya_make_build_cubes + ya_package_build_cubes

            for build in ya_make_build_cubes + ya_package_build_cubes:
                build.add_requirement(graph.get(dummy_cubes.RMMainGraphEntry.NAME))
                graph.add(build)

            altay_release_cubes = self._get_release_cubes(graph)
            for build in all_builds:
                for release in altay_release_cubes:
                    release.add_requirement(build)

            for release in altay_release_cubes:
                graph.add(release)

            return graph

    class CI(configs.ReferenceCIConfig.CI):
        a_yaml_dir = "sprav/altay"
        secret = "sec-01e12atetdnyr07vr6xds702s9"
        sb_owner_group = "ALTAY"
        grant_config_update_permissions_to_release_machine_robot = False

        ya_make_abs_paths_glob = [
            "sprav/altay/**",
            "sprav/algorithms/**",
            "sprav/feature_vault/**",
            "sprav/feature_vault_v2/**",
            "sprav/java/**",
            "sprav/lib/**",
            "sprav/protos/**",
            "sprav/rating/**",
            "sprav/scripts/**",
        ]

    class Releases(configs.ReferenceCIConfig.Releases):
        # we want to release one branch to stable and the other to test
        # but the release for testing is considered in_progress until it is deployed to stable
        # so we need this flag to allow fixes in the production branch
        main_release_flow_independent_stages = True

        @property
        def releasable_items(self):
            ya_make_resources = []
            for i in YA_MAKE_RESOURCES.values():
                item=ri.ReleasableItem(
                    name="altay_{}".format(i["suffix"]),
                    data=ri.SandboxResourceData("ALTAY_{}".format(i["suffix"].upper())),
                )
                if i["suffix"] == "cmpy":
                    item.deploy_infos=[
                        ri.single_nanny_service("sprav"),
                        ri.single_nanny_service("sprav_dev", stage=rm_const.ReleaseStatus.prestable),
                    ]
                ya_make_resources.append(item)

            ya_package_resources=[
                ri.ReleasableItem(
                    name="altay_{}".format(i),
                    data=ri.SandboxResourceData("ALTAY_{}".format(i.upper())),
                ) for i in YA_PACKAGE_RESOURCES.values()
            ]

            return ya_make_resources + ya_package_resources

    class Notify(configs.ReferenceCIConfig.Notify):
        class Mail(configs.ReferenceCIConfig.Notify.Mail):
            mailing_list = [
                'sprav-dev@yandex-team.ru',
            ]

        class Startrek(configs.ReferenceCIConfig.Notify.Startrek):
            assignee = 'ar7is7'
            queue = 'ALTAYRELEASE'
            dev_queue = 'ALTAY'
            summary_template = u'Altay {}'
            workflow = {
                'open': 'autoTesting',
                'fixProblems': 'accepting',
                'production': 'close',
                'closed': 'reopen',
                'qualityOK': 'deploying',
                'accepting': 'qualityOK',
                'autoTesting': 'autoTestsOK',
                'autoTestsOK': 'accepting',
                'deploying': 'production',
            }
            followers = []
            add_commiters_as_followers = True
            deadline = 7

    class ChangelogCfg(configs.ReferenceCIConfig.ChangelogCfg):
        wiki_page = 'altay/changelogs/'  # TODO: fix it
        dirs = [
            'arcadia/sprav'
        ]
        review_groups = [
            "sprav-infra",
        ]

    class SvnCfg(configs.ReferenceCIConfig.SvnCfg):
        start_version = 104
