# -*- coding: utf-8 -*-

from sandbox.projects.common.constants import constants as common_const
from sandbox.projects.release_machine.components import configs
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.releasable_items as ri
import sandbox.projects.release_machine.components.config_core.statistics_page as statistics_page
import sandbox.projects.release_machine.components.config_core.yappy as yappy_cfg
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils


class AppHostResourceDescription:
    def __init__(self, resource_type, local_path, deploy_params=None):
        self.resource_type, self.local_path, self.deploy_params = resource_type, local_path, deploy_params

        # Used for human description
        self.released_resource_name = resource_type.lower()

    def to_parent_data_dict(self):
        return jg_job_data.ParentDataDict(
            "component_resources",
            self.released_resource_name,
            self.resource_type
        )

    def to_releasable_item(self):
        return ri.ReleasableItem(
            name=self.released_resource_name,
            data=ri.SandboxResourceData(self.resource_type),
            deploy_infos=self.deploy_params
        )

    def to_yappy_parametrized_resource(self):
        return yappy_cfg.YappyParametrizedResource(
            param_name=self.released_resource_name,
            local_path=self.local_path
        )


# This is used by RM to to identify current stable release.
STABLE_RESOURCE_DEPLOY_PARAMS = [ri.single_nanny_service("production_app_host_vla_web_yp")]

APPHOST_RESOURCES = [
    AppHostResourceDescription(
        "APP_HOST_DAEMON_EXECUTABLE", "app_host",
        deploy_params=STABLE_RESOURCE_DEPLOY_PARAMS,
    ),
    AppHostResourceDescription(
        "APP_HOST_DAEMON_EXECUTABLE_WEB", "app_host",
        deploy_params=STABLE_RESOURCE_DEPLOY_PARAMS,
    ),
    AppHostResourceDescription("APP_HOST_EVLOGDUMP_EXECUTABLE", "evlogdump"),
    AppHostResourceDescription("APP_HOST_SERVANT_CLIENT_EXECUTABLE", "servant_client"),
    AppHostResourceDescription("APP_HOST_GRPC_CLIENT_EXECUTABLE", "grpc_client"),
    AppHostResourceDescription("APP_HOST_INSTANCECTL_CONF", "instancectl.conf"),
    AppHostResourceDescription("APP_HOST_CONFIG_BUNDLE_UNIFIED", "configs.tar.gz"),
    AppHostResourceDescription("HORIZON_NORA_AGENT_BINARY", "horizon_agent"),
    AppHostResourceDescription("APPHOST_CERT", "allCAs.pem"),
]
APPHOST_YAPPY_RESOURCES = [
    r for r in APPHOST_RESOURCES
    if r.local_path not in {"instancectl.conf"}
]


def get_resource_name(vertical):
    if vertical == "web":
        return "APP_HOST_DAEMON_EXECUTABLE_WEB"

    return "APP_HOST_DAEMON_EXECUTABLE"


nanny_service_template = {
    "web": "hamster_app_host_sas_web_yp",
    "imgs": "hamster_app_host_sas_imgs_yp",
    "video": "hamster_app_host_sas_video_yp",
}


class AppHostCfg(configs.ReferenceBranchedConfig):
    name = "app_host"
    responsible = "feldsherov"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_task_owner = "APP_HOST_RELEASE_MACHINE"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            APP_HOST_METRICS_SLA_PROJECT = "aa8286386850f0df016874cc48882d44"

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="BUILD_APP_HOST_EXECUTABLES",
                        build_item="RELEASE_EXECUTABLES",
                        out={
                            res.resource_type: 90 for res in APPHOST_RESOURCES
                        },
                        ctx={
                            common_const.USE_AAPI_FUSE: True,
                            common_const.USE_ARC_INSTEAD_OF_AAPI: True,
                            common_const.THINLTO: True,
                        },
                    )
                )

                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="TEST_YA_TOOL_APPHOST",
                        job_params={
                            "job_type": "YA_TOOL_APPHOST",
                            "job_name_parameter": "TEST"
                        }
                    )
                )
                for vertical in ["web", "video", "imgs", "test"]:
                    beta_vertical = 'imgs-yp' if vertical == 'imgs' else vertical
                    branch_part.append(
                        jg_test.JobGraphElementYappyBetaGeneratorBranched(
                            beta_conf_type="app-host-{}".format(beta_vertical),
                            job_params={
                                "job_name_parameter": vertical.upper(),
                            },
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=tuple(
                                        res.to_parent_data_dict() for res in APPHOST_RESOURCES
                                    ),
                                    job_name_parameter="RELEASE_EXECUTABLES",
                                ),
                            ),
                        )
                    )
                    if vertical == "test":
                        continue
                    full_vertical = "images" if vertical == "imgs" else vertical
                    branch_part.append(
                        jg_test.JobGraphElementLaunchMetrics(
                            search_subtype=full_vertical,
                            job_name_parameter=vertical,
                            job_arrows=(
                                jg_job_triggers.JobTriggerGenerateBeta(
                                    parent_job_data=(
                                        jg_job_data.ParentDataCtx(
                                            input_key='checked_beta',
                                            output_key='beta_name',
                                            transform=lambda x, params: '{}.hamster'.format(x),
                                        )
                                    ),
                                    job_name_parameter=vertical.upper(),
                                )
                            ),
                            ctx={
                                "custom_template_name": "experiments.json",
                                "beta_conf_type": "app-host-{}".format(beta_vertical),
                                "scraper_over_yt_pool": "apphost_{}_priemka".format(full_vertical),
                                "sla_project": self.APP_HOST_METRICS_SLA_PROJECT,
                                "sample_beta": "hamster",
                            }
                        )
                    )
                    branch_part.append(
                        jg_test.JobGraphElementTestBranchCommon(
                            task_name="CHECK_GOLOVAN_SIGNALS_QUOTA",
                            job_params={
                                "job_type": "CHECK_APP_HOST_GOLOVAN_SIGNALS_QUOTA",
                                "job_name_parameter": vertical.upper(),
                                "ignore_parent_job_error": False
                            },
                            job_arrows=(
                                jg_job_triggers.JobTriggerGenerateBeta(
                                    parent_job_data=(
                                        jg_job_data.ParentDataCtx(
                                            input_key='beta_name',
                                            output_key='beta_name'
                                        )
                                    ),
                                    job_name_parameter=vertical.upper()
                                )
                            ),
                            ctx={
                                "signals_quota": 25000
                            }
                        )
                    )
                    branch_part.append(
                        jg_test.JobGraphElementTestBranchCommon(
                            task_name="APPHOST_PERF_TESTING",
                            job_params={
                                "job_name_parameter": vertical,
                                "job_type": "APPHOST_PERF_TESTING",
                            },
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter="RELEASE_EXECUTABLES",
                                    parent_job_data=(
                                        jg_job_data.ParentDataResource(
                                            input_key="apphost_executable",
                                            resource_name=get_resource_name(vertical)
                                        )
                                    )
                                )
                            ),
                            ctx={
                                "vertical_name": vertical,
                                "ctype": "hamster",
                                "logs_amount": "1000",
                                "nanny_service": nanny_service_template[vertical],
                                "tank_scheduling_config": "line(1,25,10m)",
                                "chosen_task": "compare",
                            }
                        )
                    )
                branch_part.append(
                    jg_test.JobGraphElementTestXMLSearch(
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                jg_job_data.ParentDataCtx(
                                    input_key="beta_url",
                                    output_key="beta_name",
                                    transform=lambda x, params: "https://{}.hamster.yandex.ru".format(x),
                                ),
                                job_name_parameter="WEB",
                            ),
                            jg_job_triggers.JobTriggerGenerateBeta(
                                jg_job_data.ParentDataCtx(
                                    input_key="images_beta_url",
                                    output_key="beta_name",
                                    transform=lambda x, params: "https://{}.hamster.yandex.ru".format(x),
                                ),
                                job_name_parameter="IMGS",
                            ),
                            jg_job_triggers.JobTriggerGenerateBeta(
                                jg_job_data.ParentDataCtx(
                                    input_key="video_beta_url",
                                    output_key="beta_name",
                                    transform=lambda x, params: "https://{}.hamster.yandex.ru".format(x),
                                ),
                                job_name_parameter="VIDEO",
                            ),
                        )
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="APP_HOST_LOAD_TEST",
                        job_params={
                            "job_type": "APP_HOST_TANK_SHOOTING",
                            "job_name_parameter": "TEST",
                            "task_priority": 7,
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="beta",
                                        output_key="beta_name",
                                    ),
                                ),
                                job_name_parameter="TEST",
                            ),
                        ),
                        ctx={
                            "tank_names": "nanny:production_yandex_tank",
                            "type_or_quota": "app-host-perf-beta",
                            "tests_cgi": [
                                "/echo_test?length=50",
                                "/echo_test?length=100000",
                                "/echo_test?fast_error=1",
                            ],
                            "load_plan": "line(1,500,1000s)",
                            "time_threshold": 200,
                            "ticket": "APPHOST-3730",
                            "report_to_release_ticket": True,
                            "rm_component": "app_host",
                            "nanny_token": "nanny_oauth_token",
                        }
                    )
                )
                umbrella_parents = (
                    jg_job_triggers.JobTriggerLaunchMetrics(
                        job_name_parameter="WEB",
                    ),
                    jg_job_triggers.JobTriggerLaunchMetrics(
                        job_name_parameter="VIDEO",
                    ),
                    jg_job_triggers.JobTriggerLaunchMetrics(
                        job_name_parameter="IMGS",
                    ),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_type="CHECK_APP_HOST_GOLOVAN_SIGNALS_QUOTA",
                        job_name_parameter="WEB",
                    ),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_type="CHECK_APP_HOST_GOLOVAN_SIGNALS_QUOTA",
                        job_name_parameter="VIDEO",
                    ),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_type="CHECK_APP_HOST_GOLOVAN_SIGNALS_QUOTA",
                        job_name_parameter="IMGS",
                    ),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_type="APP_HOST_TANK_SHOOTING",
                        job_name_parameter="TEST",
                    ),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_type="APPHOST_PERF_TESTING",
                        job_name_parameter="WEB",
                    ),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_type="APPHOST_PERF_TESTING",
                        job_name_parameter="IMGS",
                    ),
                    jg_job_triggers.JobTriggerTestBranchCommon(
                        job_type="APPHOST_PERF_TESTING",
                        job_name_parameter="VIDEO",
                    ),
                    jg_job_triggers.JobTriggerTestXMLSearch(),
                )
                branch_part.append(
                    jg_test.JobGraphElementActionRunAcceptanceBranchedScheduled(
                        job_arrows=umbrella_parents,
                        job_params={
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementActionRunAcceptanceBranchedByMarker(
                        job_arrows=umbrella_parents,
                        job_params={
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                    )
                )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                for release_stage in (rm_const.ReleaseStatus.prestable, rm_const.ReleaseStatus.stable):
                    release_part.append(jg_release.JobGraphElementReleaseBranched(
                        release_to=release_stage,
                        job_arrows=[
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=tuple(
                                    res.to_parent_data_dict() for res in APPHOST_RESOURCES
                                ),
                                job_name_parameter="RELEASE_EXECUTABLES",
                            ),
                        ],
                    ))
                    release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                        release_to=release_stage,
                        job_params={
                            "frequency": (jg_utils.TestFrequency.LAZY, None),
                        }
                    ))
                return release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            @property
            def change_frequency(self):
                uppercase_name = self.name.upper()
                tests = super(self.__class__, self).change_frequency
                tests.update({
                    "_TESTING_RELEASE_{}".format(uppercase_name): rm_const.TestFrequencies.EACH_REV_TEST,
                })
                return tests

    class Releases(configs.ReferenceBranchedConfig.Releases):
        releasable_items = [
            res.to_releasable_item() for res in APPHOST_RESOURCES
        ]
        release_followers_permanent = [
            "vlad-kolotvin",
            "feldsherov",
        ]
        allow_robots_to_release_stable = True
        allow_old_releases = True

    class Notify(configs.ReferenceBranchedConfig.Notify):
        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            chats = ["apphost-releases"]
            config = configs.RmTelegramNotifyConfig(chats=chats)
            invite_link = u"https://t.me/joinchat/AzzAPEWLMBkOSt4-FHaFxg"

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            assignee = "feldsherov"
            queue = "APPHOSTRELEASES"
            dev_queue = "APPHOST"
            summary_template = u"Приемка AppHost {}"
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = [
                "vlad-kolotvin",
                "feldsherov",
            ]
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 7
            commit_importance_threshold = 5

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = "apphost/changelog/"
        ya_make_targets = [
            "arcadia/apphost/daemons/app_host",
            "arcadia/apphost/conf/combinator",
        ]
        review_groups = [
            "apphost",
        ]
        svn_paths_filter = configs.ChangelogPathsFilter(rm_const.PermissionType.CHANGED, ["arcadia/apphost"], 5)

        @staticmethod
        def calculate_importance(changelog_entry):
            for action, path in changelog_entry.revision_paths:  # Path is pair [change_type, change_path]
                if "apphost" in path:
                    return 5
            return min(4, changelog_entry.mark)

    class MetricsCfg(configs.ReferenceBranchedConfig.MetricsCfg):
        limit_s = None

    class Yappy(yappy_cfg.YappyBaseCfg):

        betas = {
            "app-host-web": yappy_cfg.YappyTemplateCfg(
                template_name="app-host-web",
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="app-host-web",
                        resources=[
                            res.to_yappy_parametrized_resource() for res in APPHOST_YAPPY_RESOURCES
                            if res.resource_type != "APP_HOST_DAEMON_EXECUTABLE"
                        ],
                        parent_service="hamster_app_host_sas_web_yp",
                    )
                ]
            ),
            "app-host-imgs-yp": yappy_cfg.YappyTemplateCfg(
                template_name="app-host-imgs",
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="app-host-imgs",
                        resources=[
                            res.to_yappy_parametrized_resource() for res in APPHOST_YAPPY_RESOURCES
                            if res.resource_type != "APP_HOST_DAEMON_EXECUTABLE_WEB"
                        ] + [
                            # Nora migration hack (roboslone@).
                            yappy_cfg.YappyStaticResource(
                                "horizon_agent.ctype-geo.yaml",
                                manage_type="SANDBOX_RESOURCE",
                                resource_id="1656320577",
                            ),
                        ],
                        parent_service="hamster_app_host_vla_imgs_yp",
                    )
                ]
            ),
            "app-host-video": yappy_cfg.YappyTemplateCfg(
                template_name="app-host-video",
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="app-host-video",
                        resources=[
                            res.to_yappy_parametrized_resource() for res in APPHOST_YAPPY_RESOURCES
                            if res.resource_type != "APP_HOST_DAEMON_EXECUTABLE_WEB"
                        ] + [
                            # Nora migration hack (roboslone@).
                            yappy_cfg.YappyStaticResource(
                                "horizon_agent.ctype-geo.yaml",
                                manage_type="SANDBOX_RESOURCE",
                                resource_id="1656320577",
                            ),
                        ],
                        parent_service="hamster_app_host_vla_video_yp",
                    )
                ]
            ),
            "app-host-test": yappy_cfg.YappyTemplateCfg(
                template_name="app-host-test",
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="app-host-test",
                        resources=[
                            res.to_yappy_parametrized_resource() for res in APPHOST_YAPPY_RESOURCES
                            if res.resource_type != "APP_HOST_DAEMON_EXECUTABLE_WEB"
                        ],
                        parent_service="hamster_app_host_sas_test",
                    )
                ],
                stop_betas_gap=0,
            ),
        }

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        use_arc = True
        use_zipatch_only = True
        allow_autobranches = True

    class ReleaseViewer(configs.ReferenceBranchedConfig.ReleaseViewer):
        statistics_page_charts = statistics_page.DEFAULT + [statistics_page.PredefinedCharts.TIMESPECTRE_METRICS]
