# -*- coding: utf-8 -*-

from sandbox.projects.release_machine.components import configs
import sandbox.projects.release_machine.core.releasable_items as ri
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.components.config_core.statistics_page as statistics_page
import sandbox.projects.release_machine.components.config_core.yappy as yappy_cfg
from sandbox.projects.release_machine.components.config_core.jg import cube as jg_cube
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import metrics as metrics_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import build as build_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import dummy as dummy_cubes
from sandbox.projects.release_machine.components.config_core.jg.preset import basic_build_presets
from sandbox.projects.release_machine.components.config_core.jg import flow


# This is used by RM to to identify current stable release.
RESOURCE_DEPLOY_PARAMS = [ri.single_nanny_service("production_app_host_vla_web_yp", stage="stable"),
                          ri.single_nanny_service("prestable_app_host_sas_web_yp", stage="prestable")]


class AppHostResourceDescription:
    def __init__(self, resource_type, local_path):
        self.resource_type, self.local_path = resource_type, local_path

        # Used for human description
        self.released_resource_name = resource_type.lower()

    def to_releasable_item(self):
        return ri.ReleasableItem(
            name=self.released_resource_name,
            data=ri.SandboxResourceData(self.resource_type),
            deploy_infos=RESOURCE_DEPLOY_PARAMS
        )

    def to_yappy_parametrized_resource(self):
        return yappy_cfg.YappyParametrizedResource(
            param_name=self.released_resource_name,
            local_path=self.local_path
        )

APPHOST_RESOURCES = [
    AppHostResourceDescription("APP_HOST_DAEMON_EXECUTABLE", "app_host"),
    AppHostResourceDescription("APP_HOST_DAEMON_EXECUTABLE_WEB", "app_host"),
    AppHostResourceDescription("APP_HOST_EVLOGDUMP_EXECUTABLE", "evlogdump"),
    AppHostResourceDescription("APP_HOST_SERVANT_CLIENT_EXECUTABLE", "servant_client"),
    AppHostResourceDescription("APP_HOST_GRPC_CLIENT_EXECUTABLE", "grpc_client"),
    AppHostResourceDescription("APP_HOST_INSTANCECTL_CONF", "instancectl.conf"),
    AppHostResourceDescription("APP_HOST_CONFIG_BUNDLE_UNIFIED", "configs.tar.gz"),
    AppHostResourceDescription("HORIZON_NORA_AGENT_BINARY", "horizon_agent"),
    AppHostResourceDescription("APPHOST_CERT", "allCAs.pem"),
]

APPHOST_YAPPY_RESOURCES = [
    r for r in APPHOST_RESOURCES
    if r.local_path not in {"instancectl.conf"}
]


def get_metrics_cube(vertical):
    if vertical == "web":
        return metrics_cubes.SearchLaunchMetricsWeb
    elif vertical == "imgs":
        return metrics_cubes.SearchLaunchMetricsImages
    elif vertical == "video":
        return metrics_cubes.SearchLaunchMetricsVideo


class ApphostCfg(configs.ReferenceCIConfig):
    name = "app_host_ci"
    display_name = "Apphost CI"
    responsible = configs.Responsible(
        abc=configs.Abc(service_name="apphost"),
        login="volcolac",
    )

    class CI(configs.ReferenceCIConfig.CI):
        display_name = "Apphost"
        a_yaml_dir = "apphost/daemons/app_host"
        secret = "sec-01desry8fbgvnkbeybem81ferv"
        sb_owner_group = "APP_HOST_RELEASE_MACHINE"
        ya_make_abs_paths_glob = [
            'apphost/lib/**',
            'apphost/conf/combinator/**',
        ]
        grant_config_update_permissions_to_release_machine_robot = False

    class JG(basic_build_presets.SingleBuildYaMakeTemplateJGCfg):
        build_task = "projects/apphost/build_apphost_executables"
        add_beta_generator = True
        apphost_metrics_sla_project = "aa8286386850f0df016874cc48882d44"

        def get_launch_metrics_cube(self, vertical, beta):
            subtype = "images" if vertical == "imgs" else vertical

            return get_metrics_cube(vertical)(
                search_subtype=subtype,
                name="launch_metrics_{}".format(vertical),
                title="Test launch metrics {}".format(vertical),
                component_name=self.component_name,
                input=jg_cube.CubeInput(
                    beta_conf_type="app-host-{}".format(vertical),
                    checked_beta=beta.output_params.new_beta_url,
                    custom_template_name="experiments.json",
                    scraper_over_yt_pool="apphost_{}_priemka".format(subtype),
                    sla_project=self.apphost_metrics_sla_project,
                    sample_beta="hamster"
                )
            )

        def _get_build_cube(self, graph):
            return build_cubes.YaMakeTemplate(
                target_resources=[ri.data.resource_type for ri in self._get_suitable_releasable_items()],
                name="build",
                task=self.build_task,
                needs=[
                    graph.get(dummy_cubes.RMMainGraphEntry.NAME),
                ],
                input=jg_cube.CubeInput(
                    checkout_arcadia_from_url="arcadia-arc:/#${tasks.new_tag.rm_events[2].new_tag_data.arcadia_path}",
                ),
            )

        @flow.release_flow(stages=basic_build_presets.DEFAULT_RELEASE_FLOW_STAGES)
        def release(self):
            graph = super(self.__class__, self).release(self)

            release_dummy = graph.get("release_stage_entry")
            for vertical in ["web", "video", "imgs"]:
                beta = graph.get("generate_beta__app_host_ci__app_host_{}".format(vertical))
                launch_metrics = self.get_launch_metrics_cube(vertical, beta)
                launch_metrics.add_requirement(beta)
                release_dummy.add_requirement(launch_metrics)
                graph.add(launch_metrics)

            beta = graph.get("generate_beta__app_host_ci__app_host_test")
            load_test_cube = jg_cube.Cube(
                name="apphost_load_test",
                title="Apphost load test",
                task="projects/apphost/app_host_load_test",
                input=jg_cube.CubeInput(
                    beta=beta.output_params.new_beta_name,
                    tank_names="nanny:production_yandex_tank",
                    type_or_quota="app-host-perf-beta",
                    tests_cgi=[
                        "/echo_test?length=50",
                        "/echo_test?length=100000",
                        "/echo_test?fast_error=1",
                    ],
                    load_plan="line(1,500,1000s)",
                    time_threshold=200,
                    ticket="APPHOST-3730",
                    report_to_release_ticket=True,
                    rm_component="app_host",
                    nanny_token="nanny_oauth_token",
                )
            )

            load_test_cube.add_requirement(beta)
            release_dummy.add_requirement(load_test_cube)
            graph.add(load_test_cube)

            return graph

    class Releases(configs.ReferenceCIConfig.Releases):
        @property
        def releasable_items(self):
            return [res.to_releasable_item() for res in APPHOST_RESOURCES]

        allow_robots_to_release_stable = True
        allow_old_releases = True
        main_release_flow_independent_stages = True

        release_followers_permanent = [
            "volcolac",
        ]

    class Notify(configs.ReferenceCIConfig.Notify):
        class Telegram(configs.ReferenceCIConfig.Notify.Telegram):
            chats = ["apphost-releases"]
            config = configs.RmTelegramNotifyConfig(chats=chats)
            invite_link = u"https://t.me/joinchat/AzzAPEWLMBkOSt4-FHaFxg"

        class Startrek(configs.ReferenceCIConfig.Notify.Startrek):
            assignee = "volcolac"
            queue = "APPHOSTRELEASES"
            dev_queue = "APPHOST"
            summary_template = u"Приемка AppHost {}"
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = [
                "volcolac"
            ]
            add_commiters_as_followers = True
            use_task_author_as_assignee = False
            deadline = 7
            commit_importance_threshold = 5

    class SvnCfg(configs.ReferenceCIConfig.SvnCfg):
        use_arc = True
        use_zipatch_only = True

        start_version = 303

    class ChangelogCfg(configs.ReferenceCIConfig.ChangelogCfg):
        wiki_page = "apphost/changelog/"
        ya_make_targets = [
            "arcadia/apphost/daemons/app_host",
            "arcadia/apphost/conf/combinator",
        ]

        use_previous_branch_as_baseline = True

        review_groups = [
            "apphost",
        ]
        svn_paths_filter = configs.ChangelogPathsFilter(rm_const.PermissionType.CHANGED, ["arcadia/apphost"], 5)

        @staticmethod
        def calculate_importance(changelog_entry):
            for action, path in changelog_entry.revision_paths:  # Path is pair [change_type, change_path]
                if "apphost" in path:
                    return 5
            return min(4, changelog_entry.mark)

    class Yappy(configs.ReferenceCIConfig.Yappy):
        betas = {
            "app_host_web": yappy_cfg.YappyTemplateCfg(
                template_name="app-host-web",
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="app-host-web",
                        resources=[
                            res.to_yappy_parametrized_resource() for res in APPHOST_YAPPY_RESOURCES
                            if res.resource_type != "APP_HOST_DAEMON_EXECUTABLE"
                        ],
                        parent_service="hamster_app_host_sas_web_yp",
                    )
                ]
            ),
            "app_host_imgs": yappy_cfg.YappyTemplateCfg(
                template_name="app-host-imgs",
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="app-host-imgs",
                        resources=[
                            res.to_yappy_parametrized_resource() for res in APPHOST_YAPPY_RESOURCES
                            if res.resource_type != "APP_HOST_DAEMON_EXECUTABLE_WEB"
                        ],
                        parent_service="hamster_app_host_vla_imgs_yp",
                    )
                ]
            ),
            "app_host_video": yappy_cfg.YappyTemplateCfg(
                template_name="app-host-video",
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="app-host-video",
                        resources=[
                            res.to_yappy_parametrized_resource() for res in APPHOST_YAPPY_RESOURCES
                            if res.resource_type != "APP_HOST_DAEMON_EXECUTABLE_WEB"
                        ],
                        parent_service="hamster_app_host_vla_video_yp",
                    )
                ]
            ),
            "app_host_test": yappy_cfg.YappyTemplateCfg(
                template_name="app-host-test",
                new_yappy=True,
                patches=[
                    yappy_cfg.YappyTemplatePatch(
                        patch_dir="app-host-test",
                        resources=[
                            res.to_yappy_parametrized_resource() for res in APPHOST_YAPPY_RESOURCES
                            if res.resource_type != "APP_HOST_DAEMON_EXECUTABLE_WEB"
                        ],
                        parent_service="hamster_app_host_sas_test",
                    )
                ],
                stop_betas_gap=0,
            ),
        }

    class MetricsCfg(configs.ReferenceCIConfig.MetricsCfg):
        limit_s = None

    class ReleaseViewer(configs.ReferenceCIConfig.ReleaseViewer):
        statistics_page_charts = statistics_page.DEFAULT + [statistics_page.PredefinedCharts.TIMESPECTRE_METRICS]
