# -*- coding: utf-8 -*-

import os
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.components.config_core.jg import flow as jg_flow
from sandbox.projects.release_machine.components.config_core.jg.graph import base as graph_base
from sandbox.projects.release_machine.components.config_core.jg.preset import basic_build_presets
from sandbox.projects.release_machine.components.config_core.jg.cube.lib import release as release_cubes
from sandbox.projects.release_machine.components.config_core.jg.cube.lib.upper_search.base import IceFlamePatched
from sandbox.projects.release_machine.components.config_core.jg.cube import base as jg_cube
from sandbox.projects.release_machine.components.configs.apphost_cachedaemon import CachedaemonTestsBundle


EXECUTABLE_PATH = "web/daemons/cached"
CONF_PATH = "web/daemons/cached/conf"
INSTANCECTL_PATH = "web/daemons/cached/deploy/instancectl"

HAMSTER_SERVICES = [
    "hamster-apphost-cache-daemon-sas-web-yp",
    "hamster-apphost-cache-daemon-man-web-yp",
    "hamster-apphost-cache-daemon-vla-web-yp",
]

PRODUCTION_SERVICES = [
    "production-apphost-cache-daemon-sas-web-yp",
    "production-apphost-cache-daemon-man-web-yp",
    "production-apphost-cache-daemon-vla-web-yp",
]

SANITIZERS = ["address", "thread", "memory", "leak", "undefined"]


class ApphostCacheDaemonCICfg(configs.ReferenceCIConfig):
    name = "apphost_cachedaemon_ci"
    display_name = "Apphost CacheDaemon CI"
    responsible = configs.Responsible(
        abc=configs.Abc(service_name="webcachedaemon"),
        login="vmordovin",
    )

    class CI(configs.ReferenceCIConfig.CI):
        a_yaml_dir = EXECUTABLE_PATH
        secret = "sec-01desry8fbgvnkbeybem81ferv"
        sb_owner_group = "NOAPACHEUPPER-ROBOT"

        ya_make_abs_paths_glob = [
            os.path.join(EXECUTABLE_PATH, "**"),
        ]

    class JG(basic_build_presets.SingleBuildYaMakeJGCfg):
        build_task = "KOSHER_YA_MAKE"

        # configuring auto-deploy
        release_manually = False

        @jg_flow.release_flow(
            stages=basic_build_presets.JOINED_BUILD_RELEASE_FLOW_STAGES,
        )
        def release(self):
            graph = super(self.__class__, self).release(self)

            main_stage_entry_cube = graph.get("main_graph_entry")
            release_entry_cube = graph.get("release_stage_entry")

            #  tests init
            tests = CachedaemonTestsBundle.get_bundle(
                sanitizers=SANITIZERS,
                entry_cube=main_stage_entry_cube,
            )

            # add cubes to graph
            for test in tests:
                release_entry_cube.add_requirement(test)
                graph.add(test)

            return graph

        # activate Morty release
        def _get_release_cubes_for_deploy_system(self, deploy_system, where, releasable_items, graph, build):
            return [
                release_cubes.ScheduleRelease(
                    name="release_{}_{}".format(where, deploy_system),
                    component_name=self.component_name,
                    where_to_release=where,
                    flows=["deploy"],
                    warden_component_name="cachedaemon",
                    warden_parent_component_name="web",
                    deploy_system=deploy_system,
                    schedule_mode="create_and_schedule",
                    input=jg_cube.CubeInput(
                        component_resources={
                            ri.name: build.output.resources[ri.data.resource_type].first().id
                            for ri in releasable_items
                        },
                    ),
                    manual=self.release_manually,
                ),
            ]

        @jg_flow.register_flow(title="Run IceFlame")
        def run_iceflame(self):
            return graph_base.Graph([IceFlamePatched(ApphostCacheDaemonCICfg.name, "cachedaemon", "hamster-apphost-cache-daemon-sas-web-yp", "apphost_cachedaemon.conf")])

    class Releases(configs.ReferenceCIConfig.Releases):
        release_followers_permanent = [
            "alex-ersh",
            "elshiko",
            "vmordovin",
        ]
        allow_robots_to_release_stable = True

        @property
        def releasable_items(self):
            deploy_infos = [
                ri.NannyDeployInfo(
                    [
                        ri.DeployService(service, tags=["hamster"]) for service in HAMSTER_SERVICES
                    ] + [
                        ri.DeployService(service, tags=["prod"]) for service in PRODUCTION_SERVICES
                    ]
                )
            ]

            return [
                ri.ReleasableItem(
                    name="cachedaemon",
                    data=ri.SandboxResourceData("CACHE_DAEMON"),
                    build_data=ri.BuildData(
                        target=EXECUTABLE_PATH,
                        artifact=os.path.join(EXECUTABLE_PATH, "cached"),
                    ),
                    deploy_infos=deploy_infos
                ),
                ri.ReleasableItem(
                    name="cachedaemon.cfg",
                    data=ri.SandboxResourceData("APPHOST_CACHEDAEMON_CONFIG"),
                    build_data=ri.BuildData(
                        target=CONF_PATH,
                        artifact=os.path.join(CONF_PATH, "cachedaemon.cfg"),
                    ),
                    deploy_infos=deploy_infos
                ),
                ri.ReleasableItem(
                    name="loop.conf",
                    data=ri.SandboxResourceData("APPHOST_CACHEDAEMON_INSTANCECTL"),
                    build_data=ri.BuildData(
                        target=INSTANCECTL_PATH,
                        artifact=os.path.join(INSTANCECTL_PATH, "instancectl.conf"),
                    ),
                    deploy_infos=deploy_infos
                ),
                ri.DynamicReleasableItem(
                    name="scheduled_bundle",
                    data=ri.SandboxResourceData("SCHEDULED_RM_RELEASE_DATA"),
                    deploy_infos=[ri.NannyDeployInfo()],
                ),
            ]

    class Notify(configs.ReferenceCIConfig.Notify):
        class Startrek(configs.ReferenceCIConfig.Notify.Startrek):
            assignee = "vmordovin"
            queue = "CACHEDAEMONREL"
            summary_template = u"Приемка Apphost Cachedaemon {}"
            add_commiters_as_followers = True
            deadline = 7

            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }

            use_task_author_as_assignee = False
            write_merges_in_table = True

    class ChangelogCfg(configs.ReferenceCIConfig.ChangelogCfg):
        observed_paths = [
            "arcadia/web/daemons/cached",
        ]

    class SvnCfg(configs.ReferenceCIConfig.SvnCfg):
        allow_autobranches = True

    class MetricsCfg(configs.ReferenceCIConfig.MetricsCfg):
        limit_s = None
