# -*- coding: utf-8 -*-

from sandbox.projects.release_machine.components import configs as cfg
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.common import constants as sandbox_constants
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows


class JobGraphElementArcDocs(jg_build.JobGraphElementBuildTrunk):
    def __init__(self, env, job_params):
        ctx = {
            "targets": "arc/docs-legacy",
            "arts": "arc/docs-legacy/docs.tar.gz",
            "result_single_file": True,
            "definition_flags": "-Dtheme=doccenter",
        }

        if env == "prod":
            ctx["daas_prod_project"] = 1190
        elif env == "test":
            ctx["daas_test_project"] = 1879

        super(JobGraphElementArcDocs, self).__init__(
            task_name="YA_MAKE_DAAS",
            job_params=job_params,
            build_item="DOCS_{}".format(env),
            ctx=ctx,
        )


class ArcClientCfg(cfg.ReferenceBranchedConfig):
    name = rm_const.RMNames.ARC_CLIENT
    display_name = "Arc VCS client"
    responsible = cfg.Responsible(abc=cfg.Abc(component_id=3185), login="nslus")

    class Testenv(cfg.ReferenceBranchedConfig.Testenv):
        trunk_task_owner = "ARC"

        class JobGraph(cfg.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _trunk_part(self):
                trunk_part = super(self.__class__, self)._trunk_part
                trunk_part.append(
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="BUILD_ARC_CLIENT_FOR_ALL",
                        job_params={
                            "job_name_parameter": "unstable_trunk",
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key=sandbox_constants.ARCADIA_URL_KEY,
                                transform=lambda x, rm_config: "arcadia:/arc/trunk/arcadia@{}".format(x.revision),
                                override=True,
                            ),
                        ),
                        ctx={
                            "project": "arc/local/bin/arc",
                            "platforms_list": ["linux", "linux-aarch64", "win32", "darwin", "darwin-arm64"],
                            "backup_to_mds": False,
                            sandbox_constants.USE_AAPI_FUSE: True,
                            sandbox_constants.USE_ARC_INSTEAD_OF_AAPI: True,
                            sandbox_constants.MINIMIZE_ARC_MOUNT_PATH: True,
                            "definition_flags": "-DARC_CHANNEL=unstable",
                        },
                        out={
                            "ARC_CLIENT": 90
                        },
                    )
                )
                trunk_part.append(
                    JobGraphElementArcDocs(
                        env="prod",
                        job_params={
                            "observed_paths": [
                                "arc/docs-legacy",
                            ],
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        },
                    )
                )
                return trunk_part

            @property
            def _branch_part(self):
                branch_part = []
                for channel in [rm_const.ReleaseStatus.stable, rm_const.ReleaseStatus.prestable]:
                    branch_part.append(
                        jg_build.JobGraphElementYaMakeBuildBranched(
                            task_name="BUILD_ARC_CLIENT_FOR_ALL",
                            job_params={
                                "job_name_parameter": channel,
                                "frequency": (jg_utils.TestFrequency.LAZY, None),
                            },
                            ctx={
                                "project": "arc/local/bin/arc",
                                "platforms_list": ["linux", "linux-aarch64", "win32", "darwin", "darwin-arm64"],
                                "backup_to_mds": False,
                                sandbox_constants.USE_AAPI_FUSE: True,
                                sandbox_constants.USE_ARC_INSTEAD_OF_AAPI: True,
                                sandbox_constants.MINIMIZE_ARC_MOUNT_PATH: True,
                                "definition_flags": "-DARC_CHANNEL={}".format(channel),
                            },
                            out={
                                "ARC_CLIENT": 90
                            },
                        )
                    )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                build_job_data = [
                    jg_job_data.ParentDataDict(
                        "component_resources",
                        "arc_client_{}".format(platform),
                        "ARC_CLIENT",
                    ) for platform in ["linux", "linux-aarch64", "win32", "darwin", "darwin-arm64"]
                ]
                for release_stage in [rm_const.ReleaseStatus.stable, rm_const.ReleaseStatus.prestable]:
                    release_part.append(
                        jg_release.JobGraphElementReleaseBranched(
                            release_to=release_stage,
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter=release_stage,
                                    parent_job_data=build_job_data,
                                ),
                            )
                        )
                    )
                    release_part.append(
                        jg_release.JobGraphElementActionReleaseBranched(
                            release_to=release_stage,
                        )
                    )
                release_part.append(
                    jg_release.JobGraphElementReleaseBase(
                        release_to=rm_const.ReleaseStatus.unstable,
                        job_params={
                            "observed_paths": [
                                "arc/api",
                                "arc/lib",
                                "arc/local",
                            ],
                            "should_add_to_db": jg_utils.should_add_to_db_trunk,
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                job_name_parameter="unstable_trunk",
                                parent_job_data=build_job_data,
                            )
                        )
                    )
                )
                release_part.append(
                    jg_release.JobGraphElementActionReleaseTagged(
                        release_to=rm_const.ReleaseStatus.unstable,
                        job_params={
                            "observed_paths": [
                                "arc/api",
                                "arc/lib",
                                "arc/local",
                            ],
                            "should_add_to_db": jg_utils.should_add_to_db_trunk,
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                        }
                    )
                )
                return release_part

        class JobPatch(cfg.ReferenceBranchedConfig.Testenv.JobPatch):
            @property
            def change_frequency(self):
                tests = super(self.__class__, self).change_frequency
                tests.update({
                    rm_const.JobTypes.rm_job_name(rm_const.JobTypes.BUILD, self.name): rm_const.TestFrequencies.LAZY_TEST
                })
                return tests

    class Notify(cfg.ReferenceBranchedConfig.Notify):

        class Startrek(cfg.ReferenceBranchedConfig.Notify.Startrek):
            assignee = "nslus"
            queue = "ARCREL"
            dev_queue = "ARC"
            summary_template = u"Arc client release {}"
            components = u"client"
            notify_on_deploy_to_feature_st_tickets = True

    class ChangelogCfg(cfg.ReferenceBranchedConfig.ChangelogCfg):
        dirs = [
            "arcadia/arc/api",
            "arcadia/arc/lib",
            "arcadia/arc/local",
        ]
        wiki_page = "arcadia/arc/changelogs/"

    class SvnCfg(cfg.ReferenceBranchedConfig.SvnCfg):
        branch_name = "arc"
        tag_name = "arc"
        arc_tag_folder = "tags/releases/experimental"
        arc_branches_folder = "releases/experimental"

        use_arc = True

    class Releases(cfg.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name="arc_client_{}".format(pl),
                    data=ri.SandboxResourceData("ARC_CLIENT", attributes={"platform": pl}),
                    deploy_infos=[
                        ri.SandboxInfo(stage=rm_const.ReleaseStatus.stable),
                        ri.SandboxInfo(stage=rm_const.ReleaseStatus.prestable),
                    ]
                ) for pl in ("linux", "linux-aarch64", "darwin", "darwin-arm64", "win32")
            ]
        allow_robots_to_release_stable = True
        deploy_system = rm_const.DeploySystem.sandbox
