# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import re

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.core import releasable_items as ri

import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils


deploy_environment_by_rm_status = {
    rm_const.ReleaseStatus.unstable: 'unstable',
    rm_const.ReleaseStatus.testing: 'testing',
    rm_const.ReleaseStatus.prestable: 'preproduction',
    rm_const.ReleaseStatus.stable: 'production',
}


def get_deploy_stage_name(rm_release_status, deploy_project):
    env = deploy_environment_by_rm_status[rm_release_status]
    return '{}-{}'.format(deploy_project, env)


def get_sandbox_resource_data(name):
    return ri.SandboxResourceData(
        name,
        docker_image_data_re=re.compile(r'stable-(?P<branch>.*)-(?P<tag>.*)\.(?P<revision>.*)\.(?P<task_id>.*)'),
    )


class AviaDockerToDeployJobGraphMixin(object):
    @property
    def avia_build_package(self):
        raise NotImplementedError()

    @property
    def avia_project_observed_paths(self):
        raise NotImplementedError()

    @property
    def avia_resource_name(self):
        raise NotImplementedError()

    @property
    def avia_trunk_auto_deploy_stages(self):
        raise NotImplementedError()

    @property
    def avia_release_button_deploy_stages(self):
        raise NotImplementedError()

    @property
    def avia_wait_for_deploy(self):
        return False

    @property
    def avia_wait_for_deploy_time_sec(self):
        return 0

    @property
    def avia_wait_for_deploy_attempts(self):
        return 0

    @property
    def avia_use_yt_store(self):
        return False

    BUILD_APIARGS = {'kill_timeout': 2700}

    @property
    def BUILD_CTX(self):
        return {
            'package_type': 'docker',
            'docker_push_image': True,
            'docker_registry': 'registry.yandex.net',
            'docker_image_repository': 'avia',
            'packages': self.avia_build_package,
            'docker_user': 'robot-avia-api-pilot',
            'docker_token_vault_name': 'AVIA_DOCKER_REGISTRY_OAUTH_TOKEN',
            'docker_build_network': 'host',
            'resource_type': self.avia_resource_name,
            'checkout': False,
            'use_aapi_fuse': True,
            'use_arc_instead_of_aapi': True,
            'ya_yt_store': self.avia_use_yt_store,
            'ignore_recurses': False,
            'env_vars': 'DOCKER_BUILDKIT=1',
            'release_to_ya_deploy': True,
            'yp_token_vault': 'ROBOT_AVIA_API_PILOT_YP_TOKEN',
        }

    @property
    def _trunk_part(self):
        default_trunk_part = super(AviaDockerToDeployJobGraphMixin, self)._trunk_part
        trunk_part = []
        for release_stage in self.avia_trunk_auto_deploy_stages:
            trunk_part += [
                jg_build.JobGraphElementBuildTrunk(
                    task_name='YA_PACKAGE',
                    job_params={
                        'apiargs': self.BUILD_APIARGS,
                    },
                    build_item='TRUNK',
                    ctx=self.BUILD_CTX,
                    out={self.avia_resource_name: 10},
                ),
                jg_release.JobGraphElementReleaseBase(
                    task_name='RELEASE_RM_COMPONENT_2',
                    release_to=release_stage,
                    job_arrows=(
                        jg_job_triggers.JobTriggerBuild(
                            parent_job_data=(
                                jg_job_data.ParentDataCtx(
                                    input_key='registry_name',
                                    output_key='output_resource_version',
                                ),
                                jg_job_data.ParentDataDict(
                                    input_key="component_resources",
                                    dict_key=self.avia_resource_name,
                                    resource_name=self.avia_resource_name,
                                )
                            ),
                            job_name_parameter='TRUNK',
                        ),
                    ),
                    job_params={
                        'frequency': (jg_utils.TestFrequency.RUN_IF_DELAY_N_MINUTES, 10),
                        'should_add_to_db': jg_utils.should_add_to_db_trunk,
                        'observed_paths': self.avia_project_observed_paths,
                        'ctx': {
                            'deploy_system': rm_const.DeploySystem.ya_deploy.name,
                        }
                    },
                ),
            ]
        return default_trunk_part + trunk_part

    @property
    def _branch_part(self):
        default_branch_part = super(AviaDockerToDeployJobGraphMixin, self)._branch_part
        branch_part = [
            jg_build.JobGraphElementBuildBranched(
                task_name='YA_PACKAGE',
                job_params={
                    'apiargs': self.BUILD_APIARGS,
                },
                ctx=self.BUILD_CTX,
                out={self.avia_resource_name: 20},
            )
        ]
        return default_branch_part + branch_part

    @property
    def _release(self):
        default_release_part = super(AviaDockerToDeployJobGraphMixin, self)._release
        release_part = [jg_release.JobGraphElementWikiFinal(
            job_params={
                "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
            }
        )]
        for release_stage in self.avia_release_button_deploy_stages:
            release_part.append(
                jg_release.JobGraphElementReleaseBranched(
                    task_name='RELEASE_RM_COMPONENT_2',
                    release_to=release_stage,
                    wait_for_deploy=self.avia_wait_for_deploy,
                    job_arrows=(
                        jg_job_triggers.JobTriggerBuild(
                            parent_job_data=(
                                jg_job_data.ParentDataCtx(
                                    input_key='registry_name',
                                    output_key='output_resource_version',
                                ),
                                jg_job_data.ParentDataDict(
                                    input_key="component_resources",
                                    dict_key=self.avia_resource_name,
                                    resource_name=self.avia_resource_name,
                                )
                            )
                        ),
                        jg_job_triggers.JobTriggerNewTag([
                            jg_job_data.ParentDataOutput('major_release_num', 'branch_number_for_tag'),
                            jg_job_data.ParentDataOutput('minor_release_num', 'new_tag_number'),
                        ]),
                    ),
                    job_params={
                        'frequency': (jg_utils.TestFrequency.LAZY, None),
                        'observed_paths': self.avia_project_observed_paths,
                        'ctx': {
                            'deploy_system': rm_const.DeploySystem.ya_deploy.name,
                            'wait_for_deploy_time_sec': self.avia_wait_for_deploy_time_sec,
                            'wait_for_deploy_attempts': self.avia_wait_for_deploy_attempts,
                        }
                    },
                ),
            )
            release_part.append(jg_release.JobGraphElementActionReleaseBranched(
                release_to=release_stage,
                job_arrows=(
                    jg_job_triggers.JobTriggerRelease(job_name_parameter=release_stage),
                ),
            ))
        return default_release_part + release_part
