# -*- coding: utf-8 -*-

from collections import namedtuple
import contextlib
import itertools
from enum import Enum

from sandbox.projects.release_machine.components import configs as cfg
from sandbox.projects.release_machine.core import const as rm_const
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils

ConductorConfig = namedtuple("ConductorConfig", ("branch", "filter"))
SystemTestConfig = namedtuple("SystemTestConfig", ("build_id", "parameters"))


@contextlib.contextmanager
def branched_config_creator(branch):
    yield lambda flt: ConductorConfig(branch=branch, filter=flt)


# New compact version of config madness
class StageConfigs(object):
    @staticmethod
    def merge_filters(configs, new_branch):
        return ConductorConfig(
            branch=new_branch,
            filter=";".join(conf.filter for conf in configs),
        )

    class Liquibase(object):
        with branched_config_creator("testing") as testing_conf:
            LOAD = testing_conf("paysys/balance2.back.timeline/paysys.balance.back.load")
            TM = testing_conf("paysys/balance2.back.timeline/paysys.balance.back.testing.testing")
            TS = testing_conf("paysys/balance2.back.timeline/paysys.balance.back.testing.staging")
            TA = testing_conf("paysys/balance2.back.timeline/paysys.balance.back.testing.assessors")
            TC = testing_conf("paysys/balance2.back.timeline/paysys.balance.back.testing.candidate")

        with branched_config_creator("prestable") as prestable_conf:
            PT = prestable_conf("paysys/balance2.back.timeline/paysys.balance.back.prestable")
            PT_HOTFIX = prestable_conf("paysys/balance2.back.env.prestable")

        with branched_config_creator("stable") as stable_conf:
            PROD = stable_conf("paysys/balance2.back.timeline")

        with branched_config_creator("hotfix") as hotfix_conf:
            HOTFIX = hotfix_conf("paysys/balance2.back.timeline")

    class BackendBinary(object):
        with branched_config_creator("testing") as testing_conf:
            TM = testing_conf("paysys/balance2.back.timeline.bydc/paysys.balance.back.test.testing")
            TC = testing_conf("paysys/balance2.back.timeline.bydc/paysys.balance.back.test.candidate")
            TS = testing_conf("paysys/balance2.back.timeline.bydc/paysys.balance.back.test.staging")
            TA = testing_conf("paysys/balance2.back.timeline.bydc/paysys.balance.back.test.assessors")
            LOAD = testing_conf("paysys/balance2.back.timeline.bydc/paysys.balance.back.load")

        with branched_config_creator("prestable") as prestable_conf:
            PT = prestable_conf("paysys/balance2.back.timeline.bydc/paysys.balance.back.prestable")
            PT_HOTFIX = prestable_conf("paysys/balance2.back.env.prestable")

        with branched_config_creator("stable") as stable_conf:
            PROD = stable_conf("paysys/balance2.back.timeline.bydc")

        with branched_config_creator("hotfix") as hotfix_conf:
            HOTFIX = hotfix_conf("paysys/balance2.back.timeline.bydc")

    class FrontendBinary(object):
        with branched_config_creator("testing") as testing_conf:
            TM = testing_conf("paysys/balance2.front.timeline.bydc/paysys.balance.front.test.testing")
            TC = testing_conf("paysys/balance2.front.timeline.bydc/paysys.balance.front.test.candidate")
            TS = testing_conf("paysys/balance2.front.timeline.bydc/paysys.balance.front.test.staging")
            TA = testing_conf("paysys/balance2.front.timeline.bydc/paysys.balance.front.test.assessors")
            LOAD = testing_conf("paysys/balance2.front.timeline.bydc/paysys.balance.front.load")

        with branched_config_creator("prestable") as prestable_conf:
            PT = prestable_conf("paysys/balance2.front.timeline.bydc/paysys.balance.front.prestable")
            PT_HOTFIX = prestable_conf("paysys/balance2.front.env.prestable")

        with branched_config_creator("stable") as stable_conf:
            PROD = stable_conf("paysys/balance2.front.timeline.bydc")

        with branched_config_creator("hotfix") as hotfix_conf:
            HOTFIX = hotfix_conf("paysys/balance2.front.timeline.bydc")


class JobGraphElementActionReleaseCustomBranched(jg_release.JobGraphElementActionReleaseBase):
    def __init__(self, release_to, job_params=None, job_arrows=(), release_item=""):
        job_name_parameter = "{}__{}".format(release_item, release_to) if release_item else release_to
        merged_job_params = jg_utils.merge_job_params(
            job_params,
            {
                "cancel_fallbehind_runs_on_fix": False,
                "should_add_to_db": jg_utils.should_add_to_db_branch,
            },
        )
        super(JobGraphElementActionReleaseCustomBranched, self).__init__(
            job_name_parameter=job_name_parameter,
            job_params=merged_job_params,
            job_arrows=job_arrows,
        )
        if release_to == rm_const.ReleaseStatus.stable:
            self.job_arrows += (jg_job_triggers.JobTriggerChangelogFinal(),)


class BalanceEnvRelease(object):
    run_system_tests_on = {
        "tm": SystemTestConfig(
            build_id="Billing_Autotesting_PythonTests_Manual",
            parameters={
                "env.balance": "tm",
            },
        ),
    }

    class ReleaseType(Enum):
        branch = 1  # Release in branch
        trunk = 2  # Release in trunk

    def __init__(self,
                 name,
                 release_to,
                 package_configs,
                 order=None,
                 templates=None,
                 st_environment=None,
                 ):
        self.name = name
        self.release_to = release_to
        self.package_configs = package_configs
        self.order = order
        self.templates = templates
        self.st_environment = st_environment or release_to

    @staticmethod
    def get_grouped_release_part(envs, release_to):
        """
        Create one action for triggering multiple releases at the same time

        :param envs: List of BalanceEnvRelease instances
        :return: release part list
        """
        return [JobGraphElementActionReleaseCustomBranched(
            release_to=release_to,
            release_item="_".join((env.name for env in envs)),
            job_arrows=[
                jg_job_triggers.JobTriggerRelease(job_name_parameter="{}__{}".format(env.name, env.release_to))
                for env in envs
            ],
        )]

    def _gen_release_task_ctx(self, release_type=ReleaseType.branch):
        env_packages = list(self.package_configs.keys())
        is_ordered = self.order is not None
        if is_ordered:
            package_index_dict = {package: i for i, package in enumerate(env_packages)}
            order_list = [
                ",".join(
                    (str(package_index_dict[package]) for package in package_group)
                )
                for package_group in self.order
            ]
        ctx = {
            "packages": env_packages,
            "ticket_branches": [
                self.package_configs[package_name].branch
                for package_name in env_packages
            ],
            "ticket_filters": [
                self.package_configs[package_name].filter
                for package_name in env_packages
            ],
            "ordered": is_ordered,
            "order_list": order_list if is_ordered else None,
            "ticket_templates": [
                self.templates[package_name]
                for package_name in env_packages
            ] if self.templates else None,
            "environment": self.st_environment,
        }
        if release_type == self.ReleaseType.trunk:
            ctx.update({
                "dont_create_comment": True,
            })
        return ctx

    def get_release_part(self, release_type=ReleaseType.branch):
        # igogor: эту часть не трогал, но имхо ее надо будет мальца порефакторить
        release_part = []
        arrows_for_release = []
        arrows_for_action_release = []

        release_item = self.name + ("_trunk" if release_type == self.ReleaseType.trunk else "")

        if release_type == self.ReleaseType.branch:
            arrows_for_release = [
                jg_job_triggers.JobTriggerBuild(
                    parent_job_data=[
                        jg_job_data.ParentDataCtx(
                            "packages_version", "output_resource_version",
                        ),
                    ],
                    job_name_parameter="BUILD_BINARIES_PACKAGES",
                ),
                jg_job_triggers.JobTriggerBuild(
                    job_name_parameter="BUILD_ASSESSOR_PACKAGE",
                ),
                # _gen_arrow_with_version("packages_version"),
                jg_job_triggers.JobTriggerBuild(
                    job_name_parameter="TEAMCITY_BUILD_MINIMALISTIC",
                ),
                jg_job_triggers.JobTriggerBuild(
                    job_name_parameter="TEAMCITY_BUILD_LIQUIBASE",
                ),
                jg_job_triggers.JobTriggerNewTag(
                    [
                        jg_job_data.ParentDataOutput(
                            "major_release_num", "branch_number_for_tag"
                        ),
                        jg_job_data.ParentDataOutput(
                            "minor_release_num", "new_tag_number"
                        ),
                        jg_job_data.ParentDataCtx(
                            "release_number", "scope_number", override=True
                        ),
                    ]
                )
            ]
        else:
            if self.name in self.run_system_tests_on:
                test_config = self.run_system_tests_on[self.name]

                def make_teamcity_parameters(revision, config):
                    params = {
                        "balance.version": "3.trunk.{revision}".format(revision=revision),
                        "balance.arcadia_branch": "{svn_branch}",
                        "balance.arcadia_revision": "{revision}",
                    }
                    params.update(config.parameters)
                    return params

                systests_build_item = "TEAMCITY_RUN_SYSTEM_TESTS_" + self.name.upper()
                release_job_name = "{}__{}".format(release_item, self.release_to) if release_item else self.release_to
                release_part.append(
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="RUN_TEAMCITY_BUILD",
                        build_item=systests_build_item,
                        ctx={
                            "build_task_id": test_config.build_id,
                            "build_timeout": 60 * 120,  # 2 hours
                        },
                        # fmt: off
                        job_arrows=[
                            jg_arrows.ParentsData(
                                input_key="teamcity_parameters",
                                triggers=[
                                    jg_arrows.ParamsData(
                                        input_key="revision",
                                        transform=lambda x, rm_config: x.revision,
                                    ),
                                    jg_arrows.JustPassData(
                                        input_key="config",
                                        value=test_config,
                                    ),
                                ],
                                transform=make_teamcity_parameters,
                            ),
                            jg_job_triggers.JobTriggerRelease(job_name_parameter=release_job_name),
                        ],
                        # fmt: on
                    )
                )
                arrows_for_action_release.append(
                    jg_job_triggers.JobTriggerBuild(
                        job_name_parameter=systests_build_item,
                    )
                )
            arrows_for_release = [
                jg_job_triggers.JobTriggerBuild(
                    parent_job_data=[
                        jg_job_data.ParentDataCtx(
                            "packages_version", "output_resource_version",
                        ),
                    ],
                    job_name_parameter="BUILD_BINARIES_PACKAGES_TRUNK",
                ),
                jg_job_triggers.JobTriggerBuild(
                    job_name_parameter="BUILD_ASSESSOR_PACKAGE_TRUNK",
                ),
                # _gen_arrow_with_version("packages_version"),
                jg_job_triggers.JobTriggerBuild(
                    job_name_parameter="TEAMCITY_BUILD_MINIMALISTIC_TRUNK",
                ),
                jg_job_triggers.JobTriggerBuild(
                    job_name_parameter="TEAMCITY_BUILD_LIQUIBASE_TRUNK",
                ),
            ]

        release_part.append(
            jg_release.JobGraphElementReleaseBranched(
                release_to=self.release_to,
                release_item=release_item,
                task_name="DO_CONDUCTOR_RELEASE",
                job_arrows=arrows_for_release,
                job_params={
                    "ctx": self._gen_release_task_ctx(release_type=release_type),
                    "should_add_to_db": (
                        jg_utils.should_add_to_db_trunk
                        if release_type == self.ReleaseType.trunk else
                        jg_utils.should_add_to_db_branch
                    ),
                },
            )
        )
        release_part.append(
            jg_release.JobGraphElementActionReleaseBranched(
                release_to=self.release_to, release_item=release_item,
                job_arrows=arrows_for_action_release,
                job_params={
                    "should_add_to_db": (
                        jg_utils.should_add_to_db_trunk
                        if release_type == self.ReleaseType.trunk else
                        jg_utils.should_add_to_db_branch
                    ),
                },
            )
        )
        return release_part


class MergedBalanceEnvRelease(BalanceEnvRelease):
    """
    Merge two or more environment config

    Ex: hotfix environment is a merge of PROD and PT configurations
    """

    class MergeType(Enum):
        ordered = 1  # Run tickets in order (not supported)
        ignore_order = 2  # Ignore all order parameters

    def __init__(self, name, release_to, environments, st_environment=None, template_suffixes=[""],
                 custom_branch=None, mode=MergeType.ignore_order):
        if mode != self.MergeType.ignore_order:
            raise NotImplementedError
        self.template_suffixes = itertools.cycle(template_suffixes)
        self.environments = environments
        self.release_to = release_to
        self.name = name
        self.mode = mode
        self.st_environment = st_environment or release_to
        self.custom_branch = custom_branch

    def _chain_data_from_env(self, data_getter):
        """
        :param data_getter: (BalanceEnvRelease) -> Iterable
        :return: Iterable chain of data for all environments
        """
        return itertools.chain(*(data_getter(env) for env in self.environments))

    def _gen_release_task_ctx(self, release_type=BalanceEnvRelease.ReleaseType.branch):
        result_ctx = {
            "packages": [],
            "ticket_branches": [],
            "ticket_filters": [],
            "ordered": False,
            "ticket_templates": [],
            "environment": self.st_environment,
        }
        for env, template_suffix in zip(self.environments, self.template_suffixes):
            task_ctx = env._gen_release_task_ctx(release_type=release_type)
            for key in ["packages", "ticket_filters"]:
                result_ctx[key].extend(task_ctx[key])
            result_ctx["ticket_branches"].extend(
                itertools.repeat(self.custom_branch, len(task_ctx["ticket_branches"]))
                if self.custom_branch is not None else
                task_ctx["ticket_branches"]
            )
            result_ctx["ticket_templates"].extend(
                map(lambda tmplt: tmplt + template_suffix, task_ctx["ticket_templates"])
            )
        if release_type == BalanceEnvRelease.ReleaseType.trunk:
            result_ctx["dont_create_comment"] = True
        return result_ctx


LIQUIBASE_TICKET_ORDER_BEFORE = [
    ("yb-liquibase-full-before",),
]

LIQUIBASE_TICKET_ORDER_AFTER = [
    ("yb-liquibase-full-after",),
]

LIQUIBASE_TICKET_ORDER = LIQUIBASE_TICKET_ORDER_BEFORE + LIQUIBASE_TICKET_ORDER_AFTER

BINARY_SERVICE_ORDER = [
    ("yb-tools-minimalistic", "yb-balance-arcadia", "yb-snout-proxy-arcadia"),
]

ASSESSORS_SERVICE_ORDER = [
    ("yb-tools-minimalistic", "yb-balance-assessors", "yb-snout-proxy-arcadia"),
]

FULL_SERVICE_ORDER = [
    ("yb-tools-minimalistic", "yb-balance-assessors", "yb-balance-arcadia", "yb-snout-proxy-arcadia"),
]

BINARY_TICKET_ORDER = (
    LIQUIBASE_TICKET_ORDER_BEFORE +
    BINARY_SERVICE_ORDER +
    LIQUIBASE_TICKET_ORDER_AFTER
)

ASSESSORS_TICKET_ORDER = (
    LIQUIBASE_TICKET_ORDER_BEFORE +
    ASSESSORS_SERVICE_ORDER +
    LIQUIBASE_TICKET_ORDER_AFTER
)

FULL_TICKETS_ORDER = (
    LIQUIBASE_TICKET_ORDER_BEFORE +
    FULL_SERVICE_ORDER +
    LIQUIBASE_TICKET_ORDER_AFTER
)

# COMPLETE_SERVICE_ORDER = [(
#     "yb-balance-arcadia",
#     "yb-snout-proxy-arcadia",
#     "yb-tools-minimalistic",
# )]
#
# COMPLETE_TICKET_ORDER = (
#     LIQUIBASE_TICKET_ORDER_BEFORE +
#     COMPLETE_SERVICE_ORDER +
#     LIQUIBASE_TICKET_ORDER_AFTER
# )

BASE_TICKET_TEMPLATES = {
    "yb-snout-proxy-arcadia": "yb_snout_proxy_arcadia",
    "yb-tools-minimalistic": "yb_tools_minimalistic",
    "yb-liquibase-full-before": "liquibase_full_before",
    "yb-liquibase-full-after": "liquibase_full_after",
}

# BINARY_TICKET_TEMPLATES = {
#     "yb-balance-arcadia": "yb_balance_arcadia",
#     "yb-snout-proxy-arcadia": "yb_snout_proxy_arcadia",
#     "yb-tools-minimalistic": "yb_tools_minimalistic",
#     "yb-liquibase-full-before": "liquibase_full_before",
#     "yb-liquibase-full-after": "liquibase_full_after",
# }

BINARY_TICKET_TEMPLATES = BASE_TICKET_TEMPLATES.copy()
BINARY_TICKET_TEMPLATES["yb-balance-arcadia"] = "yb_balance_arcadia"

# ASSESSORS_TICKET_TEMPLATES = {
#     "yb-balance-assessors": "yb_balance_assessors",
#     "yb-snout-proxy-arcadia": "yb_snout_proxy_arcadia",
#     "yb-tools-minimalistic": "yb_tools_minimalistic",
#     "yb-liquibase-full-before": "liquibase_full_before",
#     "yb-liquibase-full-after": "liquibase_full_after",
# }

ASSESSORS_TICKET_TEMPLATES = BASE_TICKET_TEMPLATES.copy()
ASSESSORS_TICKET_TEMPLATES["yb-balance-assessors"] = "yb_balance_assessors"

# FULL_TICKET_TEMPLATES = {
#     "yb-balance-arcadia": "yb_balance_arcadia",
#     "yb-balance-assessors": "yb_balance_assessors",
#     "yb-snout-proxy-arcadia": "yb_snout_proxy_arcadia",
#     "yb-tools-minimalistic": "yb_tools_minimalistic",
#     "yb-liquibase-full-before": "liquibase_full_before",
#     "yb-liquibase-full-after": "liquibase_full_after",
# }

FULL_TICKET_TEMPLATES = BASE_TICKET_TEMPLATES.copy()
FULL_TICKET_TEMPLATES["yb-balance-arcadia"] = "yb_balance_arcadia"
FULL_TICKET_TEMPLATES["yb-balance-assessors"] = "yb_balance_assessors"


def workflows(backend=None, front=None, backend_binary=None, front_binary=None, minimalistic=None, liquibase=None,
              assessors_binary=None):
    workflows_dict = {}
    if backend_binary:
        workflows_dict["yb-balance-arcadia"] = backend_binary
    if front_binary:
        workflows_dict["yb-snout-proxy-arcadia"] = front_binary
    if minimalistic:
        workflows_dict["yb-tools-minimalistic"] = minimalistic
    if liquibase:
        for packages in [
            "yb-liquibase-full-before",
            "yb-liquibase-full-after",
        ]:
            workflows_dict[packages] = liquibase
    if assessors_binary:
        workflows_dict["yb-balance-assessors"] = assessors_binary

    return workflows_dict


TC = BalanceEnvRelease(
    name="tc",
    release_to=rm_const.ReleaseStatus.unstable,
    package_configs=workflows(
        backend_binary=StageConfigs.BackendBinary.TC,
        front_binary=StageConfigs.FrontendBinary.TC,
        minimalistic=StageConfigs.BackendBinary.TC,
    ),
    templates=BINARY_TICKET_TEMPLATES,
)

TC_WITH_LIQUIBASE = BalanceEnvRelease(
    name="tc_liquibase",
    release_to=rm_const.ReleaseStatus.unstable,
    package_configs=workflows(
        backend_binary=StageConfigs.BackendBinary.TC,
        front_binary=StageConfigs.FrontendBinary.TC,
        minimalistic=StageConfigs.BackendBinary.TC,
        liquibase=StageConfigs.Liquibase.TC,
    ),
    order=BINARY_TICKET_ORDER,
    templates=BINARY_TICKET_TEMPLATES,
)

PT = BalanceEnvRelease(
    name="pt",
    release_to=rm_const.ReleaseStatus.prestable,
    package_configs=workflows(
        backend_binary=StageConfigs.BackendBinary.PT,
        front_binary=StageConfigs.FrontendBinary.PT,
        minimalistic=StageConfigs.BackendBinary.PT,
        liquibase=StageConfigs.Liquibase.PT,
    ),
    templates=BINARY_TICKET_TEMPLATES,
)

TM = BalanceEnvRelease(
    name="tm",
    release_to=rm_const.ReleaseStatus.unstable,
    package_configs=workflows(
        assessors_binary=StageConfigs.BackendBinary.TM,
        front_binary=StageConfigs.FrontendBinary.TM,
        minimalistic=StageConfigs.BackendBinary.TM,
    ),
    templates=ASSESSORS_TICKET_TEMPLATES,
)

TM_WITH_LIQUIBASE = BalanceEnvRelease(
    name="tm_liquibase",
    release_to=rm_const.ReleaseStatus.unstable,
    package_configs=workflows(
        assessors_binary=StageConfigs.BackendBinary.TM,
        front_binary=StageConfigs.FrontendBinary.TM,
        minimalistic=StageConfigs.BackendBinary.TM,
        liquibase=StageConfigs.Liquibase.TM,
    ),
    order=ASSESSORS_TICKET_ORDER,
    templates=ASSESSORS_TICKET_TEMPLATES,
)

TM_TC_LIQUIBASE = BalanceEnvRelease(
    name="tm_tc_liquibase",
    release_to=rm_const.ReleaseStatus.testing,
    package_configs=workflows(
        backend_binary=StageConfigs.BackendBinary.TC,
        assessors_binary=StageConfigs.BackendBinary.TM,
        front_binary=StageConfigs.merge_filters(
            configs=[StageConfigs.FrontendBinary.TM, StageConfigs.FrontendBinary.TC],
            new_branch=StageConfigs.FrontendBinary.TM.branch,
        ),
        minimalistic=StageConfigs.merge_filters(
            configs=[StageConfigs.BackendBinary.TM, StageConfigs.BackendBinary.TC],
            new_branch=StageConfigs.BackendBinary.TM.branch,
        ),
        liquibase=StageConfigs.merge_filters(
            configs=[StageConfigs.Liquibase.TM, StageConfigs.Liquibase.TC],
            new_branch=StageConfigs.Liquibase.TM.branch,
        ),
    ),
    order=FULL_TICKETS_ORDER,
    templates=FULL_TICKET_TEMPLATES,
)

TS = BalanceEnvRelease(
    name="ts",
    release_to=rm_const.ReleaseStatus.unstable,
    package_configs=workflows(
        backend_binary=StageConfigs.BackendBinary.TS,
        front_binary=StageConfigs.FrontendBinary.TS,
        minimalistic=StageConfigs.BackendBinary.TS,
    ),
    templates=BINARY_TICKET_TEMPLATES,
)

TS_WITH_LIQUIBASE = BalanceEnvRelease(
    name="ts_liquibase",
    release_to=rm_const.ReleaseStatus.unstable,
    package_configs=workflows(
        backend_binary=StageConfigs.BackendBinary.TS,
        front_binary=StageConfigs.FrontendBinary.TS,
        minimalistic=StageConfigs.BackendBinary.TS,
        liquibase=StageConfigs.Liquibase.TS,
    ),
    order=BINARY_TICKET_ORDER,
    templates=BINARY_TICKET_TEMPLATES,
)

TA = BalanceEnvRelease(
    name="ta",
    release_to=rm_const.ReleaseStatus.unstable,
    package_configs=workflows(
        backend_binary=StageConfigs.BackendBinary.TA,
        front_binary=StageConfigs.FrontendBinary.TA,
        minimalistic=StageConfigs.BackendBinary.TA,
    ),
    templates=BINARY_TICKET_TEMPLATES,
)

TA_WITH_LIQUIBASE = BalanceEnvRelease(
    name="ta_liquibase",
    release_to=rm_const.ReleaseStatus.unstable,
    package_configs=workflows(
        backend_binary=StageConfigs.BackendBinary.TA,
        front_binary=StageConfigs.FrontendBinary.TA,
        minimalistic=StageConfigs.BackendBinary.TA,
        liquibase=StageConfigs.Liquibase.TA,
    ),
    order=BINARY_TICKET_ORDER,
    templates=BINARY_TICKET_TEMPLATES,
)

LIQUIBASE_TEST_DB = BalanceEnvRelease(
    name="liquibase_test",
    release_to=rm_const.ReleaseStatus.testing,
    package_configs=workflows(
        liquibase=StageConfigs.merge_filters(
            configs=[StageConfigs.Liquibase.TM, StageConfigs.Liquibase.TC],
            new_branch=StageConfigs.Liquibase.TM.branch,
        )
    ),
    templates=BINARY_TICKET_TEMPLATES,
    order=LIQUIBASE_TICKET_ORDER,
)

LIQUIBASE_LOAD_DB = BalanceEnvRelease(
    name="liquibase_load",
    release_to=rm_const.ReleaseStatus.testing,
    package_configs=workflows(
        liquibase=StageConfigs.Liquibase.LOAD,
    ),
    templates=BINARY_TICKET_TEMPLATES,
    order=LIQUIBASE_TICKET_ORDER,
)

LOAD = BalanceEnvRelease(
    name="load",
    release_to=rm_const.ReleaseStatus.testing,
    package_configs=workflows(
        backend_binary=StageConfigs.BackendBinary.LOAD,
        front_binary=StageConfigs.FrontendBinary.LOAD,
        minimalistic=StageConfigs.BackendBinary.LOAD,
    ),
    templates=BINARY_TICKET_TEMPLATES,
)

LOAD_WITH_LIQUIBASE = BalanceEnvRelease(
    name="load_with_liquibase",
    release_to=rm_const.ReleaseStatus.testing,
    package_configs=workflows(
        backend_binary=StageConfigs.BackendBinary.LOAD,
        front_binary=StageConfigs.FrontendBinary.LOAD,
        minimalistic=StageConfigs.BackendBinary.LOAD,
        liquibase=StageConfigs.Liquibase.LOAD,
    ),
    order=BINARY_TICKET_ORDER,
    templates=BINARY_TICKET_TEMPLATES,
)

PROD = BalanceEnvRelease(
    name="prod",
    release_to=rm_const.ReleaseStatus.stable,
    package_configs=workflows(
        backend_binary=StageConfigs.BackendBinary.PROD,
        front_binary=StageConfigs.FrontendBinary.PROD,
        minimalistic=StageConfigs.BackendBinary.PROD,
        liquibase=StageConfigs.Liquibase.PROD,
    ),
    templates=BINARY_TICKET_TEMPLATES,
)

PROD_HOTFIX = BalanceEnvRelease(
    name="",
    release_to=rm_const.ReleaseStatus.stable,
    package_configs=workflows(
        backend_binary=StageConfigs.BackendBinary.HOTFIX,
        front_binary=StageConfigs.FrontendBinary.HOTFIX,
        minimalistic=StageConfigs.BackendBinary.HOTFIX,
        liquibase=StageConfigs.Liquibase.HOTFIX,
    ),
    templates=BINARY_TICKET_TEMPLATES,
    st_environment="hotfix",
)

PROD_HOTFIX_WITHOUT_LB = BalanceEnvRelease(
    name="hotfix",
    release_to=rm_const.ReleaseStatus.stable,
    package_configs=workflows(
        backend_binary=StageConfigs.BackendBinary.HOTFIX,
        front_binary=StageConfigs.FrontendBinary.HOTFIX,
        minimalistic=StageConfigs.BackendBinary.HOTFIX,
    ),
    templates=BINARY_TICKET_TEMPLATES,
    st_environment="hotfix",
)

PRESTABLE_HOTFIX = BalanceEnvRelease(
    name="",
    release_to=rm_const.ReleaseStatus.prestable,
    package_configs=workflows(
        backend_binary=StageConfigs.BackendBinary.PT_HOTFIX,
        front_binary=StageConfigs.FrontendBinary.PT_HOTFIX,
        minimalistic=StageConfigs.BackendBinary.PT_HOTFIX,
        liquibase=StageConfigs.Liquibase.PT_HOTFIX,
    ),
    templates=BINARY_TICKET_TEMPLATES,
    st_environment="hotfix",
)

PRESTABLE_HOTFIX_WITHOUT_LB = BalanceEnvRelease(
    name="",
    release_to=rm_const.ReleaseStatus.prestable,
    package_configs=workflows(
        backend_binary=StageConfigs.BackendBinary.PT_HOTFIX,
        front_binary=StageConfigs.FrontendBinary.PT_HOTFIX,
        minimalistic=StageConfigs.BackendBinary.PT_HOTFIX,
    ),
    templates=BINARY_TICKET_TEMPLATES,
    st_environment="hotfix",
)

HOTFIX = MergedBalanceEnvRelease(
    name="hotfix",
    release_to=rm_const.ReleaseStatus.stable,
    environments=[PROD_HOTFIX, PRESTABLE_HOTFIX],
    st_environment="hotfix",
    template_suffixes=["_prod", "_pt"],
)

HOTFIX_WITHOUT_LB = MergedBalanceEnvRelease(
    name="hotfix_without_lb",
    release_to=rm_const.ReleaseStatus.stable,
    environments=[PROD_HOTFIX_WITHOUT_LB, PRESTABLE_HOTFIX_WITHOUT_LB],
    st_environment="hotfix",
    template_suffixes=["_prod", "_pt"],
)


class BalanceCfg(cfg.ReferenceBranchedConfig):
    name = "balance"
    display_name = "Yandex Balance (Billing)"
    responsible = "igogor"

    class MergesCfg(cfg.ReferenceBranchedConfig.MergesCfg):
        never_merge_to_released_branches = False

        permissions = cfg.MergePermissions(
            permission_type=rm_const.PermissionType.ALLOWED,  # or rm_const.PermissionType.BANNED
            people_groups=cfg.PeopleGroups(
                staff_groups=["yandex_infra_int_2791_1676"],
                abc_services=[cfg.Abc(component_id=196, role_id=None)],  # To all people in our component
                logins={"igogor"},
            ),
        )

    class Testenv(cfg.ReferenceBranchedConfig.Testenv):
        new_task_owner = "BILLING-CI"
        trunk_task_owner = "BILLING-CI"
        testenv_db_owners = [
            "barsukovpt",
            "borograam",
            "ashul",
            "antropov-ivan",
            "allista",
            "d-manakovskiy",
            "igogor",
            "torvald",
            "natabers",
            "roman-nagaev",
            "aikawa",
            "isupov",
            "sfreest",
            "halty",
            "lightrevan",
            "dolvik",
            "enovikov11",
            "ozhegov",
            "agapitovds",
            "krissmacer",
            "vladbogdanov",
            "u-n-i-k",
            "kositsyn-pa",
            "scientistnik",
            "apopkov"
        ]

        class JobGraph(cfg.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrek(),
                ]

            @property
            def _branch_part(self):
                arrow_from_version_build = jg_arrows.ParentsData(
                    input_key="teamcity_parameters",
                    triggers=[
                        jg_job_triggers.JobTriggerNewTag(
                            [
                                jg_job_data.ParentDataOutput(
                                    "major_release_num", "branch_number_for_tag"
                                ),
                                jg_job_data.ParentDataOutput(
                                    "minor_release_num", "new_tag_number"
                                ),
                            ]
                        ),
                    ],
                    transform=lambda major_release_num, minor_release_num: {
                        "balance.version": "3.{}.{}".format(major_release_num, minor_release_num),
                        "balance.arcadia_branch": "{svn_branch}",
                    },
                )

                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="YA_PACKAGE",
                        build_item="BUILD_BINARIES_PACKAGES",
                        ctx={
                            "use_aapi_fuse": True,
                            "aapi_fallback": True,
                            "packages": (
                                "billing/balance/package/pkg.json;"
                                "billing/snout/yb_snout_proxy/package/pkg.json"
                            ),
                            "resource_type": (
                                "BALANCE_YA_PACKAGE;" "SNOUT_PROXY_YA_PACKAGE"
                            ),
                            "debian_distribution": "testing",
                            "publish_package": True,
                            "publish_to": "yandex-trusty",
                            "ya_yt_store": False,
                            "use_new_format": True,
                            "force_dupload": True,
                        },
                        # fmt: off
                        out={
                            "BALANCE_YA_PACKAGE": 90,
                            "SNOUT_PROXY_YA_PACKAGE": 90,
                        },
                        # fmt: on
                    )
                )
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="YA_PACKAGE",
                        build_item="BUILD_ASSESSOR_PACKAGE",
                        ctx={
                            "use_aapi_fuse": True,
                            "aapi_fallback": True,
                            "build_type": "debug",
                            "packages": (
                                "billing/balance/package/pkg_assessors.json"
                            ),
                            "resource_type": (
                                "BALANCE_ASSESSORS_YA_PACKAGE"
                            ),
                            "debian_distribution": "testing",
                            "publish_package": True,
                            "publish_to": "yandex-trusty",
                            "ya_yt_store": False,
                            "use_new_format": True,
                            "force_dupload": True,
                        },
                        # fmt: off
                        out={
                            "BALANCE_ASSESSORS_YA_PACKAGE": 90,
                        },
                        # fmt: on
                    )
                )
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="RUN_TEAMCITY_BUILD",
                        build_item="TEAMCITY_BUILD_LIQUIBASE",
                        ctx={
                            "build_task_id": "Billing_Tools_Arcadia_Build_Liquibase",
                            "build_timeout": 60 * 40,  # 40 minutes
                        },
                        # fmt: off
                        job_arrows=[
                            arrow_from_version_build,
                            jg_arrows.ParentsData(
                                input_key="build_uid",
                                triggers=[
                                    jg_job_triggers.JobTriggerNewTag(
                                        [
                                            jg_job_data.ParentDataOutput(
                                                "major_release_num", "branch_number_for_tag"
                                            ),
                                            jg_job_data.ParentDataOutput(
                                                "minor_release_num", "new_tag_number"
                                            ),
                                        ]
                                    ),
                                ],
                                transform=lambda major_release_num, minor_release_num: "liquibase_3.{}.{}".format(
                                    major_release_num,
                                    minor_release_num,
                                ),
                            ),
                        ],
                        # fmt: on
                    )
                )
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="RUN_TEAMCITY_BUILD",
                        build_item="TEAMCITY_BUILD_MINIMALISTIC",
                        ctx={
                            "build_task_id": "Billing_Tools_Arcadia_Build_YbToolsMinimalistic",
                            "build_timeout": 60 * 40,  # 40 minutes
                        },
                        # fmt: off
                        job_arrows=[
                            arrow_from_version_build,
                            jg_arrows.ParentsData(
                                input_key="build_uid",
                                triggers=[
                                    jg_job_triggers.JobTriggerNewTag(
                                        [
                                            jg_job_data.ParentDataOutput(
                                                "major_release_num", "branch_number_for_tag"
                                            ),
                                            jg_job_data.ParentDataOutput(
                                                "minor_release_num", "new_tag_number"
                                            ),
                                        ]
                                    ),
                                ],
                                transform=lambda major_release_num, minor_release_num: "minimalistic_3.{}.{}".format(
                                    major_release_num,
                                    minor_release_num,
                                ),
                            ),
                        ],
                        # fmt: on
                    )
                )
                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release

                # This is an action just to release resource (this is used to control changelog logic)
                # release_part.append(
                #     jg_release.JobGraphElementReleaseBranched(
                #         release_to=rm_const.ReleaseStatus.stable,
                #         release_item="changelog",
                #         job_arrows=[
                #             jg_job_triggers.JobTriggerBuild(
                #                 parent_job_data=[
                #                     jg_job_data.ParentDataDict(
                #                         "component_resources",
                #                         "balance_package",
                #                         "BALANCE_YA_PACKAGE",
                #                     ),
                #                     jg_job_data.ParentDataDict(
                #                         "component_resources",
                #                         "snout_package",
                #                         "SNOUT_PROXY_YA_PACKAGE",
                #                     ),
                #                 ],
                #                 job_name_parameter="BUILD_BINARIES_PACKAGES",
                #             ),
                #         ],
                #     )
                # )
                # release_part.append(
                #     jg_release.JobGraphElementActionReleaseBranched(
                #         release_to=rm_const.ReleaseStatus.stable,
                #         release_item="changelog",
                #     )
                # )

                for env in [TC, TM, TS, TA,
                            TC_WITH_LIQUIBASE, TA_WITH_LIQUIBASE, TM_WITH_LIQUIBASE, TS_WITH_LIQUIBASE,
                            TM_TC_LIQUIBASE,
                            LIQUIBASE_TEST_DB,
                            PT,
                            PROD,
                            LOAD, LOAD_WITH_LIQUIBASE, LIQUIBASE_LOAD_DB,
                            HOTFIX, HOTFIX_WITHOUT_LB]:
                    # fmt: off
                    release_part.extend(
                        env.get_release_part(),
                    )
                    # fmt: on
                # release_part.extend(BalanceEnvRelease.get_grouped_release_part([TM, TC], rm_const.ReleaseStatus.testing))
                return release_part

            @property
            def _trunk_part(self):
                arrow_from_version_build = jg_arrows.ParamsData(
                    input_key="teamcity_parameters",
                    transform=lambda x, rm_config: {
                        "balance.version": "3.trunk.{revision}".format(revision=x.revision),
                        "balance.arcadia_branch": "{svn_branch}",
                        "balance.arcadia_revision": "{revision}",
                    }
                )

                trunk_part = super(self.__class__, self)._trunk_part
                trunk_part.append(
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="YA_PACKAGE",
                        build_item="BUILD_BINARIES_PACKAGES_TRUNK",
                        ctx={
                            "use_aapi_fuse": True,
                            "aapi_fallback": True,
                            "packages": (
                                "billing/balance/package/pkg.json;"
                                "billing/snout/yb_snout_proxy/package/pkg.json"
                            ),
                            "resource_type": (
                                "BALANCE_YA_PACKAGE;" "SNOUT_PROXY_YA_PACKAGE"
                            ),
                            "debian_distribution": "testing",
                            "publish_package": True,
                            "publish_to": "yandex-trusty",
                            "ya_yt_store": False,
                            "use_new_format": True,
                            "force_dupload": True,
                        },
                        # fmt: off
                        out={
                            "BALANCE_YA_PACKAGE": 90,
                            "SNOUT_PROXY_YA_PACKAGE": 90,
                        },
                        # fmt: on
                        job_arrows=[
                            jg_arrows.ParamsData(
                                input_key="custom_version",
                                transform=lambda x, rm_config: "3.trunk.{revision}".format(
                                    revision=x.revision,
                                ),
                            ),
                        ]
                    )
                )
                trunk_part.append(
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="YA_PACKAGE",
                        build_item="BUILD_ASSESSOR_PACKAGE_TRUNK",
                        ctx={
                            "use_aapi_fuse": True,
                            "aapi_fallback": True,
                            "build_type": "debug",
                            "packages": (
                                "billing/balance/package/pkg_assessors.json"
                            ),
                            "resource_type": (
                                "BALANCE_ASSESSORS_YA_PACKAGE"
                            ),
                            "debian_distribution": "testing",
                            "publish_package": True,
                            "publish_to": "yandex-trusty",
                            "ya_yt_store": False,
                            "use_new_format": True,
                            "force_dupload": True,
                        },
                        # fmt: off
                        out={
                            "BALANCE_ASSESSORS_YA_PACKAGE": 90,
                        },
                        # fmt: on
                        job_arrows=[
                            jg_arrows.ParamsData(
                                input_key="custom_version",
                                transform=lambda x, rm_config: "3.trunk.{revision}".format(
                                    revision=x.revision,
                                ),
                            ),
                        ]
                    )
                )
                trunk_part.append(
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="RUN_TEAMCITY_BUILD",
                        build_item="TEAMCITY_BUILD_LIQUIBASE_TRUNK",
                        ctx={
                            "build_task_id": "Billing_Tools_Arcadia_Build_Liquibase",
                            "build_timeout": 60 * 40,  # 40 minutes
                        },
                        # fmt: off
                        job_arrows=[
                            arrow_from_version_build,
                            jg_arrows.ParamsData(
                                input_key="build_uid",
                                transform=lambda x, rm_config: "liquibase_3.trunk.{revision}".format(
                                    revision=x.revision,
                                ),
                            ),
                        ],
                        # fmt: on
                    )
                )
                trunk_part.append(
                    jg_build.JobGraphElementBuildTrunk(
                        task_name="RUN_TEAMCITY_BUILD",
                        build_item="TEAMCITY_BUILD_MINIMALISTIC_TRUNK",
                        ctx={
                            "build_task_id": "Billing_Tools_Arcadia_Build_YbToolsMinimalistic",
                            "build_timeout": 60 * 40,  # 40 minutes
                        },
                        # fmt: off
                        job_arrows=[
                            arrow_from_version_build,
                            jg_arrows.ParamsData(
                                input_key="build_uid",
                                transform=lambda x, rm_config: "minimalistic_3.trunk.{revision}".format(
                                    revision=x.revision,
                                ),
                            ),
                        ],
                        # fmt: on
                    )
                )

                for env in [TC, TM, TS, TA,
                            TC_WITH_LIQUIBASE, TA_WITH_LIQUIBASE, TM_WITH_LIQUIBASE, TS_WITH_LIQUIBASE]:
                    trunk_part.extend(
                        env.get_release_part(BalanceEnvRelease.ReleaseType.trunk)
                    )
                return trunk_part

        class JobPatch(cfg.ReferenceBranchedConfig.Testenv.JobPatch):
            @property
            def change_frequency(self):
                tests = super(self.__class__, self).change_frequency
                tests.update({
                    "BREST_UNIT_TESTS": rm_const.TestFrequencies.LAZY_TEST,
                    "SNOUT_API_UNIT_TESTS": rm_const.TestFrequencies.LAZY_TEST,
                    "SNOUT_PROXY_UNIT_TESTS": rm_const.TestFrequencies.LAZY_TEST,
                    "BALANCE_UNIT_TESTS": rm_const.TestFrequencies.LAZY_TEST,
                    "BREST_UNIT_TESTS_PR": rm_const.TestFrequencies.LAZY_TEST,
                    "SNOUT_API_UNIT_TESTS_PR": rm_const.TestFrequencies.LAZY_TEST,
                    "SNOUT_PROXY_UNIT_TESTS_PR": rm_const.TestFrequencies.LAZY_TEST,
                    "BALANCE_UNIT_TESTS_PR": rm_const.TestFrequencies.LAZY_TEST,
                })
                return tests

            @property
            def ignore_match(self):
                tests = super(self.__class__, self).ignore_match
                tests.extend([
                    "_BUILD__BALANCE__BUILD_BINARIES_PACKAGES_TRUNK",
                    "_BUILD__BALANCE__BUILD_ASSESSOR_PACKAGE_TRUNK",
                    "_BUILD__BALANCE__TEAMCITY_BUILD_LIQUIBASE_TRUNK",
                    "_BUILD__BALANCE__TEAMCITY_BUILD_MINIMALISTIC_TRUNK",
                    "_ACTION_RELEASE__BALANCE__TA_LIQUIBASE_TRUNK__UNSTABLE",
                    "_RELEASE__BALANCE__TA_LIQUIBASE_TRUNK__UNSTABLE",
                    "_ACTION_RELEASE__BALANCE__TA_TRUNK__UNSTABLE",
                    "_RELEASE__BALANCE__TA_TRUNK__UNSTABLE",
                    "_ACTION_RELEASE__BALANCE__TM_LIQUIBASE_TRUNK__UNSTABLE",
                    "_RELEASE__BALANCE__TM_LIQUIBASE_TRUNK__UNSTABLE",
                    "_ACTION_RELEASE__BALANCE__TM_TRUNK__UNSTABLE",
                    "_RELEASE__BALANCE__TM_TRUNK__UNSTABLE",
                    "_ACTION_RELEASE__BALANCE__TS_LIQUIBASE_TRUNK__UNSTABLE",
                    "_RELEASE__BALANCE__TS_LIQUIBASE_TRUNK__UNSTABLE",
                    "_ACTION_RELEASE__BALANCE__TS_TRUNK__UNSTABLE",
                    "_RELEASE__BALANCE__TS_TRUNK__UNSTABLE",
                    "_ACTION_RELEASE__BALANCE__TC_LIQUIBASE_TRUNK__UNSTABLE",
                    "_RELEASE__BALANCE__TC_LIQUIBASE_TRUNK__UNSTABLE",
                    "_ACTION_RELEASE__BALANCE__TC_TRUNK__UNSTABLE",
                    "_RELEASE__BALANCE__TC_TRUNK__UNSTABLE",
                ])
                tests.extend([
                    "_BUILD__BALANCE__TEAMCITY_RUN_SYSTEM_TESTS_" + key.upper()
                    for key in BalanceEnvRelease.run_system_tests_on
                ])
                return tests

    class Notify(cfg.ReferenceBranchedConfig.Notify):
        class Telegram(cfg.ReferenceBranchedConfig.Notify.Telegram):
            chats = ["balance"]
            invite_link = "https://t.me/joinchat/B0LEMw7vibuDb6gD9SpxlQ"

        class Startrek(cfg.ReferenceBranchedConfig.Notify.Startrek):
            """
            https://wiki.yandex-team.ru/releasemachine/startrekoptions/
            """

            assignee = "igogor"
            queue = "BALANCE"
            summary_template = "Balance release {}"
            components = u"Tools"
            followers = ["igogor", "torvald", "bali7", "samvel94"]
            ticket_type = "Release"
            tags = ["release-rm-task", rm_const.RMNames.BALANCE]
            use_task_author_as_assignee = True
            hide_commits_under_cut = True
            notify_on_deploy_to_feature_st_tickets = True
            nanny_reports = False
            ticket_description_suffix = "\n".join([
                '',
                '%%(wacko wrapper=shade border="1px solid grey")',
                '((https://wiki.yandex-team.ru/Balance/Release/checklist/ Регламент выкладки релиза)) ',
                '((https://wiki.yandex-team.ru/balance/timeline/ Инструкция по Таймлайну)) ',
                '',
                '**На тесте:**',
                '- разбери регрессию',
                '- проверь статусы тикетов, попадающих в релиз',
                '- проведи интеграцию с Директом <{Запускать тут:',
                'https://aqua.yandex-team.ru/#/pack/53296b57e4b00fcb8422650c',
                'https://aqua.yandex-team.ru/#/pack/54104612e4b0edbc99223fa7',
                '}>',
                '- получи cтатус регрессии TRUST у дежурного',
                '',
                '**Выкладка на предпрод:**',
                '- убедись, что не забыты второстепенные пакеты',
                '- оставь комментарий для аудиторов, если катим версию отличную от лежавшей под регрессией.',
                '',
                '**Выкладка на прод:**',
                '- до перевода в статус "Готов на Прод" убедись в наличии комментария с правильными тикетами.',
                '- для ручных действий админов используй !!~!!**ВАЖНО: <что сделать>**!!!!',
                '',
                '!!Во время выкладки и после неё следи за '
                '((https://wiki.yandex-team.ru/balance/duty/ графиками и мониторингами))!! (пункт 1)',
                '%%',
            ])

    class ChangelogCfg(cfg.ReferenceBranchedConfig.ChangelogCfg):
        """
        https://wiki.yandex-team.ru/releasemachine/changelogs/
        """

        dirs = [
            "arcadia/billing/balance_utils",
            "arcadia/billing/snout",
            "arcadia/billing/balance",
        ]
        wiki_page = ""

    class SvnCfg(cfg.ReferenceBranchedConfig.SvnCfg):
        tag_folder_template = "{branch_num}.{tag_num}"
        tag_folder_pattern = r"([0-9]+)\.([0-9]+)"

    class Releases(cfg.ReferenceBranchedConfig.Releases):
        allow_old_releases = True
        responsible = cfg.Responsible(abc=cfg.Abc(component_id=196, role_id=413), login="igogor")

        release_followers_permanent = ["igogor"]
        resources_info = [
            cfg.ReleasedResourceInfo(
                name="balance_package", resource_type="BALANCE_YA_PACKAGE",
            ),
            cfg.ReleasedResourceInfo(
                name="snout_package", resource_type="SNOUT_PROXY_YA_PACKAGE",
            ),
            # igogor: чет я не уверен что сюда это надо было добавлять
            cfg.ReleasedResourceInfo(
                name="balance_assessors_package", resource_type="BALANCE_ASSESSORS_YA_PACKAGE",
            ),
        ]
        wait_for_deploy_time_sec = 1 * 60 * 60
        deploy_system = rm_const.DeploySystem.others
