# -*- coding: utf-8 -*-

import sandbox.common.types.task as ctt
import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.config_core import notifications as rm_notifications
from sandbox.projects.release_machine.core import releasable_items as ri
import sandbox.projects.release_machine.components.job_graph.stages.pre_release_stage as jg_prerelease
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_job_arrows
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils


_META_TEST_TARGETS = ";".join([
    "balancer/test/configs/l7heavy_experiments_vla",
    "balancer/test/configs/l7heavy_production_fukraine_man",
    "balancer/test/configs/l7heavy_production_tun_man_only",
    "balancer/test/configs/l7heavy_production_tun_man",
    "balancer/test/configs/l7heavy_production_tun_sas_only",
    "balancer/test/configs/l7heavy_production_tun_sas",
    "balancer/test/configs/l7heavy_production_tun_vla_only",
    "balancer/test/configs/l7heavy_production_tun_vla",
    "balancer/test/configs/l7heavy_production_yaru_man",
    "balancer/test/configs/l7heavy_production_yaru_sas",
    "balancer/test/configs/l7heavy_production_yaru_vla",
    "balancer/test/configs/l7heavy_testing_tun_man",
    "balancer/test/configs/l7heavy_testing_tun_sas",
    "balancer/test/configs/l7heavy_testing_tun_vla",
])


class BalancerCfg(configs.ReferenceBranchedConfig):
    name = "balancer"
    responsible = "smalukav"

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        trunk_task_owner = "BALANCER"

        @staticmethod
        def db_drop_range(branch_number):
            to_save = 30
            if branch_number <= to_save:
                return range(0, 0)
            return range(branch_number - to_save, branch_number - to_save + 1)

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @property
            def _prerelease(self):
                return [
                    jg_prerelease.JobGraphElementNewBranch(),
                    jg_prerelease.JobGraphElementPreliminaryChangelogBranched(),
                    jg_prerelease.JobGraphElementCloneDb(),
                    jg_prerelease.JobGraphElementStartrek(
                        ctx={"group_changelog_by_tickets": True},
                    ),
                    jg_prerelease.JobGraphElementWiki(
                        ctx={"group_changelog_by_tickets": True},
                    ),
                    jg_prerelease.JobGraphElementActionPreReleaseStartrekWiki(),
                ]

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name='BUILD_BALANCER_BUNDLE',
                        build_item='BUILD',
                        out={'BALANCER_EXECUTABLE': 90},
                        ctx={
                            'append_gdb': True,  # MINOTAUR-1259
                            'build_corelimit_tool': True,  # MINOTAUR-1295
                            'build_tickets_deploy_tool': True,
                            'thinlto': True,  # BALANCER-3379
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name='BALANCER_LOAD_COMPARE',
                        job_params={
                            'test_type': jg_utils.TestType.CHECK_TEST,
                            'job_name_parameter': 'LOAD',
                            'frequency': (jg_utils.TestFrequency.EVERY_N_COMMIT, 1000),
                        },
                        ctx={
                            'binary_old': 'sandbox',
                            'binary_new': 'sandbox',
                            'release_component': 'balancer',
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=(
                                    jg_job_data.ParentDataResource(
                                        input_key='binary_new_sandbox_resource',
                                        resource_name='BALANCER_EXECUTABLE',
                                    ),
                                ),
                                job_name_parameter='BUILD',
                            ),
                            jg_job_arrows.ParamsData('release_number', jg_utils.get_major_release_number),
                        ),
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementMetaTestBranched(
                        job_params={
                            "job_name_parameter": "META_TESTS",
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                            "apiargs": {
                                "kill_timeout": 2 * 60 * 60,  # 2 hour
                            }
                        },
                        ctx={
                            "targets": _META_TEST_TARGETS,
                            "disable_test_timeout": True,
                        },
                    )
                )

                return branch_part

            @property
            def _release(self):
                release_part = super(self.__class__, self)._release
                for release_stage in [
                    rm_const.ReleaseStatus.stable,
                    rm_const.ReleaseStatus.testing,
                    rm_const.ReleaseStatus.unstable,
                    rm_const.ReleaseStatus.prestable,
                ]:
                    release_part.append(
                        jg_release.JobGraphElementReleaseBranched(
                            release_to=release_stage,
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=(
                                        jg_job_data.ParentDataDict(
                                            'component_resources',
                                            'balancer',
                                            'BALANCER_EXECUTABLE',
                                        ),
                                    ),
                                    job_name_parameter="BUILD",
                                ),
                            ),
                        )
                    )
                    release_part.append(
                        jg_release.JobGraphElementActionReleaseBranched(
                            release_to=release_stage,
                        )
                    )
                return release_part

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            @property
            def change_frequency(self):
                tests = super(self.__class__, self).change_frequency
                tests.update({
                    'BALANCER_LOAD_TEST': rm_const.TestFrequencies.LAZY_TEST,
                })
                return tests

    class ReleaseViewer(configs.ReferenceConfig.ReleaseViewer):
        kpi_alert = 10
        kpi_alert_skip_weekends = True

    class Releases(configs.ReferenceBranchedConfig.Releases):
        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    name="balancer",
                    data=ri.SandboxResourceData("BALANCER_EXECUTABLE"),
                    deploy_infos=[ri.NannyDeployInfo(
                        [
                            ri.DeployService(i) for i in [
                                "balancer_knoss_morda_yp_man",
                                "balancer_knoss_morda_yp_sas",
                                "balancer_knoss_morda_yp_vla",
                                "balancer_knoss_search_yp_man",
                                "balancer_knoss_search_yp_sas",
                                "balancer_knoss_search_yp_vla",
                                "balancer_l7leveler_kubr_man",
                                "balancer_l7leveler_kubr_man_only",
                                "balancer_l7leveler_kubr_man_testing",
                                "balancer_l7leveler_kubr_sas",
                                "balancer_l7leveler_kubr_sas_only",
                                "balancer_l7leveler_kubr_sas_testing",
                                "balancer_l7leveler_kubr_vla",
                                "balancer_l7leveler_kubr_vla_exp_amd",
                                "balancer_l7leveler_kubr_vla_exp_new",
                                "balancer_l7leveler_kubr_vla_experiment",
                                "balancer_l7leveler_kubr_vla_only",
                                "balancer_l7leveler_yaru_man",
                                "balancer_l7leveler_yaru_sas",
                                "balancer_l7leveler_yaru_sas_testing",
                                "balancer_l7leveler_yaru_vla",
                                "balancer_proxy_yandex_ua_man",
                                "balancer_proxy_yandex_ua_sas",
                                "balancer_proxy_yandex_ua_vla",
                                "balancer-l7leveler-kubr-yp-sas",
                                "balancer-l7leveler-kubr-yp-sas-only",
                                "balancer-l7leveler-kubr-yp-vla",
                                "balancer-l7leveler-kubr-yp-vla-only",
                                "balancer-l7leveler-yaru-yp-sas",
                                "balancer-l7leveler-yaru-yp-vla",
                                "balancer-proxy-yandex-ua-yp-sas",
                                "balancer-proxy-yandex-ua-yp-vla",
                                "production_balancer_any_man",
                                "production_balancer_any_sas",
                                "production_balancer_any_vla",
                                "rtc_balancer_knoss_fast_testing_yp_man",
                                "rtc_balancer_knoss_fast_testing_yp_sas",
                                "rtc_balancer_knoss_fast_testing_yp_vla",
                                "rtc_balancer_knoss_fast_yp_man",
                                "rtc_balancer_knoss_fast_yp_sas",
                                "rtc_balancer_knoss_fast_yp_vla",
                            ]
                        ]
                    )]
                ),
            ]
        release_followers_permanent = [
            "dmitryno",
            "mmike16",
        ]
        allow_robots_to_release_stable = True
        allow_old_releases = True
        default_stage = ctt.ReleaseStatus.UNSTABLE

    class Notify(configs.ReferenceBranchedConfig.Notify):
        notifications = [
            rm_notifications.Notification(
                event_type="MergeCommit",
                chat_name="balancer_config_comp",
                conditions=rm_notifications.NotificationCondition(conditions=[], join_strategy="OR"),
            ),
            rm_notifications.Notification(
                event_type="NewBranch",
                chat_name="balancer_config_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewTag",
                chat_name="balancer_config_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="TestenvDbClone",
                chat_name="balancer_config_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewWikiPage",
                chat_name="balancer_config_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="TicketHistory",
                chat_name="balancer_config_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="BuildTest",
                chat_name="balancer_config_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="KpiAlert",
                chat_name="balancer_config_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="ReleaseFailed",
                chat_name="balancer_config_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="ReleaseCreated",
                chat_name="balancer_config_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_created_oldschool.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseDeployed",
                chat_name="balancer_config_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_deployed_oldschool.html",
            ),
        ]

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            chats = ["balancer_config_comp"]
            config = configs.RmTelegramNotifyConfig(chats=chats)

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            assignee = "mmike16"
            queue = "BALANCERREL"
            dev_queue = "BALANCER"
            summary_template = u"Приемка балансера {}"
            workflow = {}
            add_commiters_as_followers = True
            notify_on_deploy_to_feature_st_tickets = True
            deadline = 14
            important_changes_limit = 2000

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = "balancer/releases/"
        ya_make_targets = ["arcadia/balancer/daemons/balancer"]
        review_groups = [
            "balancer",
        ]
