# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from collections import defaultdict
import re
import json
import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.release_machine.core import releasable_items as ri
from sandbox.projects.release_machine.components import configs
from sandbox.projects.release_machine.components.config_core import yappy as yappy_cfg
from sandbox.projects.release_machine.components.config_core import notifications as rm_notifications
from sandbox.projects.websearch.begemot.common import BegemotResources as BR, Begemots, BegemotTestEnvQueries
from sandbox.projects.common.constants import constants as common_const
from sandbox.projects.common import string
import sandbox.projects.release_machine.components.config_core.statistics_page as statistics_page
import sandbox.projects.release_machine.components.job_graph.job_data as jg_job_data
import sandbox.projects.release_machine.components.job_graph.job_triggers as jg_job_triggers
import sandbox.projects.release_machine.components.job_graph.stages.release_stage as jg_release
import sandbox.projects.release_machine.components.job_graph.stages.build_stage as jg_build
import sandbox.projects.release_machine.components.job_graph.stages.test_stage as jg_test
import sandbox.projects.release_machine.components.job_graph.utils as jg_utils
import sandbox.projects.release_machine.components.job_graph.job_arrows as jg_arrows


class _YappyStrings(object):
    paths = {
        # release id -> path in services
        "bstr_callback": "callback",
    }
    checkconfig = {
        "begemot": "begemot-worker-linux-bin-md5",
        "worker.cfg": "begemot-worker-cfg-md5",
    }


def make_beta_cgi(json_sources, host=None):
    try:
        beta_sources = json.loads(json_sources)
        experimental_sources = ["&search_experimental_source=%s" % source for source in beta_sources]
        if host:
            experimental_sources.append(
                "&host_config[inthosts][search]=http://addrs-testing.search.yandex.net/search/{}/".format(host)
            )
        experimental_sources_string = "".join(experimental_sources)
        return experimental_sources_string
    except Exception:
        return None


def make_beta_url(betas_sources):
    url_tpl = "http://addrs-testing.search.yandex.net/search/stable/yandsearch?{experimental_sources}"
    try:
        experimental_sources = []
        for beta_sources in betas_sources:
            experimental_sources.append(make_beta_cgi(beta_sources))
        experimental_sources_string = "".join(experimental_sources)
        experimental_sources_string = experimental_sources_string.replace("search_experimental_", "")
        return url_tpl.format(experimental_sources=experimental_sources_string)
    except Exception:
        return "failed to make_beta_url"


class BegemotCfg(configs.ReferenceBranchedConfig):
    name = "begemot"
    responsible = "gluk47"

    class MergesCfg(configs.ReferenceBranchedConfig.MergesCfg):
        never_merge_to_released_branches = True

    class Testenv(configs.ReferenceBranchedConfig.Testenv):
        """Testenv configuration"""
        trunk_db = "ws-begemot-trunk"
        trunk_task_owner = "BEGEMOT"
        branch_db_template = "ws-begemot-{testenv_db_num}"

        class JobGraph(configs.ReferenceBranchedConfig.Testenv.JobGraph):
            @staticmethod
            def _get_task_output(resources, ttl=14):
                return {res[1]: ttl for res in resources}

            @staticmethod
            def _get_parent_resources(resources):
                if not isinstance(resources, list):
                    resources = [resources]
                return tuple(
                    jg_job_data.ParentDataDict(
                        "component_resources",
                        res_id,
                        res_name,
                    )
                    for res_id, res_name in resources
                )

            # Metrics SLA-project "SLA Release Machine", more info at https://metrics.yandex-team.ru/admin/sla
            BEGEMOT_METRICS_SLA_PROJECT = "aa8286386850f0df016874cc48882d44"

            @property
            def _release(self):
                return super(self.__class__, self)._release + [
                    jg_release.JobGraphElementReleasePushBranched(
                        task_name="RELEASE_RM_COMPONENT_2",
                        release_to=rm_const.ReleaseStatus.stable,
                        job_params={
                            "task_priority": 7,
                            "ctx": {
                                "deploy_system": rm_const.DeploySystem.nanny_push.name,
                            }
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=jg_job_data.ParentDataDict(
                                    "component_resources",
                                    "scheduled_bundle",
                                    "BEGEMOT_STABLE_RELEASE_CONFIG",
                                ),
                                job_name_parameter=common_const.RELEASE_BUILD_TYPE,
                            ),
                            jg_job_triggers.JobTriggerNewTag([
                                jg_job_data.ParentDataOutput('major_release_num', 'branch_number_for_tag'),
                                jg_job_data.ParentDataOutput('minor_release_num', 'new_tag_number'),
                            ]),
                        )
                    ),
                    jg_release.JobGraphElementReleaseBranched(
                        task_name="RELEASE_RM_COMPONENT_2",
                        release_to=rm_const.ReleaseStatus.testing,
                        job_params={
                            "task_priority": 7,
                            "frequency": (jg_utils.TestFrequency.CHECK_EACH_COMMIT, None),
                            "ctx": {
                                "deploy_system": rm_const.DeploySystem.nanny_push.name,
                            }
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=jg_job_data.ParentDataDict(
                                    "component_resources",
                                    "begemot_testing_release_config_resource",
                                    "BEGEMOT_TESTING_RELEASE_CONFIG",
                                ),
                                job_name_parameter=common_const.DEBUG_BUILD_TYPE,
                            ),
                            jg_job_triggers.JobTriggerNewTag([
                                jg_job_data.ParentDataOutput('major_release_num', 'branch_number_for_tag'),
                                jg_job_data.ParentDataOutput('minor_release_num', 'new_tag_number'),
                            ]),
                        )
                    ),
                    jg_release.JobGraphElementScheduleRelease(
                        release_to=rm_const.ReleaseStatus.stable,
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=jg_job_data.ParentDataResource(
                                    "resource_with_data_to_schedule",
                                    "BEGEMOT_STABLE_RELEASE_CONFIG",
                                ),
                                job_name_parameter=common_const.RELEASE_BUILD_TYPE,
                            ),
                        ),
                        job_params={
                            "ctx": {
                                "deploy_system": rm_const.DeploySystem.nanny_push.name,
                                "schedule_mode": "just_schedule",
                                "warden_component_name": "begemot",
                                "flows": ["deploy"],
                            }
                        },
                    ),
                    jg_release.JobGraphElementActionPushReleaseBranched(rm_const.ReleaseStatus.stable),
                    jg_release.JobGraphElementActionScheduleBranched(rm_const.ReleaseStatus.stable)
                ]

            @property
            def _branch_part(self):
                branch_part = super(self.__class__, self)._branch_part
                for build_type in [common_const.RELEASE_BUILD_TYPE, common_const.DEBUG_BUILD_TYPE]:
                    out = self._get_task_output(BR.AllCommon)
                    out.update(self._get_task_output(BR.Release.Fastbuild, ttl=30))
                    out.update(self._get_task_output(BR.Betas.SpecialBinaries, ttl=30))
                    out.update({
                        "BEGEMOT_GEO_EXECUTABLE": 30,
                        "BEGEMOT_ISS_GEO": 30,
                        "BEGEMOT_STABLE_RELEASE_CONFIG": 30,
                        "BEGEMOT_TESTING_RELEASE_CONFIG": 30,
                    })
                    out.pop("BEGEMOT_FAST_BUILD_CONFIG_GEO", None)
                    branch_part.append(jg_build.JobGraphElementBuildBranched(
                        task_name="RELEASE_BEGEMOT_RESOURCES",
                        job_params={
                            "task_priority": 7,
                        },
                        build_item=build_type,
                        ctx={
                            "all_resources": True,
                            common_const.BUILD_TYPE_KEY: build_type,
                            common_const.USE_AAPI_FUSE: True,
                        },
                        out=out,
                    ))
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="RELEASE_BEGEMOT_RESOURCES",
                        job_params={
                            "task_priority": 7,
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="begemot_resources",
                                transform=lambda x, rm_config: ["BegemotFresh"],
                            ),
                        ),
                        build_item="WIZARD_RUNTIME",
                        out=self._get_task_output(BR.Release.Fresh),
                    )
                )
                branch_part.append(
                    jg_build.JobGraphElementBuildBranched(
                        task_name="DEPLOY_BEGEMOT_SPELLCHECKER_FRESH",
                        job_params={
                            "task_priority": 7,
                        },
                        job_arrows=(
                            jg_arrows.ParamsData(
                                input_key="begemot_resources",
                                transform=lambda x, rm_config: ["BegemotFresh"],
                            ),
                        ),
                        build_item="SPELLCHECKER_FRESH",
                        out=self._get_task_output(BR.Spellchecker.Fresh),
                    )
                )
                for conf_type in ["beta", "news"]:
                    branch_part.append(
                        jg_test.JobGraphElementYappyBetaGeneratorBranched(
                            beta_conf_type=conf_type,
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=self._get_parent_resources(BR.Betas.Fastbuild + BR.Betas.SpecialBinaries + BR.AllCommon),
                                    job_name_parameter=common_const.RELEASE_BUILD_TYPE,
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=self._get_parent_resources(BR.Betas.Fresh),
                                    job_name_parameter="WIZARD_RUNTIME",
                                ),
                            ),
                            job_params={
                                "job_name_parameter": conf_type.upper(),
                            },
                            ctx={
                                "force_start_beta": True,
                                "task_priority": 7,
                            },
                        )
                    )
                branch_part.append(
                    jg_test.JobGraphElementTestXMLSearch(
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="beta_url",
                                        output_key="beta_name",
                                        transform=lambda x, params: "https://{}.hamster.yandex.ru".format(x),
                                    ),
                                ),
                                job_name_parameter="BETA",
                            ),
                        ),
                    )
                )
                for name, svc in Begemots:
                    if not svc.perf_beta:
                        continue
                    if BegemotTestEnvQueries[name][0] is not None:
                        res_arrow = jg_arrows.GlobalResourceData(
                            input_key="queries",
                            resource_name=BegemotTestEnvQueries[name][0],
                        )
                    else:
                        res_arrow = jg_arrows.GlobalResourceData(
                            input_key="ah_queries",
                            resource_name=BegemotTestEnvQueries[name][1],
                        )
                    for label in ["OLD", "NEW"]:
                        prev_shooting_arrow = None if label == "OLD" else jg_job_triggers.JobTriggerTestBranchCommon(
                            parent_job_data=(
                                jg_job_data.ParentDataId(
                                    input_key="benchmark_task",
                                ),
                            ),
                            job_name_parameter="TANK_LOAD_TEST_{}_OLD".format(name)
                        )
                        branch_part.append(
                            jg_test.JobGraphElementTestBranchCommon(
                                task_name="BEGEMOT_TANK_LOAD_TEST",
                                job_params={
                                    "job_name_parameter": "TANK_LOAD_TEST_{}_{}".format(name, label),
                                    "task_priority": 7,
                                },
                                job_arrows=(
                                    jg_job_triggers.JobTriggerGenerateBeta(
                                        parent_job_data=(
                                            jg_job_data.ParentDataCtx(
                                                input_key="beta",
                                                output_key="beta_name",
                                            ),
                                        ),
                                        job_name_parameter="PERF_BETA_{}".format(label),
                                    ),
                                    res_arrow,
                                    prev_shooting_arrow,
                                ),
                                ctx={
                                    "tank_names": [
                                        "begemot-tank-yp-{}.{}.yp-c.yandex.net:8083".format(i, geo)
                                        for i in range(1, 10) for geo in ['sas', 'vla']
                                    ],
                                    "type_or_quota": "begemot-{}-perf@begemot".format(svc.perf_beta),
                                    "queries_prefix": "/wizard?",
                                    "load_plan": "line(1, 1000, 150s)" if name != 'ImagesCV' else "line(1, 300, 50s)",
                                    "rps_to_survive": 250 if name not in ['ImagesCV', 'Megamind'] else 150,
                                    "retries": 3,
                                    "ticket": "BEGEMOT-2059",
                                    "report_to_release_ticket": label == "NEW",
                                    "rm_component": "begemot",
                                }
                            )
                        )
                branch_part.append(
                    jg_test.JobGraphElementActionRunAcceptanceBranchedByMarker(
                        job_arrows=(
                            jg_job_triggers.JobTriggerLaunchMetrics(
                                job_name_parameter="GEOBEGEMOT_YAPPY",
                            ),
                            jg_job_triggers.JobTriggerGenerateBeta(
                                job_name_parameter="BETA",
                            ),
                            jg_job_triggers.JobTriggerGenerateBeta(
                                job_name_parameter="service_beta",
                            ),
                            jg_job_triggers.JobTriggerLaunchMetrics(),
                            jg_job_triggers.JobTriggerLaunchMetrics(
                                job_name_parameter="IMAGES",
                            ),
                            jg_job_triggers.JobTriggerLaunchMetrics(
                                job_name_parameter="VIDEO",
                            ),
                            jg_job_triggers.JobTriggerLaunchMetrics(
                                job_name_parameter="MUSIC",
                            ),
                            jg_job_triggers.JobTriggerLaunchMetrics(
                                job_name_parameter="NEWS_METRICS",
                            ),
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="CHECK_BETA_COREDUMPS",
                            ),
                            jg_job_triggers.JobTriggerTestXMLSearch(),
                        ) + tuple(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="TANK_LOAD_TEST_{}_NEW".format(name)
                            ) for name, svc in Begemots if svc.perf_beta
                        )
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementYappyBetaGeneratorBranched(
                        beta_conf_type="spellchecker",
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=self._get_parent_resources(BR.Common + BR.Spellchecker.Fastbuild),
                                job_name_parameter=common_const.RELEASE_BUILD_TYPE,
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=self._get_parent_resources(BR.Spellchecker.Fresh),
                                job_name_parameter="SPELLCHECKER_FRESH",
                            ),
                        ),
                        ctx={
                            "task_priority": 7,
                        },
                    )
                )

                for conf_type in ["beta", "news"]:
                    prefix = "NEWS_" if conf_type == "news" else ""
                    branch_part.append(
                        jg_test.JobGraphElementYappyBetaGeneratorBranched(
                            beta_conf_type=conf_type,
                            job_arrows=(
                                jg_job_triggers.JobTriggerBuild(
                                    parent_job_data=self._get_parent_resources(BR.Betas.Fresh),
                                    job_name_parameter="WIZARD_RUNTIME",
                                ),
                                jg_job_triggers.JobTriggerBuild(
                                    job_name_parameter=common_const.RELEASE_BUILD_TYPE,
                                ),
                            ),
                            job_params={
                                "job_name_parameter": "{}BENCHMARK".format(prefix),
                            },
                            ctx={
                                "task_priority": 7,
                                "update_beta_mode": "APPEND",
                                "beta_name_source": "LAST_RELEASED",
                            },
                        )
                    )
                metrics_job_arrows = (
                    jg_job_triggers.JobTriggerGenerateBeta(
                        parent_job_data=(
                            jg_job_data.ParentDataCtx(
                                input_key="checked_beta",
                                output_key="beta_name",
                                transform=lambda x, params: "{x}.hamster".format(x=x),
                            )
                        ),
                        job_name_parameter="BETA",
                    ),
                    jg_job_triggers.JobTriggerGenerateBeta(
                        parent_job_data=(
                            jg_job_data.ParentDataCtx(
                                input_key="sample_beta",
                                output_key="beta_name",
                                transform=lambda x, params: "{x}.hamster".format(x=x),
                            )
                        ),
                        job_name_parameter="BENCHMARK",
                    ),
                )
                branch_part.append(
                    jg_test.JobGraphElementLaunchMetrics(
                        job_arrows=metrics_job_arrows,
                        ctx={
                            "custom_template_name": "common.json",  # METRICS-6369
                            "run_findurl": True,
                            "scraper_over_yt_pool": "begemot_web_priemka",
                            "sla_project": self.BEGEMOT_METRICS_SLA_PROJECT,
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementLaunchMetrics(
                        job_arrows=metrics_job_arrows,
                        search_subtype="images",
                        ctx={
                            "autoclicker_retry_count": 2,
                            "scraper_over_yt_pool": "begemot_images_priemka",
                            "sla_project": self.BEGEMOT_METRICS_SLA_PROJECT,
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementLaunchMetrics(
                        job_arrows=metrics_job_arrows,
                        search_subtype="video",
                        ctx={
                            "autoclicker_retry_count": 5,
                            "launch_template_quota": "mm-cron-serps",
                            "scraper_over_yt_pool": "begemot_video_priemka",
                            "sla_project": self.BEGEMOT_METRICS_SLA_PROJECT,
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementLaunchMetrics(
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="checked_beta",
                                        output_key="beta_name",
                                        transform=lambda x, params: "{x}.music-web.music.qable".format(x=x),
                                    )
                                ),
                                job_name_parameter="BETA",
                            ),
                            jg_job_triggers.JobTriggerGenerateBeta(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="sample_beta",
                                        output_key="beta_name",
                                        transform=lambda x, params: "{x}.music-web.music.qable".format(x=x),
                                    )
                                ),
                                job_name_parameter="BENCHMARK",
                            ),
                        ),
                        job_name_parameter="MUSIC",
                        search_subtype="music",
                        ctx={
                            "autoclicker_retry_count": 3,
                        }
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementLaunchMetrics(
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="checked_beta",
                                        output_key="new_beta_url",
                                    ),
                                ),
                                job_name_parameter="NEWS",
                            ),
                            jg_job_triggers.JobTriggerGenerateBeta(
                                parent_job_data=(
                                    jg_job_data.ParentDataOutput(
                                        input_key="sample_beta",
                                        output_key="new_beta_url",
                                    ),
                                ),
                                job_name_parameter="NEWS_BENCHMARK",
                            ),
                        ),
                        job_params={
                            "job_name_parameter": "NEWS_METRICS",
                        },
                        search_subtype="news",
                        ctx={
                            "beta_conf_type": "news",
                            "custom_template_name": "news_runtime_acceptance.json",
                            "scraper_over_yt_pool": "news",
                            "autoclicker_metric_name": "empty-or-failed",
                        }
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="CHECK_BETA_COREDUMPS",
                        job_params={
                            "job_name_parameter": "CHECK_BETA_COREDUMPS",
                            "task_priority": 7,
                        },
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="name",
                                        output_key="beta_name",
                                    ),
                                ),
                                job_name_parameter="BETA",
                            ),
                        ),
                        ctx={
                            "itype": "begemot",
                            "signals": ["instancectl-exit_signal_{}_mmmm".format(num) for num in [6, 9, 11]],
                            "exclude_substrings": ["mmeta"],
                        }
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="ADDRS_COLLECT_RMDATA",
                        job_params={
                            "job_name_parameter": "COLLECT_AUX_RESOURCES_FOR_GEOWIZARD",
                        }
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementYappyBetaGeneratorBranched(
                        beta_conf_type="GEOBEGEMOT_ref",
                        job_params={
                            "task_name": "CREATE_ADDRS_BETA",
                            "ctx": {
                                "beta_type_choices": "wizard",
                                "wait_time": 90,
                                "launch_type_choices": "RM",
                            },
                        },
                        job_arrows=(
                            jg_arrows.ParamsData("tag", transform=lambda x, rm_config: "{}-ref".format(x.revision)),
                            jg_arrows.ParamsData("branch", jg_utils.get_major_release_number),
                            jg_arrows.ParentsData(
                                input_key="resources",
                                triggers=(
                                    jg_job_triggers.JobTriggerTestBranchCommon(
                                        parent_job_data=(
                                            jg_job_data.ParentDataCtx(
                                                input_key="BEGEMOT_GEO_EXECUTABLE",
                                                output_key="BEGEMOT_GEO_EXECUTABLE",
                                            ),
                                            jg_job_data.ParentDataCtx(
                                                input_key="BEGEMOT_ISS_GEO",
                                                output_key="BEGEMOT_ISS_GEO",
                                            ),
                                            jg_job_data.ParentDataCtx(
                                                input_key="BEGEMOT_CONFIG",
                                                output_key="BEGEMOT_CONFIG",
                                            ),
                                        ),
                                        job_name_parameter="COLLECT_AUX_RESOURCES_FOR_GEOWIZARD",
                                    ),
                                ),
                                transform=lambda BEGEMOT_GEO_EXECUTABLE, BEGEMOT_ISS_GEO, BEGEMOT_CONFIG: [
                                    BEGEMOT_GEO_EXECUTABLE,
                                    BEGEMOT_ISS_GEO,
                                    BEGEMOT_CONFIG,
                                ],
                            ),
                        )
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementYappyBetaGeneratorBranched(
                        beta_conf_type="GEOBEGEMOT_test",
                        job_params={
                            "task_name": "CREATE_ADDRS_BETA",
                            "ctx": {
                                "beta_type_choices": "wizard",
                                "wait_time": 90,
                                "launch_type_choices": "RM",
                            },
                        },
                        job_arrows=(
                            jg_arrows.ParamsData("tag", transform=lambda x, rm_config: "{}-test".format(x.revision)),
                            jg_arrows.ParamsData("branch", jg_utils.get_major_release_number),
                            jg_arrows.ParentsData(
                                input_key="resources",
                                triggers=(
                                    jg_job_triggers.JobTriggerBuild(
                                        parent_job_data=(
                                            jg_job_data.ParentDataResource(
                                                input_key="BEGEMOT_GEO_EXECUTABLE",
                                                resource_name="BEGEMOT_GEO_EXECUTABLE",
                                            ),
                                            jg_job_data.ParentDataResource(
                                                input_key="BEGEMOT_ISS_GEO",
                                                resource_name="BEGEMOT_ISS_GEO",
                                            ),
                                            jg_job_data.ParentDataResource(
                                                input_key="BEGEMOT_CONFIG",
                                                resource_name="BEGEMOT_CONFIG",
                                            ),
                                        ),
                                        job_name_parameter=common_const.RELEASE_BUILD_TYPE,
                                    ),
                                ),
                                transform=lambda BEGEMOT_GEO_EXECUTABLE, BEGEMOT_ISS_GEO, BEGEMOT_CONFIG: [
                                    BEGEMOT_GEO_EXECUTABLE,
                                    BEGEMOT_ISS_GEO,
                                    BEGEMOT_CONFIG,
                                ],
                            ),
                        ),
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementTestBranchCommon(
                        task_name="ADDRS_MAPKIT_COMPARE_TEST",
                        job_params={
                            "job_name_parameter": "LAUNCH_MAPKIT_ACCEPTANCE_GEOBEGEMOT_YAPPY",
                        },
                        ctx={
                            "behave_framevork_url": "arcadia:/arc/trunk/arcadia/search/geo/tools/geosearch_acceptance",
                            "launch_type": "RM",
                        },
                        job_arrows=(
                            jg_arrows.ParamsData("release_number", jg_utils.get_major_release_number),
                            jg_job_triggers.JobTriggerGenerateBeta(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="reference_stand_url",
                                        output_key="experimental_sources",
                                        transform=lambda x, params: make_beta_url([x]),
                                    ),
                                ),
                                job_name_parameter="GEOBEGEMOT_ref",
                            ),
                            jg_job_triggers.JobTriggerGenerateBeta(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="test_stand_url",
                                        output_key="experimental_sources",
                                        transform=lambda x, params: make_beta_url([x]),
                                    ),
                                ),
                                job_name_parameter="GEOBEGEMOT_test",
                            ),
                        )
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementLaunchMetrics(
                        search_subtype="geo",
                        job_arrows=(
                            jg_job_triggers.JobTriggerGenerateBeta(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="sample_beta",
                                        output_key="beta_name",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="sample_extra_params",
                                        output_key="experimental_sources",
                                        transform=lambda x, params: make_beta_cgi(x, "stable"),
                                    ),
                                ),
                                job_name_parameter="GEOBEGEMOT_ref",
                            ),
                            jg_job_triggers.JobTriggerGenerateBeta(
                                parent_job_data=(
                                    jg_job_data.ParentDataCtx(
                                        input_key="checked_beta",
                                        output_key="beta_name",
                                    ),
                                    jg_job_data.ParentDataCtx(
                                        input_key="checked_extra_params",
                                        output_key="experimental_sources",
                                        transform=lambda x, params: make_beta_cgi(x, "stable"),
                                    ),
                                ),
                                job_name_parameter="GEOBEGEMOT_test",
                            ),
                        ),
                        ctx={
                            "metrics_mode_type": "release_machine",
                            "custom_template_name": "common.json",
                            "launch_template_quota": "geosearch",
                            "enable_autoclicker": True,
                            "autoclicker_metric_name": "diff-2-serps-query-10",
                            "autoclicker_retry_count": 3,
                            "scraper_over_yt_pool": "geo",
                        },
                        job_name_parameter="GEOBEGEMOT_YAPPY",
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementYappyBetaGeneratorBranched(
                        beta_conf_type="perf_beta",
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=self._get_parent_resources(BR.PerfBetas.Fresh),
                                job_name_parameter="WIZARD_RUNTIME",
                            ),
                        ),
                        job_params={
                            "job_name_parameter": "PERF_BETA_OLD",
                        },
                        ctx={
                            "update_beta_mode": "REPLACE",
                            "force_start_beta": True,
                            "task_priority": 7,
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementYappyBetaGeneratorBranched(
                        beta_conf_type="perf_beta",
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=self._get_parent_resources(BR.PerfBetas.Fastbuild + BR.AllCommon),
                                job_name_parameter=common_const.RELEASE_BUILD_TYPE,
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=self._get_parent_resources(BR.PerfBetas.Fresh),
                                job_name_parameter="WIZARD_RUNTIME",
                            ),
                        ) + tuple(
                            jg_job_triggers.JobTriggerTestBranchCommon(
                                job_name_parameter="TANK_LOAD_TEST_{}_OLD".format(name)
                            ) for name, svc in Begemots if svc.perf_beta
                        ),
                        job_params={
                            "job_name_parameter": "PERF_BETA_NEW",
                        },
                        ctx={
                            "force_start_beta": True,
                            "task_priority": 7,
                        },
                    )
                )
                branch_part.append(
                    jg_test.JobGraphElementYappyBetaGeneratorBranched(
                        beta_conf_type="service_beta",
                        job_arrows=(
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=self._get_parent_resources(BR.ServiceBeta.Fastbuild + BR.AllCommon),
                                job_name_parameter=common_const.RELEASE_BUILD_TYPE,
                            ),
                            jg_job_triggers.JobTriggerBuild(
                                parent_job_data=self._get_parent_resources(BR.ServiceBeta.Fresh),
                                job_name_parameter="WIZARD_RUNTIME",
                            ),
                        ),
                        ctx={
                            "force_start_beta": True,
                            "task_priority": 7,
                        },
                    )
                )
                return branch_part

            @property
            def _trunk_part(self):
                sanitizers = {
                    "": None,
                    "asan": "address",
                    "lsan": "leak",
                    "tsan": "thread",
                }
                builds = []
                shard_binaries_with_tags = defaultdict(set)

                for shard in Begemots.values():
                    if not shard.release or shard.released_by != 'search':
                        continue
                    tags = set()  # todo: define tags here
                    shard_binaries_with_tags[shard.binary.upper()].update(tags)

                for shard_binary, tags in shard_binaries_with_tags.items():
                    shard_binary_cut = string.left_strip(shard_binary, "BEGEMOT").lstrip("_")
                    for san_name, san_value in sanitizers.items():
                        targets = ["{}_EXECUTABLE".format(shard_binary)]
                        if shard_binary == "BEGEMOT":
                            aux_targets = [
                                "BEGEMOT_AH_CONVERTER",
                                "BEGEMOT_EVLOGDUMP",
                                "BEGEMOT_EVLOGSTAT",
                                "BEGEMOT_YT_EVENTLOG_MAPPER",
                                "BEGEMOT_YT_MAPPER",
                                "BEGEMOT_FAST_BUILD_DOWNLOADER",
                                "BEGEMOT_SHARD_UPDATER",
                            ]
                            targets.extend(aux_targets)
                        if san_value is None:
                            build_type = common_const.RELEASE_BUILD_TYPE
                        else:
                            build_type = common_const.RELEASE_WITH_DEBUG_INFO_BUILD_TYPE
                        builds.append(jg_build.JobGraphElementBuildTrunk(
                            "BUILD_BEGEMOT",
                            job_params={"observed_paths": ["search/wizard/data/wizard"]},
                            build_item="_".join(filter(None, [shard_binary_cut, san_name])),
                            filter_targets=[
                                "devtools/ya",
                                "search/begemot",
                                "web/daemons/begemot",
                                "web/daemons/wizard",
                                "search/wizard/data/wizard",
                            ],
                            ctx={
                                "target_resources": targets,
                                common_const.BUILD_SYSTEM_KEY: common_const.SEMI_DISTBUILD_BUILD_SYSTEM,
                                common_const.BUILD_TYPE_KEY: build_type,
                                common_const.ALLOW_AAPI_FALLBACK: True,
                                common_const.USE_AAPI_FUSE: True,
                                common_const.USE_ARC_INSTEAD_OF_AAPI: True,
                                common_const.MUSL: san_value is None and shard_binary != "BEGEMOT_BERT",
                                "sanitize": san_value,
                            },
                            out={i: 7 for i in targets}
                        ))
                return builds

        class JobPatch(configs.ReferenceBranchedConfig.Testenv.JobPatch):
            @property
            def change_frequency(self):
                uppercase_name = self.name.upper()
                tests = super(self.__class__, self).change_frequency
                for s in Begemots.Services:
                    tests["_BUILD_RELEASE_{}_DATA_{}".format(uppercase_name, s)] = rm_const.TestFrequencies.EACH_REV_TEST
                tests.update({
                    "_TESTING_RELEASE_BEGEMOT": rm_const.TestFrequencies.EACH_REV_TEST
                })
                return tests

            @property
            def ignore_match(self):
                """List of test names which should be ignored in branch db"""
                return super(self.__class__, self).ignore_match + [
                    'PRECOMMIT_GENERATE_YAPPY_BETA_BEGEMOT',
                    'WIZARD_RUNTIME_RELEASE',  # runtime should be released only from trunk
                    # these tests are only for trunk, in branches priemka runs in another task:
                    'PRIEMKA_WIZARD',
                    "GET_BEGEMOT_RESPONSES_MULTI",  # currently it's precommit only
                ] + [
                    'COMPARE_BEGEMOT_RESPONSES_PRECOMMIT{proto}_{shard}'.format(proto=proto, shard=shard)
                    for proto in ['', '_APPHOST']
                    for shard in Begemots.keys()
                ] + [
                    'BUILD_BEGEMOT_DATA_{shard}_FULL'.format(shard=shard)
                    for shard in Begemots.keys()
                ]

            @property
            def ignore_prefix(self):
                """List of test prefixes which should be ignored in branch db"""
                return super(self.__class__, self).ignore_prefix + [
                    "PREPARE_BEGEMOT_FUZZY_INPUT",
                    "GET_WIZARD_PRODUCTION_QUERIES",
                    "BUILD_PRINTWIZARD_",
                    "RUN_LAZY",
                    "UPDATE_WIKI_",
                    "RUN_FAT_LAZY",
                    "RUN_MEDIUM_LAZY",
                    "BEGEMOT_FUZZY",
                    "PRINTWZRD",
                    "BUILD_BEGEMOT_FUZZY",
                    "CHECK_BEGEMOT_RESOURCES_AGE",
                    "GET_BEGEMOT_RESPONSES_TSAN",  # fix TSAN (BEGEMOT-2188 and alike)
                    "COMPARE_BEGEMOT_FAST_AND_FULL",
                ]

    class ReleaseViewer(configs.ReferenceConfig.ReleaseViewer):
        DAYS_FROM_COMMIT_TO_DEPLOY_CUSTOM_CHART = (
            "https://grafana.yandex-team.ru/d-solo/wz4_ABIMk/component-stats?"
            "tab=queries&orgId=1&panelId=2&theme=light&from=now-62d&to=now&description=Days+from+Commit+to+Deploy"
        )

        statistics_page_charts = [
            DAYS_FROM_COMMIT_TO_DEPLOY_CUSTOM_CHART,
            statistics_page.PredefinedCharts.RELEASE_FREQUENCY,
            statistics_page.PredefinedCharts.RELEASE_TIME_PERCENTILES,
            statistics_page.PredefinedCharts.RELEASE_STAGES,
            statistics_page.PredefinedCharts.TIMESPECTRE_METRICS,
        ]

        job_timeline_filters = ["metrics"]

        kpi_alert = 3

    class Releases(configs.ReferenceBranchedConfig.Releases):
        allow_robots_to_release_stable = True
        deploy_system = rm_const.DeploySystem.nanny_push
        allow_to_remove_resources = True
        allow_old_releases = True

        @property
        def releasable_items(self):
            return [
                ri.ReleasableItem(
                    BR.MainBinary[0],
                    ri.SandboxResourceData(BR.MainBinary[1]),
                    deploy_infos=[ri.single_nanny_service("begemot_failcache_yp_prod_vla")]
                ),
                ri.DynamicReleasableItem(
                    "scheduled_bundle",
                    ri.SandboxResourceData("BEGEMOT_STABLE_RELEASE_CONFIG"),
                    [ri.NannyDeployInfo()],
                ),
                ri.DynamicReleasableItem(
                    "begemot_testing_release_config_resource",
                    ri.SandboxResourceData("BEGEMOT_TESTING_RELEASE_CONFIG"),
                    [ri.NannyDeployInfo(stage=rm_const.ReleaseStatus.testing)],
                ),
            ]

        release_followers_permanent = [
            "gluk47",
        ]

    class Notify(configs.ReferenceBranchedConfig.Notify):
        notifications = [
            rm_notifications.Notification(
                event_type="NewBranch",
                chat_name="begemot_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewTag",
                chat_name="begemot_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_SUCCESS_AND_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="TestenvDbClone",
                chat_name="begemot_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="NewWikiPage",
                chat_name="begemot_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="TicketHistory",
                chat_name="begemot_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="BuildTest",
                chat_name="begemot_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="KpiAlert",
                chat_name="begemot_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
            ),
            rm_notifications.Notification(
                event_type="NewBetaGeneration",
                chat_name="begemot_comp",
                conditions=rm_notifications.NotificationCondition(
                    conditions=rm_notifications.TASK_PROBLEM_CONDITIONS,
                    join_strategy="OR",
                ),
            ),
            rm_notifications.Notification(
                event_type="ReleaseFailed",
                chat_name="begemot_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_failed_with_release_errors.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseDeployed",
                chat_name="begemot_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_deployed_oldschool.html",
            ),
            rm_notifications.Notification(
                event_type="ReleaseCreated",
                chat_name="begemot_comp",
                conditions=rm_notifications.CONDITION_ALWAYS,
                message_template_file="notifications/release_created_oldschool_begemot.html",
            ),
        ]

        class Mail(configs.ReferenceBranchedConfig.Notify.Mail):
            mailing_list = [
                "search-components-releases@yandex-team.ru",
            ]

        class Telegram(configs.ReferenceBranchedConfig.Notify.Telegram):
            chats = ["begemot_comp"]
            config = configs.RmTelegramNotifyConfig(chats=chats)
            invite_link = "https://t.me/joinchat/B_pLAEN5kbM7sfB2P__Z4A"

        class Startrek(configs.ReferenceBranchedConfig.Notify.Startrek):
            assignee = "gluk47"
            queue = "BEGEMOTREL"
            dev_queue = "BEGEMOT"
            summary_template = "Приемка Begemot {}"
            workflow = {
                "open": "autoTesting",
                "fixProblems": "accepting",
                "production": "close",
                "closed": "reopen",
                "qualityOK": "deploying",
                "accepting": "qualityOK",
                "autoTesting": "autoTestsOK",
                "autoTestsOK": "accepting",
                "deploying": "production",
            }
            followers = [
                "dmitryno",
                "yurakura",
                "krivokon",
            ]
            add_commiters_as_followers = True
            deadline = 7
            notify_on_deploy_to_feature_st_tickets = True

            def on_release_close_tickets_by_query(self, st_issue_key=None):
                if st_issue_key:
                    return '"Linked To": {} AND Status: Commited (Queue: "BEGEMOT" OR Queue: "REQWIZARD")'.format(
                        st_issue_key
                    )

    class ChangelogCfg(configs.ReferenceBranchedConfig.ChangelogCfg):
        wiki_page = "JandeksPoisk/KachestvoPoiska/begemot/releases/"
        observed_paths = [
            "arcadia/search/begemot",
            "arcadia/search/wizard/data",
            "arcadia/web/daemons/begemot",
        ]
        svn_paths_filter = configs.ChangelogPathsFilter(rm_const.PermissionType.BANNED, [
            "arcadia/search/wizard/data/fresh",
        ])
        markers = [
            (4, re.compile(r"\bdiff *: *(begemot|wizard)", flags=re.IGNORECASE)),
        ]

    class MetricsCfg(configs.ReferenceBranchedConfig.MetricsCfg):
        limit_s = None

    class Yappy(yappy_cfg.YappyBaseCfg):

        @staticmethod
        def _get_resources_for_shard(shard):
            storage = None if shard == "Bert" else "/ssd"
            return [
                yappy_cfg.YappyParametrizedResource(
                    local_path=_YappyStrings.paths.get(key, key),
                    checkconfig_name=_YappyStrings.checkconfig.get(key),
                    param_name=key,
                    storage=storage,
                )
                for key, resource in BR.Common + BR.Bstr if not (key == 'begemot' and getattr(Begemots.Service[shard], "binary_resource_name"))
            ] + [
                yappy_cfg.YappyParametrizedResource(
                    local_path={
                        "fresh_resource_name": "fresh",
                        "fast_build_config_resource_name": "fast_build_config.json",
                        "binary_resource_name": "begemot"
                    }[resource],
                    param_name=Begemots.gen_shard_resources_list(shard, resource)[0][0],
                    storage=storage
                )
                for resource in ["fresh_resource_name", "fast_build_config_resource_name", "binary_resource_name"]
                if getattr(Begemots.Service[shard], resource)
            ] + [
                yappy_cfg.YappyDeleteResource(local_path="\\+rm\\+.*")
            ]

        @property
        def betas(self):
            return {
                "beta": yappy_cfg.YappyTemplateCfg(
                    template_name="begemot",
                    new_yappy=True,
                    patches=[
                        yappy_cfg.YappyTemplatePatch(
                            patch_dir="begemot-{}".format(svc.beta),
                            resources=self._get_resources_for_shard(name),
                            parent_service=svc.beta_parent,
                        )
                        for name, svc in Begemots if svc.beta
                    ]
                ),
                "perf_beta": yappy_cfg.YappyTemplateCfg(
                    template_name="begemot-perf",
                    new_yappy=True,
                    patches=[
                        yappy_cfg.YappyTemplatePatch(
                            patch_dir="begemot-{}-perf".format(svc.perf_beta),
                            resources=self._get_resources_for_shard(name),
                            parent_service=svc.beta_parent,
                        )
                        for name, svc in Begemots if svc.perf_beta
                    ],
                    stop_betas_gap=0,
                ),
                "service_beta": yappy_cfg.YappyTemplateCfg(
                    template_name="service-wizard",
                    new_yappy=True,
                    patches=[
                        yappy_cfg.YappyTemplatePatch(
                            patch_dir="service-wizard",
                            resources=self._get_resources_for_shard("ServiceWizard"),
                            parent_service=Begemots["ServiceWizard"].beta_parent,
                        )
                    ]
                ),
                "cv_tagmodel": yappy_cfg.YappyTemplateCfg(
                    template_name="begemot-cv-tagmodel",
                    new_yappy=True,
                    patches=[
                        yappy_cfg.YappyTemplatePatch(
                            patch_dir="begemot_cv_yp_priemka_tagmodel",
                            resources=self._get_resources_for_shard("ImagesCV"),
                            parent_service=Begemots["ImagesCV"].beta_parent,
                        )
                    ]
                ),
                "spellchecker": yappy_cfg.YappyTemplateCfg(
                    template_name="begemot-spellchecker",
                    new_yappy=True,
                    patches=[
                        yappy_cfg.YappyTemplatePatch(
                            patch_dir="begemot-spellchecker",
                            resources=self._get_resources_for_shard("Spellchecker") + [
                                yappy_cfg.YappyStaticResource(
                                    local_path="framework_adapter",
                                    manage_type="SANDBOX_RESOURCE",
                                    resource_id="805472725",
                                ),
                            ],
                            parent_service="spellcheckerrus_production_man",
                        ),
                    ],
                    working_betas_limit=1,
                ),
                "news": yappy_cfg.YappyTemplateCfg(
                    template_name="begemot-news",
                    new_yappy=True,
                    patches=[
                        yappy_cfg.YappyTemplatePatch(
                            patch_dir="news-begemot-{}".format(svc.beta),
                            resources=self._get_resources_for_shard(name),
                            parent_service=svc.beta_parent,
                        )
                        for name, svc in Begemots if svc.beta
                    ]
                ),
            }
        working_betas_limit = 2

    class SvnCfg(configs.ReferenceBranchedConfig.SvnCfg):
        allow_autobranches = True
